import * as React from 'react';
import { RouteComponentProps } from 'react-router';

import { Dict } from '../../../../../types';
import { Button, CancelButton, DeleteButton, SaveButton } from '../../../../ui/Button';
import { Confirm, IWithConfirmState, Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import Select from '../../../../ui/Select';
import { ConstantsKey } from '../../../../utils/fetchConstants';
import { Request2 } from '../../../../utils/request';
import { buttonNameDetails } from '../../../../utils/sendLogs/eventTypes/buttonDetails';
import { deepCopy } from '../../../../utils/utils';
import { Copy } from '../../../Copy';
import { FormConstructor } from '../../../FormConstructor';
import Spin from '../../../Spin';
import { IBot, IBotProposition } from '../component';
import { BOTS_REQUESTS, REQUESTS } from '../request';
import BotConfigModal from './BotConfigModal';
import BotProposeButton from './BotProposeButton';

interface IBotModalProps extends RouteComponentProps<{ bot_name?: string }> {
    onClose: (success?: boolean) => void;
    updateBot: (state: Dict<any>) => void;
    constants: Promise<any>;
    error: Error | null;
    botName?: string;
    propositions: IBotProposition[];
}

interface IBotModalState extends IWithConfirmState {
    isLoading: boolean;
    schemas: Dict<any>;
    schema: string;
    bot: IBot | null;
    propositions: IBotProposition[];
    initialBotData: Dict<any> | null;
    botName: string;
    error: Error | null;
    formData: Dict<any>;
    isOpenBotConfigModal: boolean;
    isFormValid: boolean;
    isFormChanged: boolean;
}

export default class BotModal extends React.Component<IBotModalProps, IBotModalState> {
    state: IBotModalState = {
        isLoading: false,
        schemas: {},
        schema: '',
        bot: null,
        propositions: [],
        initialBotData: null,
        botName: '',
        error: null,
        formData: {},
        confirmIsOpen: false,
        question: '',
        accept: () => {},
        isWorking: false,
        isOpenBotConfigModal: false,
        isFormValid: false,
        isFormChanged: false,
    };
    request = new Request2({ requestConfigs: BOTS_REQUESTS });

    componentDidMount(): void {
        this.getBots();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getBots() {
        this.setState({ error: null, isLoading: true }, () => {
            this.props.constants
                .then(schemaResponse => {
                    const schemas = schemaResponse?.[ConstantsKey?.IFACE_RT_BACKGROUND_SETTINGS] ?? {};
                    this.setState(
                        { schemas }, () => {
                            const botName = this.props.match.params?.bot_name
                                ? decodeURIComponent(this.props.match.params.bot_name)
                                : this.props.botName ?? '';

                            botName
                                ? this.request.exec(REQUESTS.GET_BOTS, { queryParams: { ids: [botName] } })
                                    .then(botResponse => {
                                        this.setState({
                                            isLoading: false,
                                            bot: botResponse.rt_backgrounds[0],
                                            propositions: botResponse?.propositions,
                                            initialBotData: botResponse.rt_backgrounds[0],
                                            botName,
                                            schema: botResponse.rt_backgrounds[0].bp_type,
                                        });
                                    })
                                    .catch(error => {
                                        this.setState({ error, isLoading: false });
                                    })
                                : this.setState({
                                    isLoading: false,
                                });
                        });
                })
                .catch(error => {
                    this.setState({ error, isLoading: false });
                });
        });
    }

    componentDidUpdate(prevProps: Readonly<IBotModalProps>): void {
        if (decodeURIComponent(this.props.match.params.bot_name || '')
            !== decodeURIComponent(prevProps.match.params.bot_name || '')
            || this.props.botName !== prevProps.botName) {
            this.getBots();
        }
    }

    onSchemaSelect(schema: string) {
        this.setState({ schema });
    }

    onChangeForm(formData: Dict<any>, isFormValid: boolean, isFormChanged: boolean) {
        this.setState({ formData, isFormValid, isFormChanged });
    }

    onDeleteClick() {
        const name = this.state.bot?.bp_name;

        this.setState({
            error: null,
            confirmIsOpen: true,
            question: <>Удалить робота <b>{name}</b>?</>,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    this.request.exec(REQUESTS.REMOVE_BOTS, { body: { ids: [name] } })
                        .then(() => {
                            this.props.onClose(true);
                        })
                        .catch(error => {
                            this.setState({
                                error,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    closeDeleteModal() {
        this.setState({ confirmIsOpen: false });
    }

    openBotConfigModal() {
        this.setState({ isOpenBotConfigModal: true });
    }

    closeBotConfigModal() {
        this.setState({ isOpenBotConfigModal: false });
    }

    onConfigurationSaveClick(data: { filtersFormData: any; itemsFormData: any; actionsFormData: any }) {
        const { formData } = this.state;
        const initialBotData = deepCopy(formData);

        const { filtersFormData, itemsFormData, actionsFormData } = data;
        const { bp_settings = {} } = initialBotData;

        if (bp_settings) {
            bp_settings.fetchers = filtersFormData;
            bp_settings.checkers = itemsFormData;
            if (!bp_settings.action) {
                bp_settings.action = { actions: [] };
            }

            bp_settings.action.actions = actionsFormData;
        }

        this.setState({ initialBotData }, () => {
            this.closeBotConfigModal();
        });
    }

    checkIfPropositionExist() {
        const { propositions } = this.props;
        const { formData, schema } = this.state;

        return propositions?.find((el) => el.bp_type === schema && el.bp_name === formData?.bp_name);
    }

    render() {
        const { onClose } = this.props;
        const {
            bot, question, accept, isWorking, error, initialBotData,
            schemas, isFormChanged, formData, schema, propositions,
        } = this.state;

        const options = schemas
            ? Object.keys(schemas)
                .sort((a: string, b: string) => a.localeCompare(b))
                .map(key => ({ text: key, value: key }))
            : [];

        const schemaExist = !!schemas[this.state.schema.toString()];
        const currentSchema = schemaExist ? schemas[this.state.schema.toString()] : {};
        const href = `#/settings/bots/proposition/${propositions?.[0]?.bp_type}/${propositions?.[0]?.bp_name}`;
        const confirmationLink = `${location.origin}${location.pathname}${href}`;

        return <Window onClose={onClose}
                       title={`Бот ${this.state.botName ?? ''}`}
                       closeWithConfirm={isFormChanged}
                       error={this.state.error || this.props.error}>
            {this.state.isLoading
                ? <Spin/>
                : <div>
                    <div>
                        {
                            schemas
                                ? <Select options={options}
                                          placeholder={'Тип робота'}
                                          onSelect={this.onSchemaSelect.bind(this)}
                                          initialValues={this.state.schema ? [this.state.schema] : []}/>
                                : null
                        }
                        <FormConstructor schema={currentSchema}
                                         initialData={initialBotData ? deepCopy(initialBotData) : null}
                                         onChange={this.onChangeForm.bind(this)}/>
                    </div>
                    {propositions?.length
                        ? <div>
                            <Link href={href}>Предложение</Link>
                            &nbsp;—&nbsp;
                            <Copy text={confirmationLink}>
                                Копировать ссылку на предложение
                            </Copy>
                        </div>
                        : ''}
                    <div className={coreStyle.button_container}>
                        <CancelButton onClick={onClose.bind(this, false)}/>
                        <DeleteButton disabled={!schemaExist || !bot} onClick={this.onDeleteClick.bind(this)}/>
                        {this.state.schema === 'common_alerts'
                            ? <Button basic
                                      onClick={this.openBotConfigModal.bind(this)}
                                      ytLog={{ button_name: buttonNameDetails.CONFIGURE_BOT }}>
                                Конфигурация бота
                            </Button>
                            : null}

                        <BotProposeButton schemaExist={schemaExist}
                                          formData={formData}
                                          bp_type={schema?.toString()}
                                          checkIfPropositionExist={this.checkIfPropositionExist.bind(this)}
                                          isNew={!initialBotData || initialBotData?.bp_name !== formData?.bp_name}
                                          onClose={onClose.bind(this)}/>

                        <SaveButton disabled={!schemaExist}
                                    onClick={this.props.updateBot.bind(this, this.state)}/>
                    </div>
                </div>}
            {this.state.confirmIsOpen
                ? <Confirm question={question}
                           accept={accept.bind(this)}
                           isWorking={isWorking}
                           error={error}
                           onClose={this.closeDeleteModal.bind(this)}/>
                : null}
            {this.state.isOpenBotConfigModal
                ? <BotConfigModal bot={this.state.formData ?? {}}
                                  onSaveClick={this.onConfigurationSaveClick.bind(this)}
                                  onClose={this.closeBotConfigModal.bind(this)}/>
                : null}
        </Window>;
    }
}
