import React from 'react';
import { connect } from 'react-redux';
import { RouteComponentProps } from 'react-router';
import { withRouter } from 'react-router-dom';

import { Button, ButtonTypes } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import { ConstantsKey, fetchConstants } from '../../../../utils/fetchConstants';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import { deepCopy } from '../../../../utils/utils';
import { IStore } from '../../../App/store';
import { FormConstructor } from '../../../FormConstructor';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { IBotProposition } from '../component';
import style from '../index.css';
import { BOTS_REQUESTS, REQUESTS } from '../request';

interface IBotPropositionModalProps extends RouteComponentProps<{
    bot_name: string;
    bot_type: string;
}> {
    getConstants?: Promise<Record<string, any>>;
    propositions: IBotProposition[];
    onClose: (success: boolean) => void;
}

interface IBotPropositionModalState {
    isAcceptLoading: boolean;
    isRejectLoading: boolean;
    isLoading: boolean;
    error: Error | null;
    actionError: Error | null;
    currentSchema: Record<string, any> | null;
    currentBot: Record<string, any> | null;
}

class BotPropositionModal extends React.Component<IBotPropositionModalProps, IBotPropositionModalState> {
    state: IBotPropositionModalState = {
        isAcceptLoading: false,
        isRejectLoading: false,
        isLoading: false,
        error: null,
        actionError: null,
        currentSchema: null,
        currentBot: null,
    };
    request = new Request2({ requestConfigs: BOTS_REQUESTS });

    componentDidMount() {
        this.getBots();
    }

    componentDidUpdate(prevProps): void {
        if (!isObjectEqual(this.props.match, prevProps.match)) {
            this.getBots();
        }
    }

    getBots() {
        const { getConstants, match } = this.props;

        this.setState({ error: null, isLoading: true }, () => {
            getConstants
                ?.then(schemaResponse => {
                    const schemas = schemaResponse?.[ConstantsKey?.IFACE_RT_BACKGROUND_SETTINGS] ?? {};
                    const { bot_name, bot_type } = match.params;

                    bot_name &&
                    this.request.exec(REQUESTS.GET_BOTS, { queryParams: { ids: [bot_name] } })
                        .then(response => {
                            const currentSchema = schemas[bot_type ?? ''];
                            this.setState({
                                currentSchema,
                                isLoading: false,
                                currentBot: response.propositions?.[0],
                            });
                        })
                        .catch(error => {
                            this.setState({
                                error,
                                isLoading: false,
                            });
                        });
                })
                .catch(error => {
                    this.setState({ error, isLoading: false });
                });
        });
    }

    onReject() {
        const { currentBot } = this.state;
        const { onClose } = this.props;

        this.setState({ isRejectLoading: true, actionError: null }, () => {
            this.request.exec(REQUESTS.REJECT_BOT, {
                body: {
                    proposition_ids: [currentBot?.proposition_id ?? ''],
                },
                queryParams: {
                    comment: encodeURIComponent(currentBot?.proposition_description ?? ''),
                },
            })
                .then(() => {
                    this.setState({ isRejectLoading: false }, () => {
                        onClose(true);
                    });
                })
                .catch(actionError => {
                    this.setState({
                        actionError,
                        isRejectLoading: false,
                    });
                });
        });
    }

    onAccept() {
        const { currentBot } = this.state;
        const { onClose } = this.props;

        this.setState({ isAcceptLoading: true, actionError: null }, () => {
            this.request.exec(REQUESTS.CONFIRM_BOT, {
                body: {
                    proposition_ids: [currentBot?.proposition_id ?? ''],
                },
                queryParams: {
                    comment: encodeURIComponent(currentBot?.proposition_description ?? ''),
                },
            })
                .then(() => {
                    this.setState({ isAcceptLoading: false }, () => {
                        onClose(true);
                    });
                })
                .catch(actionError => {
                    this.setState({
                        actionError,
                        isAcceptLoading: false,
                    });
                });
        });
    }

    render() {
        const {
            actionError,isAcceptLoading, isRejectLoading,
            currentSchema, isLoading, error, currentBot,
        } = this.state;
        const { onClose } = this.props;

        return <Window onClose={onClose.bind(this)}
                       title={'Предложение'}
                       error={error}>
            {isLoading
                ? <Spin/>
                : <>
                    {currentSchema && currentBot
                        ? <FormConstructor schema={currentSchema}
                                           initialData={deepCopy(currentBot)}
                                           onChange={() => {
                                           }}/>
                        : <h4>Не удалось получить данные</h4>
                    }
                    <Link href={`#/settings/bots/item/${currentBot?.bp_name}`}>Базовое значение</Link>

                    {actionError
                        ? <SimpleError error={actionError}/>
                        : null
                    }

                    <div className={style.prop_button_container}>
                        <Button colorType={ButtonTypes.negative}
                                isLoading={isRejectLoading}
                                onClick={this.onReject.bind(this)}>
                                    Отклонить
                        </Button>

                        <Button colorType={ButtonTypes.positive}
                                isLoading={isAcceptLoading}
                                onClick={this.onAccept.bind(this)}>
                                    Принять
                        </Button>
                    </div>
                </>
            }
        </Window>;
    }
}

export function mapStateToProps() {
    return {};
}

export function mapDispatchToProps(dispatch) {
    return { getConstants: fetchConstants([ConstantsKey.IFACE_RT_BACKGROUND_SETTINGS], dispatch) };
}

export default withRouter<any, any>(
    connect<IStore, Record<string, unknown>>(mapStateToProps, mapDispatchToProps)(BotPropositionModal),
);
