import React from 'react';
import { renderToString } from 'react-dom/server';
import ReactTooltip from 'react-tooltip';
import XLSX from 'xlsx';

import { Link } from '../../../../ui/Link';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import style from './index.css';

interface IBotsStatWarningProps {
    bots: any[];
}

const CTYPE_NORMAL = ['prestable_maps', 'stable_maps', 'drive_frontend'];
const WARNING_ID = 'warning_id';

export class BotsStatWarning extends React.Component<IBotsStatWarningProps, any> {
    shouldComponentUpdate(
        nextProps: Readonly<IBotsStatWarningProps>,
        nextState: Readonly<any>,
        nextContext: any,
    ): boolean {
        return !isObjectEqual(nextProps.bots, this.props.bots);
    }

    getStat() {
        const stat = this.props.bots?.reduce((_p, _c) => {

            if (!CTYPE_NORMAL.some(el => _c?.bp_settings?.host_filter?.ctype?.includes(el))) {
                _p[`enabled_${_c?.bp_settings?.bp_enabled}`]?.ctype?.push(_c);
            }

            if (_c?.bp_settings?.host_filter?.fqdn_host_pattern) {
                _p[`enabled_${_c?.bp_settings?.bp_enabled}`]?.fqdn_host_pattern?.push(_c);
            }

            if (_c?.bp_settings?.host_filter?.host_pattern) {
                _p[`enabled_${_c?.bp_settings?.bp_enabled}`]?.host_pattern?.push(_c);
            }

            return _p;
        }, {
            enabled_true: {
                host_pattern: [],
                ctype: [],
                fqdn_host_pattern: [],
            },
            enabled_false: {
                host_pattern: [],
                ctype: [],
                fqdn_host_pattern: [],
            },
        });

        return stat;
    }

    download() {
        const wb = XLSX.utils.book_new();
        const ws_data: any = [];
        const bots = this.props?.bots;
        ws_data.push(
            [
                'bp_name',
                'ctype',
                'host_pattern',
                'fqdn_host_pattern',
                'bp_enabled',
            ]);
        bots?.map(bot => {
            ws_data.push(
                [
                    bot.bp_name,
                    bot.bp_settings?.host_filter?.ctype,
                    bot.bp_settings?.host_filter?.host_pattern,
                    bot.bp_settings?.host_filter?.fqdn_host_pattern,
                    `${bot?.bp_settings?.bp_enabled}`,
                ],
            );
        });
        const ws = XLSX.utils.aoa_to_sheet(ws_data);
        XLSX.utils.book_append_sheet(wb, ws, 'Bots');

        XLSX.writeFile(wb, `bots-${bots.length}.xlsx`);
    }

    render() {
        const { bots } = this.props;
        const stat = bots?.length && this.getStat() || null;

        return <div className={`${style.component}`}>
            <ReactTooltip id={WARNING_ID} html type={'info'}/>
            <h4>Нужно проверить настройку роботов: {bots?.length ?
                <Link onClick={this.download.bind(this)}>export {bots.length} bots XLSX</Link> : null}</h4>
            {
                stat ? <>
                    <div><strong>ВЫКЛ.</strong> <Line type={'enabled_false'} stat={stat}/></div>
                    <div><strong>ВКЛ.</strong> <Line type={'enabled_true'} stat={stat}/></div>
                </> : null
            }
        </div>;
    }
}

const Line = React.memo((props: { stat: any; type: string }) => {
    const { stat, type } = props;

    return <span>
        <LineItem key={'ctype'}
                  field={'ctype'}
                  data={stat?.[type]}
                  title_tip={`<div>Разрешённые вхождения:</div> <strong>${CTYPE_NORMAL.join(', ')}</strong>`}/>
        <LineItem key={'host_pattern'}
                  field={'host_pattern'}
                  data={stat?.[type]}
                  title_tip={'не пустой параметр <strong>host_pattern</strong>'}/>
        <LineItem key={'fqdn_host_pattern'}
                  field={'fqdn_host_pattern'}
                  data={stat?.[type]}
                  title_tip={'не пустой параметр <strong>fqdn_host_pattern</strong>'}/>
    </span>;
});

const LineItem = React.memo((props: { data: any; field: string; title_tip: string }) => {
    const { field, title_tip, data } = props;
    const dataTip = data?.[field].map((el, index) => <tr key={index}>
        <td><strong>{el.bp_name}</strong></td>
        <td>{el?.bp_settings?.host_filter?.['ctype']}</td>
        <td>{el?.bp_settings?.host_filter?.['host_pattern']}</td>
        <td>{el?.bp_settings?.host_filter?.['fqdn_host_pattern']}</td>
    </tr>);

    const tip = <table className={style.table}>
        <thead>
            <tr>
                <th>{field}</th>
                <th>ctype</th>
                <th>host_pattern</th>
                <th>fqdn_host_pattern</th>
            </tr>
        </thead>
        <tbody>{dataTip}</tbody>
    </table>;

    return <><i data-for={WARNING_ID} data-tip={title_tip}> {field}</i>:
        <strong data-tip={renderToString(tip)} data-for={WARNING_ID}> {data?.[field]?.length || 0}</strong>;</>;
});
