import React, { RefObject } from 'react';
import { Route, Switch } from 'react-router';
import ReactTooltip from 'react-tooltip';

import { Dict, LSSettingItems } from '../../../../types';
import { ONE_SECOND } from '../../../constants';
import { Button } from '../../../ui/Button';
import Checkbox, { CheckboxExternal } from '../../../ui/Checkbox';
import FormatDate from '../../../ui/FormatDate';
import { JsonModal } from '../../../ui/FullModal/JsonModal';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import Select, { WARNING_LABEL } from '../../../ui/Select';
import { TabItem, Tabs } from '../../../ui/Tabs';
import { IPlaneTreeItem, VirtualTreeList } from '../../../ui/VirtualTreeList';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import LS from '../../../utils/localStorage/localStorage';
import { Request2 } from '../../../utils/request';
import { deepCopy, deleteNull } from '../../../utils/utils';
import { Copy } from '../../Copy';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import BotGroupModal from './BotGroupModal/component';
import BotModal from './BotModal';
import BotModalStat from './BotModalStat';
import BotPropositionModal from './BotPropositions/BotPropositionModal';
import { BotPropositions } from './BotPropositions/component';
import { BotsStatWarning } from './BotsStatWaring';
import * as style from './index.css';
import { BOTS_REQUESTS, REQUESTS } from './request';
import Timeout = NodeJS.Timeout;

const PROPOSITION_ITEM_URL = '/settings/bots/proposition/';
const NEW_ITEM_URL = '/settings/bots/item/';
const NEW_STAT_URL = '/settings/bots/stat/';
const NEW_ITEM_BACK_URL = '/settings/bots';
const UPDATE_FREQUENCY = 5000;
const ALL_ATTRIBUTES_TAB = 'AllAttributes';

export interface IBP_STATE {
    last_event_id: string;
    status: string;
    type: string;
    last_execution: number;
    last_flush?: number;
    host: string;
    last_error: any;
}

export interface IBotProposition {
    bp_type?: string;
    bp_revision?: number;
    bp_name: string;
    bp_id?: string;
    bp_settings?: Record<string, any>;
    confirmations: any[];
    confirmations_count: number;
    confirmations_need: number;
    confirmators: any[];
    proposition_author: string;
    proposition_description: string;
    proposition_id: string;
}

export interface IBot {
    bp_type?: string;
    bp_revision?: number;
    bp_name: string;
    bp_id?: string;
    background_process_state?: IBP_STATE;
    propositions?: IBotProposition[];
    bp_settings?: {
        period?: number;
        bp_enabled: boolean;
        host_filter: {
            fqdn_host_pattern: string;
            ctype: string;
            host_pattern: string;
        };
        bp_description: string;
        freshness: string;
        bp_owners: any[];
        groupping_attributes: any[];
        tags_filter: string;
        fetchers: Dict<any>[];
        group_name?: string;
    };
}

enum Selected {
    selectedBotsTypes = 'selectedBotsTypes',
    selectedBotsCtypes = 'selectedBotsCtypes'
}

type SelectedType = {
    [selected in Selected]?: string[]
}

interface IBotsViewState extends SelectedType {
    isLoading: boolean;
    bots: any[];
    propositions: any[];
    users: Record<string, any>;
    filterValue: string;
    error: Error | null;
    upsertError: Error | null;
    isWorking: boolean;
    autoUpdate: boolean;
    isPropositions: boolean;
    attributesTabs: TabItem[];
    botsTypes: any[];
    botsCtypes: string[];
    currentTab: string | null;
    compactMode: boolean;

}

export default class BotsView extends React.Component<{}, IBotsViewState> {
    ls = new LS();
    state: IBotsViewState = {
        isLoading: false,
        bots: [],
        propositions: [],
        users: {},
        filterValue: '',
        error: null,
        upsertError: null,
        isWorking: false,
        autoUpdate: false,
        isPropositions: false,
        attributesTabs: [],
        botsTypes: [],
        botsCtypes: [],
        selectedBotsCtypes: this.ls.get(LSSettingItems.selectedBotsCtypes),
        selectedBotsTypes: this.ls.get(LSSettingItems.selectedBotsTypes),
        currentTab: ALL_ATTRIBUTES_TAB,
        compactMode: this.ls.get(LSSettingItems.compact_bots_view),
    };
    request = new Request2({ requestConfigs: BOTS_REQUESTS });
    listRef: RefObject<HTMLDivElement> | null = null;
    timer: Timeout;
    ref: RefObject<any>;

    constructor(props) {
        super(props);
        this.ref = React.createRef();
    }

    componentDidMount() {
        this.getBots(true);
    }

    componentWillUnmount(): void {
        clearTimeout(this.timer);
    }

    changeAutoUpdate(autoUpdate: boolean) {
        clearTimeout(this.timer);

        this.setState({ autoUpdate }, () => {
            if (autoUpdate) {
                this.getBots();
            }
        });
    }

    changePropositions(isPropositions: boolean) {
        this.setState({ isPropositions });
    }

    changeCompactMode(compactMode: boolean) {
        clearTimeout(this.timer);
        this.setState({ compactMode }, () => {
            this.ls.set(LSSettingItems.compact_bots_view, compactMode);
            this.getBots(true);
        });
    }

    getBots(force = false) {
        this.setState({
            isLoading: force,
        }, () => {
            this.request.exec(REQUESTS.GET_BOTS,
                {
                    queryParams: {
                        report: this.state.compactMode ? 'compact' : null,
                        bp_type: this.state[Selected.selectedBotsTypes]?.join(','),
                        ctype: this.state[Selected.selectedBotsCtypes]?.join(','),
                    },
                })
                .then(response => {
                    const bots: IBot[] = response?.rt_backgrounds || [];
                    const propositions: IBotProposition[] = response?.propositions || [];
                    const users: Record<string, string> = response?.users || {};

                    const type = new Set<string>();
                    const ctype = new Set<string>();
                    let attributes = [];

                    bots.forEach(b => {
                        type.add(b.bp_type || '');
                        ctype.add(b?.bp_settings?.host_filter?.ctype || '');
                        attributes = attributes.concat(...(b?.bp_settings?.groupping_attributes || []));
                    });

                    attributes = Array.from(new Set(attributes));

                    const attributesTabs = [ALL_ATTRIBUTES_TAB, ...attributes].map(attribute => {
                        return {
                            name: attribute,
                            link: attribute,
                            active: true,
                        };
                    });

                    propositions.map((proposition) => {
                        const index = bots.findIndex((el) => {
                            return el.bp_id === proposition.bp_id;
                        });

                        if (index !== -1) {
                            bots[index] = {
                                ...bots[index],
                                propositions: bots[index].propositions
                                    ? [...bots[index].propositions, proposition]
                                    : [proposition],
                            };
                        }
                    });

                    this.setState({
                        attributesTabs,
                        isLoading: false,
                        bots,
                        propositions,
                        users,
                        botsTypes: Array.from(type),
                        botsCtypes: Array.from(ctype),
                    }, () => {
                        if (this.state.autoUpdate) {
                            this.timer = setTimeout(() => {
                                this.getBots(false);
                            }, UPDATE_FREQUENCY);
                            this.ref.current?.getPlaneTree(false);
                        }
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        isLoading: false,
                    }, () => {
                        this.timer = setTimeout(() => {
                            this.getBots(false);
                        }, UPDATE_FREQUENCY);
                    });
                });
        });
    }

    treeBuilder(bots: IBot[]): IPlaneTreeItem<IBot>[] {
        return bots?.reduce((tree: IPlaneTreeItem<IBot>[], bot) => {
            const env = bot?.bp_settings?.host_filter
                ? [
                    bot.bp_settings.host_filter?.ctype,
                    bot.bp_settings.host_filter?.fqdn_host_pattern,
                    bot.bp_settings.host_filter?.host_pattern,
                ]
                    .filter(hostFilter => hostFilter)
                    .join('.')
                : '';

            const type = bot.bp_type || '';

            let currentEnvNode = tree.filter(treeNode => treeNode.data?.bp_name === env)[0];

            if (!currentEnvNode) {
                tree.push(
                    {
                        data: {
                            bp_name: env,
                        },
                        meta: {
                            id: env,
                            children: [],
                            nestingLevel: 0,
                            parentsIds: [],
                            active: false,
                            bots_included: {
                                enabled: 0,
                                disabled: 0,
                            },
                            className: style.bot_tree_item_env,
                        },
                    });
                currentEnvNode = tree.filter(treeNode => treeNode.data?.bp_name === env)[0];
            }

            let currentTypeNode = currentEnvNode.meta?.children
                ?.filter(treeNode => treeNode.data?.bp_type === type)[0];

            if (!currentTypeNode) {
                currentEnvNode.meta?.children?.push(
                    {
                        data: {
                            bp_name: type,
                            bp_type: type,
                        },
                        meta: {
                            id: `${env}.${type}`,
                            children: [],
                            nestingLevel: 1,
                            parentsIds: [env],
                            active: false,
                            bots_included: {
                                enabled: 0,
                                disabled: 0,
                            },
                            className: style.bot_tree_item_type,
                        },
                    });
                currentTypeNode = currentEnvNode.meta?.children
                    ?.filter(treeNode => treeNode.data?.bp_type === type)[0];
            }

            currentEnvNode.meta.children = currentEnvNode.meta.children?.sort((tree1, tree2) => {
                return tree1.meta.id.localeCompare(tree2.meta.id);
            }) as IPlaneTreeItem<IBot>[];

            const currentBot = {
                data: bot,
                meta: {
                    id: `${env}.${type}.${bot.bp_name}`,
                    children: null,
                    nestingLevel: 2,
                    active: true,
                    parentsIds: [env, type],
                },
            };

            const bots_included_env = currentEnvNode.meta.bots_included;
            const bots_included_type = currentTypeNode?.meta.bots_included;
            if (bot?.bp_settings?.bp_enabled) {
                bots_included_env.enabled += 1;
                bots_included_type.enabled += 1;
            } else {
                bots_included_env.disabled += 1;
                bots_included_type.disabled += 1;
            }

            currentTypeNode?.meta?.children?.push(currentBot);

            if (currentTypeNode) {
                currentTypeNode.meta.children = currentTypeNode?.meta.children?.sort((tree1, tree2) => {
                    return tree1.meta.id.localeCompare(tree2.meta.id);
                }) as IPlaneTreeItem<IBot>[];
            }

            return tree;
        }, [])
            .sort((tree1, tree2) => {
                return tree1.meta.id.localeCompare(tree2.meta.id);
            });
    }

    closeNotifierModal(success: boolean) {
        location.hash = NEW_ITEM_BACK_URL;
        this.setState({ upsertError: null });
        if (success) {
            this.getBots();
        }
    }

    closeStatNotifierModal() {
        location.hash = NEW_ITEM_BACK_URL;
        this.setState({ upsertError: null });
    }

    closePropositionModal(success: boolean) {
        location.hash = NEW_ITEM_BACK_URL;
        this.setState({ upsertError: null });
        if (success) {
            this.getBots();
        }
    }

    onFilterChange(filterValue: string) {
        this.setState({ filterValue });
    }

    onAddClick() {
        location.hash = NEW_ITEM_URL;
    }

    onModalClose() {
        location.hash = NEW_ITEM_BACK_URL;
    }

    filterTree() {
        const { currentTab } = this.state;
        if (this.state.filterValue || currentTab !== ALL_ATTRIBUTES_TAB) {
            return (notifier: IPlaneTreeItem<IBot>) => {
                const groupping_attributes = notifier.data?.bp_settings?.groupping_attributes ?? [];

                const isFilteredByAttribute = currentTab === ALL_ATTRIBUTES_TAB
                    ? true
                    : groupping_attributes.includes(currentTab);

                return isFilteredByAttribute
                    && notifier.data.bp_name?.toLowerCase().includes(this.state.filterValue?.toLowerCase());
            };
        }

        return null;

    }

    upsertBot(state: Dict<any>) {
        const { formData, initialBotData } = state;
        if (initialBotData && initialBotData.bp_name !== formData?.bp_name) {
            delete formData.bp_id;
            delete formData.bp_revision;
        }

        this.setState({ upsertError: null, isWorking: true }, () => {
            this.request.exec(REQUESTS.UPSERT_BOTS,
                {
                    body: {
                        backgrounds:
                            [Object.assign({}, formData, { bp_type: state.schema.toString() })],
                    },
                })
                .then(() => {
                    this.getBots();
                    this.onModalClose();
                })
                .catch(upsertError => {
                    this.setState({
                        upsertError, isWorking: false,
                    });
                });
        });
    }

    selectTab(currentTab: string) {
        this.setState({ currentTab });
    }

    onSelectType(key: Selected, value) {
        const state: SelectedType = {
            [key]: value,
        };

        this.setState(state, () => {
            this.ls.set(LSSettingItems[key], value);
            this.getBots(true);
        });
    }

    render() {
        const {
            bots,
            error,
            upsertError,
            isLoading,
            filterValue,
            autoUpdate,
            attributesTabs,
            currentTab,
            compactMode,
            isPropositions,
            propositions,
            users,
        } = this.state;
        if (error) {
            return <SimpleError error={this.state.error} data={{ label: 'Ошибка при загрузке ботов' }}/>;
        }

        const treeBuilder = this.treeBuilder.bind(this, bots);
        const treeFilter = this.filterTree.bind(this);

        const botsTypes = this.state.botsTypes.map(text => {
            return { text, value: text };
        });
        const botsCtypes = this.state.botsCtypes.map(text => {
            return { text, value: text };
        });

        return <div className={style.bots}>
            <Switch>
                <Route path={`${NEW_ITEM_URL}:bot_name?`}
                       render={() => {
                           return <BotModal updateBot={this.upsertBot.bind(this)}
                                            error={upsertError}
                                            propositions={propositions}
                                            onClose={this.closeNotifierModal.bind(this)}/>;
                       }}/>
                <Route path={`${PROPOSITION_ITEM_URL}:bot_type?/:bot_name?`}
                       render={() => {
                           return <BotPropositionModal propositions={propositions}
                                                       onClose={this.closePropositionModal.bind(this)}/>;
                       }}/>
                <Route path={`${NEW_STAT_URL}:bot_name?`}
                       render={() => {
                           return <BotModalStat onClose={this.closeStatNotifierModal.bind(this)}/>;
                       }}/>
            </Switch>

            <BotsStatWarning bots={this.state.bots}/>
            <div className={style.controls}>
                <div className={style.controls_row}>
                    <Input value={filterValue}
                           className={style.filter_input}
                           onChange={this.onFilterChange.bind(this)}
                           placeholder={'Фильтр'}/>
                    <Select options={botsTypes}
                            addingOddVariants
                            className={style.filter_input}
                            placeholder={'bp_type'}
                            multiSelect
                            initialValues={this.state[Selected.selectedBotsTypes]}
                            onSelect={this.onSelectType.bind(this, Selected.selectedBotsTypes)}/>
                    <Select options={botsCtypes}
                            className={style.filter_input}
                            placeholder={'ctype'}
                            addingOddVariants
                            multiSelect
                            initialValues={this.state[Selected.selectedBotsCtypes]}
                            onSelect={this.onSelectType.bind(this, Selected.selectedBotsCtypes)}/>
                    <div className={style.auto_update}>
                        <div className={style.title}>Сокращённый вид ({compactMode ? 'ВКЛ' : 'ВЫКЛ'}):</div>
                        <Checkbox onChange={this.changeCompactMode.bind(this)}
                                  checked={compactMode}
                                  placeholder={'Сокращённый вид'}/>
                    </div>
                </div>
                <div className={style.controls_row}>
                    <Button basic onClick={this.onAddClick.bind(this)}>Добавить робота</Button>
                    <div className={style.auto_update}>
                        <div className={style.title}>Автообновление:</div>
                        <Checkbox onChange={this.changeAutoUpdate.bind(this)}
                                  checked={autoUpdate}
                                  placeholder={'Автообновление'}/>
                    </div>

                    <div className={style.auto_update}>
                        <div className={style.title}>Предложения:</div>
                        <Checkbox onChange={this.changePropositions.bind(this)}
                                  checked={isPropositions}
                                  placeholder={'Показать предложения'}/>
                    </div>
                </div>
            </div>
            {isLoading
                ? <Spin/>
                : !isPropositions
                    ? <div>
                        <Tabs tabs={attributesTabs}
                              currentTab={currentTab || ALL_ATTRIBUTES_TAB}
                              selectTab={this.selectTab.bind(this)}/>
                        {bots.length
                            ? <>
                                <ReactTooltip id={'bots-tooltip'}
                                              html/>
                                <VirtualTreeList treeBuilder={treeBuilder}
                                                 ref={this.ref}
                                                 treeFilter={treeFilter}
                                                 initCollapsedLevels={[0, 1]}
                                                 isFilterValueExist={!!filterValue}
                                                 treeListItem={(props) => <IBotTreeItem {...props}
                                                                                        compactMode={compactMode}/>}/>
                            </>
                            : <h3>Не найдено ни одного бота</h3>}
                    </div>
                    : <BotPropositions propositions={propositions} users={users}/>
            }
        </div>;
    }
}

interface IBotTreeItemProps {
    item: IPlaneTreeItem<IBot>;
    showError: (error) => void;
    compactMode: boolean;
}

interface IBotTreeItemState {
    isEnabledChecked: boolean | null;
    isDisabled: boolean;
    isBotGroupModalOpen: boolean;
    groupName: string;
    error: Error | null;
    last_error: any;
}

export class IBotTreeItem extends React.Component<IBotTreeItemProps, IBotTreeItemState> {
    state: IBotTreeItemState = {
        isEnabledChecked: this.getCheckedState(),
        isDisabled: false,
        isBotGroupModalOpen: false,
        groupName: '',
        error: null,
        last_error: null,
    };
    request = new Request2({ requestConfigs: BOTS_REQUESTS });

    componentDidMount(): void {
        this.getEnabledData();
    }

    getCheckedState() {
        return this.props.item.data?.bp_settings?.hasOwnProperty('bp_enabled')
            ? this.props.item.data?.bp_settings?.bp_enabled
            : null;
    }

    getEnabledData() {
        const isEnabledChecked = this.getCheckedState();

        this.setState({ isEnabledChecked });
    }

    changeBotStatus() {
        const { item } = this.props;
        const { data } = item;

        const _state = deepCopy(data);

        const { bp_type } = _state;

        delete _state['background_process_state'];
        _state.bp_settings.bp_enabled = !_state.bp_settings.bp_enabled;
        deleteNull(_state);

        this.upsertBot({ _state, schema: bp_type });
    }

    upsertBot(state: Dict<any>) {
        this.setState({ isDisabled: true }, () => {
            this.request.exec(REQUESTS.UPSERT_BOTS,
                {
                    body: {
                        backgrounds:
                            [Object.assign({}, state._state, { bp_type: state.schema.toString() })],
                    },
                })
                .then(() => {
                    //Do not change isDisabled because user musn't change 'enabled'
                    this.setState({ isEnabledChecked: !this.state.isEnabledChecked });
                })
                .catch(error => {
                    this.setState({
                        error, isDisabled: false,
                    });
                });
        });
    }

    openBotGroupModal(groupName: string) {
        this.setState({ isBotGroupModalOpen: true, groupName });
    }

    closeBotGroupModal() {
        this.setState({ isBotGroupModalOpen: false, groupName: '' });
    }

    showErrors(last_error) {
        if (isValidJSONString(last_error)) {
            last_error = JSON.parse(last_error);
        } else {
            last_error = { data: last_error };
        }

        this.setState({ last_error });
    }

    render() {
        const { isBotGroupModalOpen, groupName } = this.state;
        if (this.state.error) {
            return <SimpleError error={this.state.error}
                                data={{ label: 'Ошибка при обновлении бота!!!' }}/>;
        }

        const { item } = this.props;
        const { data } = item;
        const { bp_name: name, bp_type = '', bp_settings } = data;
        const group_name = bp_settings?.group_name;
        const bp_description = bp_settings?.bp_description ?? '';
        const bots_included = item.meta?.bots_included ?? null;
        const ctype = bp_settings?.host_filter?.ctype ?? null;

        const last_error = item?.data?.background_process_state?.last_error;

        return <div className={style.bot_tree_item}>
            <div title={name} className={style.id}>
                {bp_settings
                    ? <>
                        <div className={style.key_value_container}>
                            {ctype &&
                                <span className={style.golovan} title={'GOLOVAN'}>
                                    <Link href={`https://yasm.yandex-team.ru/template/panel/RTRobotAdmin/rt_name=${name};ctype=${ctype}`}
                                          target={'_blank'}>📈</Link>
                                </span>
                            }
                            {bp_type === 'common_alerts' && group_name
                                ? <Link className={style.group_link}
                                        title={group_name}
                                        onClick={this.openBotGroupModal.bind(this, group_name)}>{group_name}:</Link>
                                : null}
                        </div>
                        <div className={style.key_value_container}>
                            <Link title={name}
                                  className={style.bot_name}
                                  href={`#${NEW_ITEM_URL}${encodeURIComponent(name)}`}>
                                {!name ? WARNING_LABEL : ''}{name}
                            </Link>
                        </div>
                        <div className={style.description} title={bp_description}>{bp_description}</div>
                    </>
                    : <div className={style.title}>
                        <span title={name}>
                            {!name ? WARNING_LABEL : ''}{name}
                        </span>
                        {bots_included
                            ? <div className={style.bot_included}>
                                <div className={style.enabled}>{bots_included.enabled}</div>
                                :
                                <div className={style.disabled}>{bots_included.disabled}</div>
                            </div>
                            : null}
                    </div>}
            </div>
            <div className={style.enabled}>
                {data.hasOwnProperty('bp_settings')
                && data?.bp_settings?.hasOwnProperty('bp_enabled')
                && !this.props.compactMode
                    ? <CheckboxExternal onChange={this.changeBotStatus.bind(this)}
                                        disabled={this.state.isEnabledChecked === null || this.state.isDisabled}
                                        checked={this.state.isEnabledChecked ?? false}/>
                    : null}
            </div>
            <div className={style.propositions}>
                {data?.propositions?.length
                    ? <Link href={`#/settings/bots/proposition/${data?.bp_type}/${data?.bp_name}`} className={style.link}>
                        Предложение
                    </Link>
                    : null}
            </div>
            <div className={style.period}>
                {data?.bp_settings?.period
                    ?
                    <>
                        <span className={style.key}>Частота</span>
                        <span className={style.value}>{data?.bp_settings?.period}</span>
                    </>
                    : null}
                {
                    this.props.compactMode
                    && data.bp_revision
                    && <Link title={name}
                             className={style.bot_name}
                             href={`#${NEW_STAT_URL}${encodeURIComponent(data.bp_name)}`}>
                        статус
                    </Link>
                }
            </div>
            {data.bp_revision
                ? <div className={style.revision}>
                    Ревизия: {data.bp_revision}
                </div>
                : null}

            <div className={style.status}>

                {data.background_process_state
                    ? <>
                        <div className={style.key_value_container}>
                            <div className={style.key_value_container}>
                                <span className={style.key}>Статус</span>
                                <span className={style.value}>{data.background_process_state?.status}</span>
                            </div>
                            <div className={style.key_value_container}>
                                <span className={style.key}>Тип</span>
                                <span className={style.value}
                                      title={data.background_process_state?.type}>
                                    <Copy externalTooltipId={'bots-tooltip'}>
                                        {data.background_process_state?.type}
                                    </Copy>
                                </span>
                            </div>
                        </div>
                        <div className={style.key_value_container}>
                            <span className={style.key}>Host</span>
                            <span title={data.background_process_state?.host}
                                  className={style.value}>
                                <Copy externalTooltipId={'bots-tooltip'}>{data.background_process_state?.host}</Copy>
                            </span>
                        </div>
                        <div className={style.key_value_container}>
                            {last_error && <Link className={style.errors}
                                                 onClick={this.showErrors.bind(this, last_error)}>⚡⚡⚡ Показать
                                ошибки ⚡⚡⚡</Link>}
                        </div>
                    </>
                    : null}
            </div>
            <div className={style.exec_data}>
                {data.background_process_state
                    ? <>
                        {data.background_process_state?.last_execution
                            ? <div className={style.key_value_container}>
                                <span className={style.key}>Последний запуск</span>
                                <span className={style.value}>
                                    <FormatDate withSecond={true}
                                                value={data.background_process_state?.last_execution * ONE_SECOND}/>
                                </span>
                            </div>
                            : null}
                        {data.background_process_state.last_event_id
                            ? <div className={style.key_value_container}>
                                <span className={style.key}>Last event id</span>
                                <span className={style.value}>
                                    <Copy externalTooltipId={'bots-tooltip'}>
                                        {data.background_process_state.last_event_id}
                                    </Copy>
                                </span>
                            </div>
                            : null}
                    </>
                    : null}
            </div>
            {
                this.state.last_error && <JsonModal obj={this.state.last_error}
                                                    onClose={this.showErrors.bind(this, null)}/>
            }
            {isBotGroupModalOpen
                ? <BotGroupModal onClose={this.closeBotGroupModal.bind(this)} groupName={groupName}/>
                : null}
        </div>;
    }
}
