import React, { useEffect, useState } from 'react';

import { Dict } from '../../../../../types';
import { EMPTY_DATA } from '../../../../constants';
import { getRenderedBBCode } from '../../../../ui/BBCodeRenderer';
import { WARNING_LABEL } from '../../../../ui/Select';
import { PRE_ACTION_MESSAGE_TYPE, SUPPORT_CHAT } from '../constants';
import { IChat, INode } from '../types';
import * as style from './index.css';

interface IChatIntroScreenPreviewProps {
    node: INode | null;
    chat: IChat | null;
    setNextNode?: (node_id: string) => void;
}

const SEPARATOR = '|';

export const ChatIntroScreenPreview = React.memo((props: IChatIntroScreenPreviewProps) => {
    const { node, chat } = props;
    const [activeOption, setActiveOptions]: any = useState(null);
    const isSupportChat = chat?.name === SUPPORT_CHAT;

    const getActiveOptions = (): any => {
        const optionsArray = node?.schema?.options ?? [];
        const chosenOption = optionsArray.find(el => {
            return el.text === activeOption?.text;
        });

        return chosenOption?.options ?? [];
    };

    const showOptions = (optionsArray: any): any => {
        return optionsArray.map((option, ind) => {
            return <div key={option.text + ind}
                        className={style.control_container}
                        onClick={isSupportChat ? setActiveOptions.bind(null, option) : null}>
                <div className={style.control}>
                    {option.text}
                </div>
            </div>;
        });
    };

    const getActiveOptionMessage = (): any[] => {
        return activeOption?.node?.split(SEPARATOR) ?? [];
    };

    const checkForTransition = () => {
        if (!activeOption?.options) {
            const nextId = getActiveOptionMessage()?.[0];
            nextId && props.setNextNode?.(nextId);
        }
    };

    useEffect(() => {
        checkForTransition();
    }, [activeOption]);

    useEffect(() => {
        setActiveOptions(null);
    }, [node?.id]);

    const preActionMessages = node?.pre_action_messages ?? [];
    const chatTheme = chat?.meta?.theme;
    const styles: Dict<any> = chatTheme?.styles ?? {};

    return <div className={style.chat_node_preview_container}>
        <div className={style.chat_node_preview} style={{ backgroundColor: styles?.bg_color }}>
            {node?.id
                ? <>
                    {preActionMessages.map((preActionMessage, index) => {
                        const { type } = preActionMessage;

                        switch (type) {
                        case PRE_ACTION_MESSAGE_TYPE.PLAIN_TEXT:
                            return <div key={index} className={style.chat_node_message}>
                                {getRenderedBBCode(preActionMessage.text)}
                            </div>;
                        case PRE_ACTION_MESSAGE_TYPE.WELCOME_TEXT:
                            return <div key={index} className={style.welcome}>
                                {getRenderedBBCode(preActionMessage.text)}
                            </div>;
                        case PRE_ACTION_MESSAGE_TYPE.IMAGE:
                            const { text: imageLink } = preActionMessage;
                            const imageAlt = imageLink?.split('/')
                                [imageLink?.split('/').length - 1]?.split('.')[0];

                            return <img key={index}
                                        src={imageLink}
                                        alt={imageAlt}
                                        title={imageAlt}
                                        className={`${style.chat_node_message} ${style.image_message}`}/>;
                        case PRE_ACTION_MESSAGE_TYPE.IMAGE_LINK:
                            const { icon: imageLinkLink, link: href, text } = preActionMessage;
                            const imageLinkAlt = text;

                            return <a href={href} key={index}>
                                <img src={imageLinkLink}
                                     alt={imageLinkAlt}
                                     title={imageLinkAlt}
                                     className={`${style.chat_node_message} ${style.image_link_message}`}/>
                            </a>;
                        case PRE_ACTION_MESSAGE_TYPE.IMAGES_LIST:
                            return preActionMessage.text?.split(',')
                                ?.map((imageLink, index) => {
                                    return <img key={`${imageLink}_${index}`}
                                                src={imageLink}
                                                alt={imageLink}
                                                className={`${style.chat_node_message} ${style.image_message}`}/>;
                                });
                        case PRE_ACTION_MESSAGE_TYPE.VIDEO:
                            return <div className={`${style.chat_node_message} ${style.video_message}`}>
                                <video src={preActionMessage.text}
                                       className={style.video_big}
                                       autoPlay={false}
                                       controls/>
                            </div>;
                        default:
                            return <div key={index}
                                        className={`${style.chat_node_message} ${
                                            preActionMessage.resource_link ? style.video_message : style.unknown_type
                                        }`}>
                                {
                                    preActionMessage.resource_link
                                        ? <video src={preActionMessage.resource_link}
                                                 className={style.video_big}
                                                 autoPlay={false}
                                                 controls/> : <>{WARNING_LABEL} не найдена ссылка на ресурс</>
                                }
                            </div>;
                        }
                    })}

                    {node?.action_button_text
                        ? <div className={style.control_container}>
                            <div className={style.control}>
                                {node?.action_button_text ?? EMPTY_DATA}
                            </div>
                        </div>
                        : null
                    }

                    {showOptions(
                        activeOption
                            ? getActiveOptions()
                            : node?.schema?.options ?? [],
                    )}
                </>
                : <div className={style.no_selected_node}>Выберите ноду чата для просмотра</div>}
        </div>

        <div className={`${style.reset_button} ${!activeOption ? style.disable : ''}`}
             onClick={activeOption ? setActiveOptions.bind(null, null) : null}>
            Сбросить
        </div>
    </div>;
});
