import React from 'react';

import { Dict } from '../../../../types';
import { ONE_SECOND } from '../../../constants';
import { DeleteButton, SaveButton } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Confirm } from '../../../ui/FullModal';
import * as coreStyle from '../../../ui/index.css';
import { Input } from '../../../ui/Input';
import Select, { IOptionInfo } from '../../../ui/Select';
import { IPlaneTreeItem, VirtualTreeList } from '../../../ui/VirtualTreeList';
import { Request2 } from '../../../utils/request';
import { deepCopy } from '../../../utils/utils';
import { FormConstructor } from '../../FormConstructor';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { LANDINGS_REQUESTS, REQUESTS as LANDING_REQUESTS } from '../Landings/request';
import { NOTIFIERS_REQUESTS, REQUESTS as NOTIFIER_REQUESTS } from '../Notifiers/request';
import { INotifier } from '../Notifiers/types';
import { ChatIntroScreenPreview } from './ChatIntroscreenPreview/component';
import { CHAT_NODE_SCHEMA, SUPPORT_CHAT } from './constants';
import { IChatsEditorProps } from './index';
import * as style from './index.css';
import { IChat, INode } from './types';

const DEEPLINKS_SPLITTER = ';';

const EMPTY_NODE_ID = '__EMPTY_NODE';

interface IChatsEditorState {
    notifiers: INotifier[];
    landings: any[];
    chats: IChat[];
    isWorking: boolean;
    dataIsLoading: boolean;
    loadingError: Error | null;
    isOnlyLandings: boolean;
    filterValue: string;
    selectedTreeNode: any;
    chatFormData: Dict<any>;
    savingError: Error | null;
    removingError: Error | null;
    chatNameOptions: IOptionInfo[];
    selectedChat: string | null;
    deleteChatConfirmIsOpen: boolean;
    chatNodeSchema: Dict<any>;
    isSaving: boolean;
}

export default class ChatsEditor extends React.Component<IChatsEditorProps, IChatsEditorState> {
    state: IChatsEditorState = {
        notifiers: [],
        landings: [],
        chats: [],
        isWorking: false,
        dataIsLoading: false,
        loadingError: null,
        isOnlyLandings: true,
        filterValue: '',
        selectedTreeNode: {},
        chatFormData: {},
        savingError: null,
        removingError: null,
        chatNameOptions: [],
        selectedChat: null,
        deleteChatConfirmIsOpen: false,
        chatNodeSchema: CHAT_NODE_SCHEMA,
        isSaving: false,
    };
    requestNotifier = new Request2({ requestConfigs: NOTIFIERS_REQUESTS });
    requestLandings = new Request2({ requestConfigs: LANDINGS_REQUESTS });

    componentDidMount() {
        this.getNotifiers(true);
        this.getLinks();
    }

    getNotifiers(force = false) {
        this.setState({
            loadingError: null,
            dataIsLoading: force,
        }, () => {
            Promise.all([
                this.requestNotifier.exec(NOTIFIER_REQUESTS.GET_NOTIFIERS),
                this.requestLandings.exec(LANDING_REQUESTS.GET_LANDINGS),
            ])
                .then(response => {
                    const [notifiersResponse, landingsResponse] = response;
                    const notifiers: any[] = notifiersResponse?.objects || [];
                    const landings: any[] = landingsResponse?.landings || [];

                    const chats: IChat[] = notifiers.filter(notifier => notifier.type === 'native_chat');
                    const chatNameOptions: IOptionInfo[] = chats.map(chat => {
                        return {
                            value: chat.name,
                            text: chat.name,
                        };
                    });

                    this.setState({
                        dataIsLoading: false, chats, notifiers, landings, chatNameOptions
                        , isSaving: false,
                    });
                })
                .catch(loadingError => {
                    this.setState({
                        loadingError,
                        dataIsLoading: false,
                    });
                });
        });
    }

    getLinks() {
        const deeplinks = this.props?.app_deeplink?.split(DEEPLINKS_SPLITTER).map(link => `yandexdrive://${link.trim()}`);
        const chatNodeSchema = deepCopy(this.state.chatNodeSchema);

        if (chatNodeSchema?.__introScreenType?.variants_fields?.deeplink?.link) {
            chatNodeSchema.__introScreenType.variants_fields.deeplink.link.templateList = deeplinks;
        }

        this.setState({ chatNodeSchema });
    }

    chatListTreeBuilder(): IPlaneTreeItem<IChat | INode>[] {
        const { landings, chats } = this.state;

        const landings_chats = landings.filter(landing => landing.landing_chat_id).map(landing => {
            return {
                landing_chat_messages_group: landing.landing_chat_messages_group,
                landing_chat_id: landing.landing_chat_id,
            };
        });

        return chats.reduce((result: IPlaneTreeItem<IChat | INode>[], chat) => {
            const isIncludesLandings = landings_chats
                .some(landings_chat => landings_chat.landing_chat_id === chat.name);

            const children = chat.meta.chat_script.items.map(node => {
                const isLanding = landings_chats.some(landings_chat => landings_chat.landing_chat_id === chat.name
                    && landings_chat.landing_chat_messages_group === node.id);

                return {
                    data: node,
                    meta: {
                        landing: {
                            isConnected: isLanding,
                        },
                        id: `${chat.name}.${node.id}`,
                        children: null,
                        nestingLevel: 1,
                        parentsIds: [chat.name],
                        __isNewNodeItem: false,
                        active: true,
                    },
                };
            })
                .sort((child1, child2) => {
                    return child1.meta.id.localeCompare(child2.meta.id);
                });

            children.push({
                data: {} as INode,
                meta: {
                    landing: {
                        isConnected: false,
                    },
                    id: `${chat.name}.${EMPTY_NODE_ID}`,
                    children: null,
                    nestingLevel: 1,
                    parentsIds: [chat.name],
                    __isNewNodeItem: true,
                    active: true,
                },
            });

            result.push({
                data: chat,
                meta: {
                    landing: {
                        isConnected: isIncludesLandings,
                    },
                    id: chat.name,
                    children,
                    nestingLevel: 0,
                    parentsIds: [],
                    active: true,
                },
            });

            return result;
        }, []);
    }

    onSelect(selectedTreeNode: IPlaneTreeItem<INode>) {
        const selectedChat: string = selectedTreeNode.meta.parentsIds?.[0] ?? '';
        this.setState({ selectedChat, selectedTreeNode: deepCopy(selectedTreeNode) });
    }

    onChatSelect(selectedChat: string) {
        this.setState({ selectedChat });
    }

    treeFilter(): ((node: IPlaneTreeItem<IChat & INode>) => boolean) | null {
        const { filterValue, isOnlyLandings } = this.state;
        if (filterValue || isOnlyLandings) {
            return (node) => {
                const isFilteredByValue = node.meta?.id?.toLowerCase()?.includes(filterValue?.toLowerCase())
                    || node.data?.name?.toLowerCase()?.includes(filterValue?.toLowerCase());

                const parentsIds = node?.meta?.parentsIds || [];
                const isFilteredByLanding = isOnlyLandings ? parentsIds.includes('static_announcements') : true;

                return isFilteredByValue && isFilteredByLanding;
            };
        }

        return null;
    }

    onOnlyLandingsChange(isOnlyLandings: boolean) {
        this.setState({ isOnlyLandings });
    }

    onFilterChange(filterValue: string) {
        this.setState({ filterValue });
    }

    onFormChange(chatFormData: Dict<any>) {
        this.setState({ chatFormData });
    }

    onSaveClick() {
        this.setState({ savingError: null, isSaving: true }, () => {
            const chatFormData: any = this.state.chatFormData;
            const { chats, selectedChat } = this.state;
            let savingError: Error | null = null;
            const nodeParent = selectedChat ?? null;

            if (!chatFormData.next_step) {
                chatFormData.next_step = null;
            }

            if (nodeParent) {
                const currentChat = chats?.filter(chat => chat.name === nodeParent)?.[0];
                const currentChatNodes = currentChat?.meta?.chat_script?.items ?? null;

                if (currentChatNodes) {
                    let isChatIDExist = false;
                    const newCurrentChatNodes = currentChatNodes.map(currentChatNode => {
                        if (currentChatNode.id === chatFormData.id) {
                            isChatIDExist = true;

                            return chatFormData;
                        }

                        return currentChatNode;
                    });
                    if (!isChatIDExist) {
                        newCurrentChatNodes.push(chatFormData);
                    }

                    currentChat.meta.chat_script.items = newCurrentChatNodes;
                    const data = { objects: [Object.assign({}, currentChat, { type: 'native_chat' })] };

                    this.requestNotifier.exec(NOTIFIER_REQUESTS.UPSERT_NOTIFIERS, { body: data })
                        .then(() => {

                            setTimeout(() => {
                                this.getNotifiers(true);
                            }, ONE_SECOND);

                        })
                        .catch(savingError => {
                            this.setState({
                                savingError,
                                isWorking: false,
                                isSaving: false,
                            });
                        });
                } else {
                    savingError = new Error('Найденный чат не имеет массива нод');
                }
            } else {
                savingError = new Error('Не найден чат для добавления ноды');
            }

            if (savingError) {
                this.setState({ savingError });
            }
        });
    }

    openDeleteConfirm() {
        this.setState({ deleteChatConfirmIsOpen: true });
    }

    closeDeleteConfirm() {
        this.setState({ deleteChatConfirmIsOpen: false });
    }

    onDelete() {
        this.setState({ removingError: null, isWorking: true }, () => {

            const chatFormData: any = this.state.chatFormData;

            const { chats, selectedChat } = this.state;
            let removingError: Error | null = null;

            const nodeParent = selectedChat ?? null;

            if (!chatFormData.next_step) {
                chatFormData.next_step = null;
            }

            if (nodeParent) {
                const currentChat = chats?.filter(chat => chat.name === nodeParent)?.[0];
                const currentChatNodes = currentChat?.meta?.chat_script?.items ?? null;

                if (currentChatNodes) {
                    currentChat.meta.chat_script.items = currentChatNodes.filter(currentChatNode => {
                        return currentChatNode.id !== chatFormData.id;
                    });

                    const data = { objects: [Object.assign({}, currentChat, { type: 'native_chat' })] };

                    this.requestNotifier.exec(NOTIFIER_REQUESTS.UPSERT_NOTIFIERS, { body: data })
                        .then(() => {
                            this.getNotifiers(true);
                            this.closeDeleteConfirm();
                        })
                        .catch(removingError => {
                            this.setState({
                                removingError,
                                isWorking: false,
                            });
                        });
                } else {
                    removingError = new Error('Найденный чат не имеет массива нод');
                }
            } else {
                removingError = new Error('Не найден чат для добавления ноды');
            }

            if (removingError) {
                this.setState({ removingError });
            }
        });
    }

    setNextNode(id: string) {
        const supportNotifier: any = this.state.notifiers?.find(notif => {
            return notif.name === SUPPORT_CHAT;
        });

        const allNodes = supportNotifier.meta?.chat_script?.items ?? [];
        const selectedNode = allNodes.find(node => node.id === id);

        if (selectedNode) {
            const selectedTreeNode = {
                data: selectedNode,
                meta: {
                    landing: {},
                    id: selectedNode.name,
                    nestingLevel: 0,
                    parentsIds: [],
                },
            };

            const chatFormData = {
                action_type: selectedNode.action_type,
                id: selectedNode.id,
                pre_action_messages: selectedNode.pre_action_messages,
                schema: selectedNode.schema,
            };

            this.setState({ selectedTreeNode, chatFormData });
        }
    }

    render() {
        const {
            chats,
            isOnlyLandings,
            filterValue,
            selectedTreeNode,
            chatFormData,
            dataIsLoading,
            chatNameOptions,
            selectedChat,
            deleteChatConfirmIsOpen,
            removingError,
            loadingError,
            savingError,
            chatNodeSchema,
            isSaving,
        } = this.state;
        const node = chatFormData as INode ?? selectedTreeNode?.data ?? null;
        const chat = selectedChat ? chats.filter(chat => chat.name === selectedChat)[0] : null;

        return <div className={style.chats_editor}>
            <div className={style.chats_nodes_list}>
                {loadingError
                    ? <SimpleError error={loadingError} data={{ label: 'Ошибка при загрузке чатов' }}/>
                    : dataIsLoading
                        ? <Spin/>
                        : <>
                            <div className={style.controls}>
                                <Input onChange={this.onFilterChange.bind(this)}
                                       value={filterValue}
                                       placeholder={'Фильтр'}/>
                                <div className={style.checkbox_wrapper}>
                                    <div className={style.controls_select}>
                                        <div className={style.controls_select_title}>Лендинги:</div>
                                        <Checkbox checked={isOnlyLandings}
                                                  onChange={this.onOnlyLandingsChange.bind(this)}/>
                                    </div>
                                </div>
                            </div>
                            <VirtualTreeList treeBuilder={this.chatListTreeBuilder.bind(this, chats)}
                                             treeFilter={this.treeFilter.bind(this)}
                                             isFilterValueExist={!!filterValue}
                                             treeListItem={[ChatListItem, {
                                                 onSelect: this.onSelect.bind(this),
                                                 selectedTreeNodeId: selectedTreeNode?.meta?.id,
                                             }]}/>
                        </>}
            </div>
            <div className={style.form_container}>
                <Select options={chatNameOptions}
                        placeholder={'Родительский чат'}
                        initialValues={selectedChat ? [selectedChat] : []}
                        onSelect={this.onChatSelect.bind(this)}/>
                <FormConstructor schema={chatNodeSchema}
                                 className={style.form}
                                 initialData={selectedTreeNode.data}
                                 onChange={this.onFormChange.bind(this)}/>
                {savingError
                    ? <SimpleError error={savingError} data={{ label: 'Ошибка при сохранении ноды' }}/>
                    : null}
                <div className={coreStyle.button_container}>
                    <DeleteButton disabled={!selectedTreeNode?.data?.id
                    || selectedTreeNode?.data?.id !== chatFormData?.id}
                                  onClick={this.openDeleteConfirm.bind(this)}/>
                    <SaveButton onClick={this.onSaveClick.bind(this)} isLoading={isSaving}/>
                </div>
            </div>
            <ChatIntroScreenPreview node={node}
                                    chat={chat}
                                    setNextNode={this.setNextNode.bind(this)}/>
            {deleteChatConfirmIsOpen
                ? <Confirm question={`Удалить ноду ${selectedTreeNode.data.id}?`}
                           error={removingError}
                           accept={this.onDelete.bind(this)}
                           onClose={this.closeDeleteConfirm.bind(this)}/>
                : null}
        </div>;
    }
}

class ChatListItem extends React.Component<any, any> {

    render() {
        const { item, onSelect, selectedTreeNodeId } = this.props;
        const { data, meta } = item;
        const isNode = !item.data?.meta?.chat_id;
        const isIntroScreen = item.meta?.landing?.isConnected;

        const isSelected = selectedTreeNodeId && (selectedTreeNodeId === meta?.id);
        const isNewNodeItem = item.meta.__isNewNodeItem;

        return <div className={style.chat_list_item} onClick={isNode ? onSelect?.bind(this, item) : null}>
            {isNewNodeItem
                ? <div className={style.new_item}>
                    Добавить
                </div>
                : <>
                    <div className={`${style.item_id} ${isNode ? style.node : style.chat} ${isSelected ? style.selected : ''}`}
                         title={item.meta.id}>
                        {data?.name || data?.id}
                    </div>
                    {isIntroScreen
                        ? <div className={style.landing_icon_container}>
                            <div className={style.landing_icon}
                                 title={isNode ? 'Интроскрин' : 'Чат содержит интроскрин'}/>
                        </div>
                        : null}
                </>}
        </div>;
    }
}
