import React from 'react';
import ReactFlow, { Background, BackgroundVariant, Controls } from 'react-flow-renderer';

import { Button } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import { ViewerJSON } from '../../../ui/FullModal/ViewerJSON';
import Select, { IOptionInfo } from '../../../ui/Select';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { IChat } from '../ChatsEditor/types';
import { NOTIFIERS_REQUESTS, REQUESTS } from '../Notifiers/request';
import { ChatsTreeInfoBlock } from './ChatsTreeInfoBlock';
import style from './index.css';

const DAGRE = require('dagre');

const INTRO_ID = 'intro';
const CHAT_TYPE = 'native_chat';
const SPLITTER = '|';
const CONDITION_ELEMENT = '%';
const NODE_WIDTH = 230;
const NODE_HEIGHT = 50;
const DEFAULT_ZOOM = 0.8;
const MIN_ZOOM = 0.03;

const OPTION_NODE_TYPES = {
    options: 'options',
    message: 'message',
};

const ACTION_TYPES = {
    context_buttons: 'context_buttons',
    tree: 'tree',
    chat_closed: 'chat_closed',
};

export enum INFO_BLOCK_DATA_TYPES {
    introscreen = 'introscreen',
    option = 'option',
    edge = 'edge',
}

interface IChatsTreeState {
    error: Error | null;
    isNotifiersLoading: boolean;
    notifierJSONModalOpen: boolean;
    notifiers: IChat[];
    branches: IOptionInfo[];

    currentNotifier: IChat;
    activeNode: any;
    graphData: any[];
}

export class ChatsTree extends React.Component<any, IChatsTreeState> {
    state: IChatsTreeState = {
        error: null,
        isNotifiersLoading: false,
        notifierJSONModalOpen: false,
        notifiers: [],
        branches: [],

        currentNotifier: {} as IChat,
        activeNode: {},
        graphData: [],
    };
    request = new Request2({ requestConfigs: NOTIFIERS_REQUESTS });
    graph: any;

    componentDidMount() {
        this.getNotifiers();
    }

    getNotifiers() {
        this.setState({
            error: null,
            isNotifiersLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_NOTIFIERS)
                .then(response => {
                    let notifiers: IChat[] = response?.objects || [];
                    notifiers = notifiers.filter(el => el.type === CHAT_TYPE);
                    this.setState({
                        isNotifiersLoading: false,
                        notifiers,
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        isNotifiersLoading: false,
                    });
                });
        });
    }

    getGraphData(branchName) {
        this.graph = new DAGRE.graphlib.Graph();
        this.graph.setGraph({
            ranksep: 70,
            edgesep: 0,
            nodesep: 0,
        });

        this.graph.setDefaultEdgeLabel(() => {
            return {};
        });

        const branchNode = this.getBranchNode(branchName) ?? {};
        const initialNodeId = branchNode.text;
        this.setGNode(initialNodeId);
        this.addChildrenNodes(branchNode);

        this.layoutGraph();
    }

    addChildrenNodes(parent) {
        if (parent.type) {
            if (parent.type === OPTION_NODE_TYPES.options) {
                const parentId = parent.text;
                const nodes = parent.options ?? [];

                nodes.forEach(node => {
                    this.setNodeAndEdge(node.text, parentId);
                    this.addChildrenNodes(node);
                });
            } else if (parent.type === OPTION_NODE_TYPES.message) {
                const curNodeId = parent.node.split(SPLITTER)?.[0];
                const curNode = this.getNodeById(curNodeId) ?? {};

                if (curNode.action_type === ACTION_TYPES.context_buttons) {
                    curNode.schema.options.forEach(option => {
                        const nodeId = this.parseNodeId(option, parent);

                        if (!this.graph._nodes?.[nodeId]) {
                            this.setNodeAndEdge(nodeId, parent.text);
                            this.addChildrenNodes(option);
                        } else {
                            this.graph.setEdge(parent.text, nodeId);
                        }
                    });
                }
            }
        }

        return;
    }

    setNodeAndEdge(nodeId, parentId) {
        this.setGNode(nodeId);
        this.graph.setEdge(parentId, nodeId);
    }

    setGNode(node_id) {
        this.graph.setNode(node_id, {
            id: node_id,
            label: node_id,
            width: NODE_WIDTH,
            height: NODE_HEIGHT,
        });
    }

    parseNodeId(node, parentNode) {
        if (node.text?.includes(CONDITION_ELEMENT)) {
            const index = node.text.slice(1);
            const options = parentNode.node?.split(SPLITTER);

            return options[index];
        }

        return node.text;

    }

    layoutGraph() {
        DAGRE.layout(this.graph);
        const graphData: any[] = [];

        this.graph.nodes().forEach(el => {
            const node = this.graph.node(el) || null;

            if (node) {
                graphData.push({
                    id: node.id,
                    data: { label: this.getNodeLabel(node.label) },
                    position: { x: node.x, y: node.y },
                    connectable: false,
                });
            }
        });

        this.graph.edges().forEach(el => {
            if (el.v && el.w) {
                graphData.push({
                    id: el.v + SPLITTER + el.w,
                    source: el.v,
                    target: el.w,
                });
            }
        });

        this.setState({ graphData });
    }

    getNodeLabel(text) {
        return <div className={style.node_label}>{text}</div>;
    }

    onNodeClick(event, activeNode) {
        this.setState({ activeNode });
    }

    getElementId(fullId) {
        const allIds = fullId?.split(SPLITTER) || [];

        return allIds[allIds.length - 1];
    }

    openNotifierJSONModal(notifierJSONModalOpen) {
        this.setState({ notifierJSONModalOpen });
    }

    getSelectOptions() {
        return this.state.notifiers?.map(n => {
            return { value: n.name };
        });
    }

    setCurrentNotifier(notificator_name) {
        const currentNotifier = this.state.notifiers.find(el => el.name === notificator_name) || {} as IChat;
        const branches = this.getBranches(currentNotifier) ?? [];
        this.setState({ currentNotifier, branches });
    }

    setSelectedBranch(selectedBranch) {
        selectedBranch && this.getGraphData(selectedBranch);
    }

    getBranches(notifier): any {
        const introItem = this.getIntroNode(notifier);

        return introItem?.schema?.options?.map(el => {
            return { value: el.text };
        });
    }

    getBranchNode(branchName) {
        const introNode = this.getIntroNode() ?? {};
        const options = introNode?.schema?.options ?? [];

        return options.find(el => el.text === branchName);
    }

    getIntroNode(currentNotifier?: any): any {
        const notifier = currentNotifier ?? this.state.currentNotifier;
        const allItems = notifier?.meta?.chat_script?.items ?? [];

        return allItems?.find(el => el.id === INTRO_ID) ?? {};
    }

    getNodeById(id: string): any {
        const allNodes = this.state.currentNotifier?.meta?.chat_script?.items ?? [];

        return allNodes.find(el => el.id === id);
    }

    render() {
        const { error, isNotifiersLoading, graphData, currentNotifier, notifierJSONModalOpen, activeNode } = this.state;
        const isJSONDisabled = !Object.keys(currentNotifier)?.length;
        const modalTitle = 'Notifier ' + currentNotifier.name;

        return <>
            {isNotifiersLoading
                ? <Spin/>
                : error
                    ? <SimpleError error={error}/>
                    : <div className={style.main}>
                        <div className={style.controlBar}>
                            <Select options={this.getSelectOptions()}
                                    placeholder={'Нотификатор'}
                                    className={style.select}
                                    onSelect={this.setCurrentNotifier.bind(this)}/>

                            <Select options={this.state.branches}
                                    placeholder={'Ветка'}
                                    disabled={!this.state.branches?.length}
                                    className={style.select}
                                    onSelect={this.setSelectedBranch.bind(this)}/>

                            <Button basic={true}
                                    disabled={isJSONDisabled}
                                    className={style.json_btn}
                                    onClick={this.openNotifierJSONModal.bind(this, true)}>
                                JSON
                            </Button>

                            <ChatsTreeInfoBlock activeNode={activeNode}/>
                        </div>

                        <div className={style.main_block}>
                            <div className={style.nodes_wrapper}>
                                <ReactFlow elements={graphData}
                                           minZoom={MIN_ZOOM}
                                           defaultZoom={DEFAULT_ZOOM}
                                           onElementClick={this.onNodeClick.bind(this)}>
                                    <Background variant={BackgroundVariant.Dots}/>
                                    <Controls/>
                                </ReactFlow>
                            </div>
                        </div>
                    </div>
            }

            {notifierJSONModalOpen
                ? <Window title={modalTitle} onClose={this.openNotifierJSONModal.bind(this, false)}>
                    <ViewerJSON JSONData={currentNotifier}/>
                </Window>
                : null
            }
        </>;
    }
}
