import React from 'react';

import { ClassificationType } from '../../../../types';
import { CategorizationCategory } from '../../../models/tag';
import { Button } from '../../../ui/Button';
import { Confirm } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { IPlaneTreeItem, VirtualTreeList } from '../../../ui/VirtualTreeList';
import { Request2, specialQueryParamsKeys } from '../../../utils/request';
import { ITreeOption } from '../../Chats/ClassificationDialog/TreeSuggest';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { ClassificationTreeListItem } from './ClassificationTreeListItem';
import * as styleItem from './ClassificationTreeListItem/index.css';
import { EditModal } from './EditModal';
import * as style from './index.css';
import { REQUESTS, TREE_REQUESTS } from './request';

interface IClassificationTreeState {
    filterValue: string;
    error: Error | null;
    errorEditing: Error | null;
    isLoading: boolean;
    tree: any;
    node_scheme: any;
    editModalIsOpen: boolean;
    item: IPlaneTreeItem<CategorizationCategory> | null;
    isWorking: boolean;
    parent_id: string | null;
    confirmOpen: boolean;
    remove_id: string | null;
}

interface IClassificationTreeProps {
    readOnly?: boolean;
    type?: ClassificationType;
    onSelect?: (value: ITreeOption) => void;
}

export enum TREE_ACTIONS {
    add = 'add',
    modify = 'modify',
    remove_node = 'remove_node'
}

export default class ClassificationTree extends React.Component<IClassificationTreeProps, IClassificationTreeState> {
    request = new Request2({ requestConfigs: TREE_REQUESTS });
    state: IClassificationTreeState = {
        filterValue: '',
        isLoading: false,
        error: null,
        errorEditing: null,
        tree: {},
        node_scheme: {},
        editModalIsOpen: false,
        item: null,
        isWorking: false,
        parent_id: null,
        confirmOpen: false,
        remove_id: null,
    };

    constructor(props: {}) {
        super(props);
    }

    componentDidMount(): void {
        this.getData({ isLoading: true });
    }

    getData(options: { isLoading: boolean }) {
        const { isLoading = false } = options;
        this.setState({
            isLoading,
            error: null,
        }, () => {
            this.request.exec(REQUESTS.GET_TREE)
                .then(response => {
                    this.setState({
                        isLoading: false,
                        //node_scheme: /*response?.node_scheme || {}*/schema || {},
                        node_scheme: response?.node_scheme || {},
                        tree: response.tree,
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    onFilterChange(filterValue: string) {
        this.setState({ filterValue });
    }

    makePlain(elements): IPlaneTreeItem<CategorizationCategory>[] {
        const items = elements ?? [];

        return items
            .sort((a: CategorizationCategory, b: CategorizationCategory) => {
                return a.meta.order - b.meta.order;
            })
            .filter(el => {
                const isEnabled = el?.meta?.enabled;
                const isCorrectOrigin = el?.meta?.origins?.some(el => el.includes(this.props.type));

                return !this.props.readOnly || (isEnabled && isCorrectOrigin);
            })
            .map(_i => {
                return {
                    data: _i,
                    meta: {
                        id: _i.id,
                        children: _i.hasOwnProperty('children') ? this.makePlain(_i.children) : null,
                        className: `${_i.meta.enabled ? styleItem.enabled : styleItem.disabled} `
                            + `${_i.children?.length ? styleItem.has_children : styleItem.has_not_children}`,
                        active: true,
                    },
                };
            });

    }

    treeBuilder(): IPlaneTreeItem<CategorizationCategory>[] {
        return this.makePlain(this.state.tree?.children) || [];
    }

    filterTree() {
        if (this.state.filterValue) {
            return (treeItem: IPlaneTreeItem<CategorizationCategory>) => {
                return treeItem.data.meta.label?.toLowerCase().includes(this.state.filterValue?.toLowerCase())
                    || treeItem.data.meta?.keywords
                        ?.some(el => el.toLowerCase().includes(this.state.filterValue?.toLowerCase()));
            };
        }

        return null;

    }

    showModal(item, parent_id) {
        this.setState({
            editModalIsOpen: true,
            item,
            parent_id,
        });
    }

    showModalRemove(remove_id) {
        this.setState({
            confirmOpen: true,
            remove_id,
        });
    }

    closeConfirm() {
        this.setState({
            confirmOpen: false,
            remove_id: null,
        });
    }

    closeModal() {
        this.setState({
            editModalIsOpen: false,
            item: null,
        });
    }

    update(data) {
        this.setState({
            isWorking: true,
        }, () => {
            this.request.exec(REQUESTS.EDIT_TREE, {
                body: {
                    id: this.state.item?.meta.id,
                    parent_id: this.state.parent_id,
                    action: this.state.item?.meta.id ? TREE_ACTIONS.modify : TREE_ACTIONS.add,
                    ...data,
                },
                specialQueryParams: {
                    parent_id: {
                        key: specialQueryParamsKeys.FORCE,
                        value: this.state.parent_id,
                    },
                },
            })
                .then(() => {
                    this.setState({
                        errorEditing: null,
                        isWorking: false,
                    });
                    this.closeModal();
                    this.getData({ isLoading: false });
                })
                .catch(errorEditing => {
                    this.setState({
                        errorEditing,
                        isWorking: false,
                    });
                });
        });
    }

    render() {
        const {
            filterValue, editModalIsOpen, node_scheme, item,
            isWorking, errorEditing, confirmOpen, remove_id, isLoading, error,
        } = this.state;

        return <div className={`${style.component}`}>
            <div className={style.controls}>
                <Input onChange={this.onFilterChange.bind(this)}
                       value={filterValue}
                       className={style.filter_input}
                       placeholder={'Фильтр'}/>

                {!this.props.readOnly
                    ? <Button onClick={this.showModal.bind(this, null, null)}>Добавить новый узел</Button>
                    : null
                }
            </div>
            <div>
                {
                    editModalIsOpen
                        ? <EditModal schema={node_scheme}
                                     initialData={item}
                                     isWorking={isWorking}
                                     update={this.update.bind(this)}
                                     error={errorEditing}
                                     onClose={this.closeModal.bind(this)}/>
                        : null
                }
                {
                    confirmOpen
                        ? <Confirm error={errorEditing}
                                   onClose={this.closeConfirm.bind(this)}
                                   isWorking={isWorking}
                                   accept={() => {
                                       this.setState({ isWorking: true }, () => {
                                           this.request.exec(REQUESTS.EDIT_TREE, {
                                               body: {
                                                   action: TREE_ACTIONS.remove_node,
                                                   id: remove_id,
                                               },
                                           })
                                               .then(() => {
                                                   this.setState({
                                                       isWorking: false,
                                                       errorEditing: null,
                                                   });
                                                   this.closeConfirm();
                                                   this.getData({ isLoading: false });
                                               })
                                               .catch(errorEditing => {
                                                   this.setState({
                                                       isWorking: false,
                                                       errorEditing,
                                                   });
                                               });
                                       });
                                   }}
                                   question={'Удалить?'}/>
                        : null
                }
                {
                    isLoading
                        ? <Spin/>
                        : error
                            ? <SimpleError error={error}/>
                            : <div>
                                <VirtualTreeList treeBuilder={this.treeBuilder.bind(this)}
                                                 treeFilter={this.filterTree.bind(this)}
                                                 isFilterValueExist={!!filterValue}
                                                 treeListItem={[ClassificationTreeListItem, {
                                                     showModal: this.showModal.bind(this),
                                                     showModalRemove: this.showModalRemove.bind(this),
                                                     readOnly: this.props.readOnly,
                                                     onSelect: this.props.onSelect?.bind(this),
                                                 }]}/>
                            </div>
                }
            </div>
        </div>;
    }
}
