import moment from 'moment';
import React from 'react';

import { EMPTY_DATA, ONE_SECOND } from '../../../../../constants';
import { IRequestQueueItem, useRequestQueueHandler } from '../../../../../hooks/useRequestQueueHandler';
import { Button } from '../../../../../ui/Button';
import { Collapse } from '../../../../../ui/Collapse';
import DatePicker from '../../../../../ui/DatePicker';
import { IsChecked } from '../../../../../ui/IsChecked';
import { Link } from '../../../../../ui/Link';
import styleTable from '../../../../../ui/Table/index.css';
import { Request2 } from '../../../../../utils/request';
import { deepCopy } from '../../../../../utils/utils';
import { buildOptionsForIDMRole } from '../../../../Clients/UserRolesView/buildOptionsForIDMRole';
import { FormatDateAndDeadline } from '../../../../Clients/UserRolesView/FormatDateAndDeadline';
import { getEnvironment } from '../../../../Clients/UserRolesView/getEnvironment';
import { CORE_DRIVE_LENS_REQUESTS as requestConfigs, REQUESTS } from '../../../../Clients/UserRolesView/request';
import { IRole } from '../../../../Clients/UserRolesView/types';
import { ProgressBar } from '../../../../ProgressBar';
import { SimpleError } from '../../../../SimpleError';
import { IAddManageRolesModalProps } from '../index';
import style from './index.css';

export const AddManageRolesModal = (props: IAddManageRolesModalProps) => {
    const { onClose, selectedRoles, getUsersData, userNamesToCopy } = props;
    const request = new Request2({ requestConfigs });
    const [requestOptions, setRequestOptions] = React.useState<IRequestQueueItem[]>([]);
    const [selectedRolesState, setSelectedRolesState] = React.useState<IRole[]>(selectedRoles);
    const [selectedRoleIndex, setSelectedRoleIndex] = React.useState<number>(1);
    const [makeSimulateRequest, setMakeSimulateRequest] = React.useState<boolean>(false);
    const [idmRolesErrors, setIdmRolesErrors] = React.useState({});
    const [isLoading, setIsLoading] = React.useState<boolean>(false);

    const [, response, , makeResponse, progressBarObj] = useRequestQueueHandler(request, requestOptions);

    React.useEffect(() => {
        const idmRolesArray = selectedRolesState.filter(role => role?.role_info?.role_is_idm === '1');
        const requests = idmRolesArray.map(role => {
            setIsLoading(true);

            return request.exec(REQUESTS.SINGLE_REQUEST_IDM_ROLE, {
                body: {
                    deprive_at: moment(+role.deadline * ONE_SECOND).format('YYYY-MM-DD'),
                    path: `/${role.role_id}/`,
                    user: userNamesToCopy[0].username,
                    user_type: 'user',
                    system: getEnvironment(),
                    simulate: true,
                    comment: role.role_id,
                },
            });
        });

        Promise.allSettled(requests).then(responses => {
            const errors = {};
            responses.map((response: any, index) => {
                checkSimulatedResponse(response, idmRolesArray, errors, index);
            });
            setIsLoading(false);
            setIdmRolesErrors(errors);
        });

        return () => {
            request.abort();
        };
    }, []);

    React.useEffect(() => {
        if (response && !progressBarObj.errors.length) {
            onClose();
            getUsersData();
        } else if (response) {
            getUsersData();
        }
    }, [response]);

    const checkSimulatedResponse = (response, idmRolesArray, errors, index) => {
        if(response.status === 'rejected') {
            let error = response.reason;
            let showSimpleError = true;

            if(response.reason?.data?.message === 'Invalid data sent') {
                error = response.reason.data.errors?.deprive_at?.[0];
                showSimpleError = false;
            } else if (response.reason?.data?.message) {
                error = response.reason.data.message;
                showSimpleError = false;
            }

            errors[response.reason.request.body.comment] = {
                error,
                showSimpleError,
            };
        } else if(response.status === 'fulfilled'){
            setSelectedRolesState(prev => {
                prev.map(item => {
                    if(item.role_id === idmRolesArray[index]?.role_id) {
                        item.approvers = response.value.approvers[0];
                    }
                });

                return prev;
            });
        }
    };

    const simulateIDMRole = (role) => {
        setIsLoading(true);
        request.exec(REQUESTS.SINGLE_REQUEST_IDM_ROLE, {
            body: {
                deprive_at: moment(+role.deadline * ONE_SECOND).format('YYYY-MM-DD'),
                path: `/${role.role_id}/`,
                user: userNamesToCopy[0].username,
                user_type: 'user',
                system: getEnvironment(),
                simulate: true,
            },
        }).then((res) => {
            setIdmRolesErrors(prev => {
                delete prev[role.role_id];

                return prev;
            });

            setSelectedRolesState(prev => {
                prev.map(item => {
                    if(item.role_id === role.role_id) {
                        item.approvers = res.approvers[0];
                    }
                });

                return prev;
            });
        }).catch(err => {
            let errorData = err;
            let showSimpleError = true;

            if (err?.data?.message === 'Invalid data sent') {
                errorData = err.data.errors.deprive_at[0];
                showSimpleError = false;
            } else if(err?.data?.message) {
                errorData = err.data.message;
                showSimpleError = false;
            }

            setIdmRolesErrors(prev => {
                prev[role.role_id] = {};
                prev[role.role_id].error = errorData;
                prev[role.role_id].showSimpleError = showSimpleError;

                return prev;
            });
        }).finally(() => {
            setIsLoading(false);
            setMakeSimulateRequest(false);
        });
    };

    const setDeadlineIDMRole = (index, deadline) => {
        const selectedRolesCopy = deepCopy(selectedRolesState);
        selectedRolesCopy[index].deadline = `${+deadline / ONE_SECOND}`;
        setSelectedRolesState(selectedRolesCopy);
        setSelectedRoleIndex(index);
        setMakeSimulateRequest(true);
    };

    const copyRoles = () => {
        setRequestOptions([]);
        const requestOptions: IRequestQueueItem[] = [];
        userNamesToCopy.map(user => {
            selectedRolesState.map(role => {
                if (role?.role_info?.role_is_idm === '1') {
                    requestOptions.push({
                        requestName: REQUESTS.BATCH_REQUEST_IDM_ROLE,
                        requestOptions: buildOptionsForIDMRole({
                            deadline: +role.deadline,
                            selectedRoles: [role.role_id],
                            user: user.username,
                            system: getEnvironment() || '',
                            comment: '',
                        }),
                    },
                    );
                } else {
                    requestOptions.push(
                        {
                            requestName: REQUESTS.ADD_USER_ROLE,
                            requestOptions: {
                                queryParams: {
                                    user_id: user.user_id,
                                },
                                body: {
                                    role_id: role.role_id.toString(),
                                    deadline: Math.floor(+role.deadline),
                                    active: role.active,
                                    forced: role?.forced,
                                    user_id: user.user_id,
                                },
                            },
                        },
                    );
                }
            });
        });

        setRequestOptions(requestOptions);
        makeResponse();
    };

    React.useEffect(() => {
        if (makeSimulateRequest) {
            simulateIDMRole(selectedRolesState[selectedRoleIndex]);
        }
    }, [selectedRolesState, selectedRoleIndex]);

    return (
        <>
            <Button onClick={copyRoles}
                    disabled={!!Object.keys(idmRolesErrors).length}
                    isLoading={isLoading}>
                Скопировать роли
            </Button>
            <ProgressBar allLength={progressBarObj.allLength}
                         successLength={progressBarObj.successLength}
                         errors={progressBarObj.errors}/>
            {
                selectedRolesState.map(item => {
                    if(idmRolesErrors?.[item.role_id]?.error && idmRolesErrors?.[item.role_id]?.showSimpleError) {
                        return <Collapse key={item.role_id}
                                         collapsed={true}
                                         title={<span className={style.idmRoleError}> Error: {item.role_id}</span>}>
                            <SimpleError error={idmRolesErrors?.[item.role_id].error}/>
                        </Collapse>;
                    }

                    return null;

                })
            }
            <table className={styleTable.table}>
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Дедлайн</th>
                        <th>Active</th>
                        <th>Forced</th>
                        <th>Id</th>
                        <th>Описание</th>
                        <th>Ошибки</th>
                        <th>Подтверждающие</th>
                    </tr>
                </thead>
                <tbody>
                    {
                        selectedRolesState.map((item, index) => {
                            const isActive = item.active === '1';
                            const isRoleIDM = item?.role_info?.role_is_idm === '1';

                            return <tr key={item.role_id} className={styleTable.noTrHover}>
                                <td>{index + 1}</td>
                                <td className={style.deadlineBlock}>
                                    {isRoleIDM
                                        ? <DatePicker placeholder={'Дедлайн'}
                                                      onlyDate
                                                      {...useInput(
                                                          +item.deadline * ONE_SECOND,
                                                          setDeadlineIDMRole.bind(null, index))}/>
                                        : <FormatDateAndDeadline deadline={+item.deadline}/>
                                    }
                                </td>
                                <td>
                                    <IsChecked checked={isActive}/>
                                </td>
                                <td>
                                    <IsChecked checked={item.forced ?? false}/>
                                </td>
                                <td>
                                    <span>{item.role_id}</span>
                                </td>
                                <td>{item?.role_info?.role_description || EMPTY_DATA}</td>
                                <td>
                                    {
                                        idmRolesErrors?.[item.role_id]?.error
                                        && !idmRolesErrors?.[item.role_id]?.showSimpleError
                                            ? <span className={style.idmRoleError}>
                                                {idmRolesErrors[item.role_id].error}
                                            </span>
                                            : null
                                    }
                                </td>
                                <td>
                                    {
                                        item?.approvers
                                            ? item.approvers.map(user => {
                                                return <Link key={user.username}
                                                             className={style.approveLink}
                                                             href={`https://staff.yandex-team.ru/${user.username}`}
                                                             target="_blank">
                                                    {user.full_name}
                                                </Link>;
                                            })
                                            : null
                                    }
                                </td>
                            </tr>;
                        })
                    }
                </tbody>
            </table>
        </>
    );
};

const useInput = (initialValue: number, setDeadlineIDMRole) => {
    const [value, setValue] = React.useState(initialValue);

    const onChange = (inputValue) => {
        setValue(inputValue);
        setDeadlineIDMRole(inputValue);
    };

    React.useEffect(() => {
        setValue(initialValue);
    }, [initialValue]);

    return {
        value,
        onChange,
    };
};
