import React from 'react';
import ReactTooltip from 'react-tooltip';

import { BasicUserInfo, UserInfoHandler } from '../../../../models/user';
import { Button, ButtonTypes } from '../../../../ui/Button';
import Checkbox from '../../../../ui/Checkbox';
import styleTable from '../../../../ui/Table/index.css';
import { deepCopy } from '../../../../utils/utils';
import { IRole } from '../../../Clients/UserRolesView/types';
import Spin from '../../../Spin';
import { AddManageRolesWindow } from '../AddManageRolesWindow';
import { RoleStatus } from '../RoleStatus';
import style from './index.css';

const selectButtonTooltipId = 'selectButtonTooltipId';

interface ITableRolesProps {
    roles: Record<string, IRole[]>;
    users: Record<string, BasicUserInfo>;
    selectedUser: BasicUserInfo;
    getUsersData: () => void;
    isLoading: boolean;
}

export interface IUserNamesToCopy{
    username: string;
    user_id: string;
}

export const TableRoles = (props: ITableRolesProps) => {
    const { roles, users, selectedUser, getUsersData, isLoading } = props;
    const [tableDataState, setTableDataState] = React.useState<[string, any][]>([]);
    const [usersIdArray, setUsersIdArray] = React.useState<string[]>([]);
    const [selectedRoles, setSelectedRoles] = React.useState<IRole[]>([]);
    const [hideSimilar, setHideSimilar] = React.useState<boolean>(true);
    const [copyRolesWindow, setCopyRolesWindow] = React.useState<boolean>(false);
    const [checkboxesHashState, setCheckboxesHashState] = React.useState({});

    const formattedTableData = () => {
        const temp = {};
        roles && Object.entries(roles)?.forEach((_c: [any, any]) => {
            _c[1]?.forEach(role => {
                if (!temp.hasOwnProperty(role.role_id)) {
                    temp[role.role_id] = {
                        role,
                        users: {
                            [_c[0]]: role,
                        },
                    };
                } else {
                    temp[role.role_id].users[_c[0]] = role;
                }
            });
        });

        return Object.entries(temp);
    };

    React.useEffect(() => {
        if(roles) {
            const usersIdArray = Object.keys(users);
            setTableDataState(formattedTableData());
            setUsersIdArray(usersIdArray);
            const checkboxesHash = {};

            formattedTableData().map(arr => {
                checkboxesHash[arr[0]] = {};

                const array: any = arr;

                for(let i = 0; i < usersIdArray.length; i++) {
                    if(array[1].users?.[usersIdArray[i]]?.active) {
                        checkboxesHash[arr[0]][usersIdArray[i]] = {};
                        checkboxesHash[arr[0]][usersIdArray[i]].checked = false;
                    }
                }
            });

            setCheckboxesHashState(checkboxesHash);
        }
    }, [roles]);

    React.useEffect(() => {
        getSelectedRoles(checkboxesHashState, tableDataState);
        ReactTooltip.rebuild();
    }, [selectedUser, hideSimilar]);

    const enableRoleDuplication = Object.entries(users).length > 1;
    const selectedUserId = UserInfoHandler.getId.call(selectedUser);
    const selectedUserUsername = UserInfoHandler.getUsername.call(selectedUser);

    const getSelectedRoles = (checkboxesHash, tableData) => {
        const selectedRoles: any = [];

        tableData.filter(role => {
            return !hideSimilar
                ? true
                : filterRoles(role);
        }).forEach((arr) => {
            if(checkboxesHash[arr[0]]?.[selectedUserId]?.checked) {
                selectedRoles.push(arr[1].users[selectedUserId]);
            }
        });

        setSelectedRoles(selectedRoles);
    };

    const onChangeCheckbox = (checked: boolean, roleId: string, userId: string) => {
        const checkboxesHashCopy = deepCopy(checkboxesHashState);
        checkboxesHashCopy[roleId][userId].checked = checked;
        setCheckboxesHashState(checkboxesHashCopy);
        getSelectedRoles(checkboxesHashCopy, tableDataState);
    };

    const applyAllCheckboxes = (checked, userId) => {
        const checkboxesHashCopy = deepCopy(checkboxesHashState);
        tableDataState.filter(role => {
            return !hideSimilar
                ? true
                : filterRoles(role);
        }).forEach(arr => {
            if(checkboxesHashCopy[arr[0]]?.[userId]) {
                checkboxesHashCopy[arr[0]][userId].checked = checked;
            }
        });
        setCheckboxesHashState(checkboxesHashCopy);

        getSelectedRoles(checkboxesHashCopy, tableDataState);
    };

    const filterRoles = (role) => {
        const array: any = [];

        for(let i = 0; i < usersIdArray.length; i++) {
            array.push(role[1].users?.[usersIdArray[i]]?.active);
        }

        return !array.every(el => el === array[0]);
    };

    const userNamesToCopy = React.useMemo(() => {
        const selectedUsername = UserInfoHandler.getUsername.call(selectedUser);

        return usersIdArray.reduce((acc: IUserNamesToCopy[], userId) => {
            const username = UserInfoHandler.getUsername.call(users?.[userId]);
            if (username !== selectedUsername) {
                acc.push({ username: username, user_id: userId });
            }

            return acc;
        }, []);
    }, [usersIdArray, selectedUser]);

    return <div className={style.roles}>
        {
            enableRoleDuplication
                ? <div className={style.controls}>
                    <div>
                        <span className={style.selectedRolesAmount}>{`Выбрано ${selectedRoles.length} ролей`}</span>
                        <Button className={style.copyRolesButton}
                                disabled={!selectedUserId || selectedRoles.length === 0}
                                onClick={() => setCopyRolesWindow(true)}>
                            {`Скопировать роли ${selectedUserUsername || ''}`}
                        </Button>
                    </div>
                    <div className={style.similarCheckboxContainer}>
                        Скрыть одинаковые
                        <Checkbox checked={hideSimilar}
                                  onChange={(checked) => setHideSimilar(checked)}
                                  className={style.similarCheckbox}/>
                    </div>
                </div>
                : null
        }
        <ReactTooltip id={selectButtonTooltipId} type={'info'} html/>
        <table className={`${styleTable.table}`}>
            <thead>
                <tr>
                    <th>#</th>
                    <th>роль</th>
                    {
                        usersIdArray.map((userId, index) => {
                            const username = UserInfoHandler.getUsername.call(users?.[userId]);

                            return <th key={index} className={`${enableRoleDuplication ? style.userName : ''}`}>
                                <div className={style.userNameBlock}>
                                    {
                                        enableRoleDuplication
                                            ? <div data-for={selectButtonTooltipId}
                                                   data-tip={`<span>Выбрать все</span>`}>
                                                <Button className={style.selectAllButton}
                                                        colorType={ButtonTypes.positive}
                                                        onClick={() => applyAllCheckboxes(true, userId)}
                                                        disabled={selectedUserId !== userId}>+</Button>
                                            </div>
                                            : null
                                    }
                                    {username}
                                    {
                                        enableRoleDuplication
                                            ? <div data-for={selectButtonTooltipId}
                                                   data-tip={`<span>Снять выбор</span>`}>
                                                <Button className={style.selectAllButton}
                                                        colorType={ButtonTypes.negative}
                                                        onClick={() => applyAllCheckboxes(false, userId)}
                                                        disabled={selectedUserId !== userId}>-</Button>
                                            </div>
                                            : null
                                    }
                                </div>
                            </th>;
                        })
                    }
                </tr>
            </thead>
            <tbody>
                { isLoading
                    ? <tr>
                        <td><Spin/></td>
                    </tr>
                    : tableDataState
                        .filter(role => {
                            return hideSimilar && enableRoleDuplication
                                ? filterRoles(role)
                                : true;
                        })
                        .map(([role_id, roleObj]: [string, { role: IRole; users: { active: boolean} }], index) => {
                            const isIDMRole = roleObj.role?.role_info?.role_is_idm === '1';

                            return <tr key={role_id}>
                                <td>{index + 1}</td>
                                <td className={style.roleInfo}>
                                    <div className={style.roleInfoBlock}>
                                        <div className={style.role_id}>{role_id}</div>
                                        <div className={style.role_description}>
                                            {roleObj.role?.role_info?.role_description}
                                        </div>
                                    </div>
                                    {
                                        isIDMRole
                                            ? <>
                                                <span>(IDM)</span>
                                                <span className={style.idmIcon}/>
                                            </>
                                            : null
                                    }
                                </td>
                                {
                                    usersIdArray.map((user_id, index) => {
                                        return <RoleStatus key={index}
                                                           user={roleObj.users?.[user_id]}
                                                           printCheckboxes={enableRoleDuplication}
                                                           onChangeCheckbox={onChangeCheckbox}
                                                           roleId={role_id}
                                                           userId={user_id}
                                                           selectedUser={selectedUser}
                                                           checkboxesHash={checkboxesHashState}/>;
                                    })
                                }
                            </tr>;
                        })
                }
            </tbody>
        </table>
        {
            copyRolesWindow
                ? <AddManageRolesWindow userNamesToCopy={userNamesToCopy}
                                        onClose={() => setCopyRolesWindow(false)}
                                        selectedUser={selectedUser}
                                        selectedRoles={selectedRoles}
                                        getUsersData={getUsersData}/>
                : null
        }
    </div>;
};
