import React from 'react';

import { Dict } from '../../../../types';
import { BasicUserInfo } from '../../../models/user';
import { GLOBAL_SEARCH_OBJECTS } from '../../../ui/IDSelect/constants';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { Request2 } from '../../../utils/request';
import { getEnvironment } from '../../Clients/UserRolesView/getEnvironment';
import { IRole } from '../../Clients/UserRolesView/types';
import { FormConstructor } from '../../FormConstructor';
import { controlType, ISchemaItem, SchemaItemVisual } from '../../FormConstructor/types';
import { requestConfigs, REQUESTS } from './request';
import { UsersView } from './UsersView';

const schema: Dict<ISchemaItem> = {
    'users': {
        'type': controlType.array_types,
        'display_name': 'Список пользователей',
        'order': 6,
        'array_type': {
            'type': controlType.string,
            'display_name': 'Пользователь',
            visual: SchemaItemVisual.ID_SELECT,
            object: GLOBAL_SEARCH_OBJECTS.users,
        },
        default: [],
    },
};

interface ICompareRolesState {
    users: string[];
    usersObj: Record<string, BasicUserInfo>;
    rolesObj: Record<string, IRole[]>;
    isLoading: boolean;
}

export class CompareRoles extends React.Component<any, ICompareRolesState> {
    state: ICompareRolesState = {
        users: [],
        usersObj: {},
        rolesObj: {},
        isLoading: false,
    };

    request = new Request2({ requestConfigs });

    onChange(value) {
        this.setState({
            users: value.users,
        });
    }

    componentDidUpdate(prevProps: Readonly<any>, prevState: Readonly<ICompareRolesState>, snapshot?: any) {
        if (!isObjectEqual(this.state.users, prevState.users)) {
            this.getUsersData();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getUsersData() {
        if (!this.state.users.length) {
            this.setState({
                usersObj: {},
            });
        } else {
            this.setState({
                isLoading: true,
            }, async () => {
                const users = await Promise.all(this.state.users.filter(el => el)
                    .map(user_id => this.request.exec(REQUESTS.GET_USER_DATA,
                        { queryParams: { user_id } })));

                const roles = await Promise.all(users.map(user => this.request.exec(REQUESTS.GET_ROLES,
                    { queryParams: { user_id: user.id } })));

                this.setState({
                    usersObj: users?.reduce((_p, _c) => {
                        if (!_p.hasOwnProperty(_c.id)) {
                            _p[_c.id] = _c;
                        }

                        return _p;
                    }, {}),
                    rolesObj: roles?.reduce((_p, _c) => {
                        const id = _c?.report?.[0]?.user_id;
                        if (id && !_p.hasOwnProperty(id)) {
                            _p[id] = _c.report.filter(role => {
                                const isRoleIDM = role.role_info.role_is_idm === '1';
                                if(isRoleIDM && getEnvironment() === undefined) {
                                    return false;
                                }

                                return true;

                            }) || [];
                        }

                        return _p;
                    }, {}),
                    isLoading: false,
                });
            });
        }
    }

    render() {
        return <div>
            <div>
                <FormConstructor schema={schema} onChange={this.onChange.bind(this)}/>
            </div>

            <div>
                <UsersView users={this.state.usersObj}
                           roles={this.state.rolesObj}
                           getUsersData={this.getUsersData.bind(this)}
                           isLoading={this.state.isLoading}/>
            </div>
        </div>;
    }
}
