import React, { useEffect, useState } from 'react';

import { Link } from '../../../../../ui/Link';
import { Request2 } from '../../../../../utils/request';
import { deepCopy } from '../../../../../utils/utils';
import { REQUESTS } from '../../../../Migrate/request';
import { SimpleError } from '../../../../SimpleError';
import { DM } from '../../request';
import { DM_ADMIN_ROLE_ID } from '../../types';
import style from './index.css';

interface ICreatingFlagProgressBarProps {
    flagActionData: Record<string, any> | null;
    env: 'prestable' | 'testing';
    isCreating: boolean | null;
    status: boolean | null;
    onStatusChange: (status: boolean | null) => void;
}

const CreatingFlagProgressBar = ({
    isCreating,
    flagActionData,
    env,
    status,
    onStatusChange,
}: ICreatingFlagProgressBarProps) => {
    const [flagCreatingStatus, setFlagCreatingStatus] = useState<boolean | null>(null);
    const [attachingFlagToRoleStatus, setAttachingFlagToRoleStatus] = useState<boolean | null>(null);
    const [error, setError] = useState<Error | null>(null);
    const request = new Request2({ requestConfigs: DM });

    useEffect(() => {
        if (isCreating === true) {
            createFlag();
        }
    }, [isCreating]);

    const createFlag = () => {
        setFlagCreatingStatus(null);
        setError(null);

        request.exec(REQUESTS.ADD_ACTION,
            {
                body: flagActionData,
                queryParams: { backend_cluster: env },
                removeExtraCgi: true,
            })
            .then(() => {
                attachFlagToRole();
                setFlagCreatingStatus(true);
            })
            .catch(error => {
                setFlagCreatingStatus(false);
                onStatusChange(false);
                setError(error);
            });
    };

    const attachFlagToRole = () => {
        setAttachingFlagToRoleStatus(null);
        setError(null);

        Promise.all([
            request.exec(REQUESTS.GET_ACTIONS, {
                queryParams: { backend_cluster: env },
                removeExtraCgi: true,
            }),
            request.exec(REQUESTS.GET_ROLES, {
                queryParams: { backend_cluster: env },
                removeExtraCgi: true,
            })])
            .then(response => {
                const [actionsResponse, rolesResponse] = response;
                const actions = actionsResponse?.report ?? [];
                const roles = rolesResponse?.report ?? [];

                const action = actions.find(action => action && action.action_id === flagActionData?.action_id) ?? null;
                const role = roles.find(role => role && role.role_id === DM_ADMIN_ROLE_ID) ?? null;

                if (action && role) {
                    const actionMeta = action.time_restrictions
                        && action.time_restrictions.restrictions
                        && action.time_restrictions.restrictions.length && {
                        time_restrictions: action.time_restrictions,
                    } || {};
                    const actionForAttaching = {
                        action,
                        action_id: action.action_id,
                        role_action_meta: actionMeta,
                        role_id: DM_ADMIN_ROLE_ID,
                    };
                    const newRole = deepCopy(role);

                    if (!newRole.actions) {
                        newRole.actions = [];
                    }

                    newRole.actions.push(actionForAttaching);

                    if (newRole.role_group !== undefined) {
                        newRole.role_group = +newRole.role_group;
                    }

                    if (newRole.parent_role_ids) {
                        delete newRole.parent_role_ids;
                    }

                    if (newRole.role_groupping_tags !== undefined) {
                        delete newRole.role_groupping_tags;
                    }

                    return request.exec(REQUESTS.ROLES_UPSERT, {
                        body: [newRole],
                        queryParams: { backend_cluster: env },
                        removeExtraCgi: true,
                    });
                }

                throw new Error('No action or role');

            })
            .then(() => {
                setAttachingFlagToRoleStatus(true);
                onStatusChange(true);
            })
            .catch(() => {
                setAttachingFlagToRoleStatus(false);
                onStatusChange(false);
                setError(error);
            });
    };

    return <div className={`${style.progress_bar} ${status === true
        ? style.success
        : status === false ? style.error : ''}`}>
        <span>{env}</span>
        <div className={`${style.progress_item} `
        + ` ${flagCreatingStatus === true ? style.success : flagCreatingStatus === false ? style.error : ''}`}>
            Создание флага
        </div>
        <div className={`${style.progress_item} `
        + ` ${attachingFlagToRoleStatus === true
            ? style.success
            : attachingFlagToRoleStatus === false ? style.error : ''}`}>
            Привязка флага к роли админа
        </div>
        {flagCreatingStatus === false || attachingFlagToRoleStatus === false
            ? <Link onClick={flagCreatingStatus === false ? createFlag : attachFlagToRole}>Повторить</Link>
            : null}
        {error ? <SimpleError error={error}/> : null}
    </div>;
};

export default CreatingFlagProgressBar;
