import React, { useEffect, useState } from 'react';

import { EMPTY_DATA } from '../../../../constants';
import { useRequestQueueHandler } from '../../../../hooks/useRequestQueueHandler';
import { Button, CancelButton } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import coreStyles from '../../../../ui/index.css';
import { IOptionInfo } from '../../../../ui/Select';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import { controlType } from '../../../FormConstructor/types';
import { DM, KEY_FLAG, REQUESTS } from '../request';
import { CREATE_FLAG_SCHEMA } from '../types';
import AttachingFlagToSTProgressBar from './AttachingFlagToSTProgressBar';
import CreatingFlagProgressBar from './CreatingFlagProgressBar';
import FinalMessageBar from './FinalMessageBar';

interface ICreateFlagModalProps {
    onClose: (success?: boolean) => void;
}

const CreateFlagModal = ({ onClose }: ICreateFlagModalProps) => {
    const [isCreating, setIsCreating] = useState<boolean | null>(null);
    const [flagFormData, setFlagFormData] = useState<Record<string, any>>({});
    const [flagData, setFlagData] = useState<Record<string, any> | null>(null);
    const [creatingPrestableStatus, setCreatingPrestableStatus] = useState<boolean | null>(null);
    const [creatingTestingStatus, setCreatingTestingStatus] = useState<boolean | null>(null);

    const [attachingToSTTicket, setAttachingToSTTicket] = useState<boolean | null>(null);
    const [attachingToSTError, setAttachingToSTError] = useState<Error | null>(null);

    const [stTasksOptions, setStTasksOptions] = useState<IOptionInfo[]>([]);

    const [showFinalMessage, setShowFinalMessage] = useState<boolean>(false);

    const request = new Request2({ requestConfigs: DM });
    const requestEdit = new Request2({ requestConfigs: DM });

    const requestOptions = React.useMemo(() => [
        {
            requestName: REQUESTS.GET_TASKS,
            requestOptions: {
                queryParams: {
                    query: 'Queue: DRIVEFRONT AND Components: "drivematics" AND (DRIVEFRONT.flag: empty() OR DRIVEFRONT.flag: "")',
                    perPage: 100,
                },
            },
        },
    ], []);

    const [
        isLoadingTasks, , ,
        makeResponse,
    ] = useRequestQueueHandler<Record<string, any>>(request, requestOptions, response => {
        const [stTasks] = response ?? {};

        const stTasksOptions = (stTasks ?? []).map(stTask => {
            const { id, key, summary } = stTask;

            return {
                value: id,
                text: summary || key || EMPTY_DATA,
                description: summary ? key : '',
            };
        });

        setStTasksOptions(stTasksOptions);
    });

    useEffect(() => {
        makeResponse();
    }, []);

    useEffect(() => {
        if (flagData) {
            setIsCreating(true);
        }
    }, [flagData]);

    useEffect(() => {
        if (creatingPrestableStatus !== null && creatingTestingStatus !== null) {
            if (creatingPrestableStatus && creatingTestingStatus) {
                if (flagFormData.task) {
                    attachFlagToSTTicket();
                } else {
                    setShowFinalMessage(true);
                }
            }
        }
    }, [creatingPrestableStatus, creatingTestingStatus]);

    const attachFlagToSTTicket = () => {
        setAttachingToSTError(null);

        requestEdit.exec(REQUESTS.EDIT_TASK,
            {
                queryParams: { id: flagFormData.task },
                body: { [KEY_FLAG]: flagFormData?.key || '' },
            })
            .finally(() => {
                setIsCreating(false);
            })
            .then(() => {
                setAttachingToSTTicket(true);
                setShowFinalMessage(true);
            })
            .catch(error => {
                setAttachingToSTError(error);
                setAttachingToSTTicket(false);
            });
    };

    const onCreate = () => {
        const flagData: Record<string, any> = {
            action_type: 'flags',
            deprecated: false,
            action_meta: {
                client_version_check_policy: 'Default',
                experiment: { max_shard: 65536, min_shard: 0 },
                user_tags_in_point: [],
                user_undefined_position_policy: 'AcceptUndefined',
            },
        };

        flagData.action_id = `flag_${flagFormData?.key || ''}`;
        flagData.enabled = flagFormData?.enabled || false;
        flagData.action_description = flagFormData?.action_description || '';
        flagData.action_meta.flags = [
            {
                key: flagFormData?.key || '',
                value: flagFormData?.value || '',
            },
        ];

        setFlagData(flagData);
    };

    const onChange = (formData: Record<string, any>) => {
        setFlagFormData(formData);
    };

    const onStatusChange = (env: 'prestable' | 'testing', status: boolean | null) => {
        if (env === 'prestable') {
            setCreatingPrestableStatus(status);
        } else if (env === 'testing') {
            setCreatingTestingStatus(status);
        }
    };

    const SCHEMA = Object.assign({}, CREATE_FLAG_SCHEMA, {
        task: {
            type: controlType.variants,
            display_name: 'Тикет',
            variants: stTasksOptions as IOptionInfo[] ?? [],
        },
    });

    const TICKET_KEY = stTasksOptions
        .find(stTasksOption => stTasksOption.value === flagFormData.task)?.description as string;

    return <Window onClose={onClose} title={'Создание флага'}>
        <FormConstructor schema={SCHEMA} onChange={onChange}/>
        <div className={coreStyles.button_container}>
            <CancelButton onClick={onClose}/>
            <Button isLoading={isCreating ?? false}
                    disabled={!flagFormData.key || isLoadingTasks}
                    onClick={onCreate}>
                Создать
            </Button>
        </div>
        {isCreating !== null
            ? <>
                <CreatingFlagProgressBar isCreating={isCreating}
                                         status={creatingPrestableStatus}
                                         onStatusChange={onStatusChange.bind(null, 'prestable')}
                                         flagActionData={flagData}
                                         env={'prestable'}/>
                <CreatingFlagProgressBar isCreating={isCreating}
                                         status={creatingTestingStatus}
                                         onStatusChange={onStatusChange.bind(null, 'testing')}
                                         flagActionData={flagData}
                                         env={'testing'}/>
                {flagFormData.task
                    ? <AttachingFlagToSTProgressBar status={attachingToSTTicket}
                                                    ticketKey={TICKET_KEY}
                                                    error={attachingToSTError}
                                                    attachFlagToSTTicket={attachFlagToSTTicket}/>
                    : null}
                {showFinalMessage
                    ? <FinalMessageBar creatingPrestableStatus={creatingPrestableStatus}
                                       creatingTestingStatus={creatingTestingStatus}
                                       actionName={`flag_${flagFormData?.key || ''}`}
                                       attachingToSTTicket={attachingToSTTicket}
                                       ticketKey={TICKET_KEY}
                                       onClose={onClose.bind(null, true)}/>
                    : null}
            </>
            : null}
    </Window>;
};

export default CreateFlagModal;
