import React, { useEffect, useState } from 'react';

import { EMPTY_DATA } from '../../../constants';
import { useRequestQueueHandler } from '../../../hooks/useRequestQueueHandler';
import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import FormatDate from '../../../ui/FormatDate';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import * as styleTable from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import Spin from '../../Spin';
import CreateFlagModal from './CreateFlagModal';
import styles from './index.css';
import { DM, KEY_FLAG, REQUESTS } from './request';

const DEFAULT_ROLE = 'v3_default_role';

interface IFlagSettings {
    _st: {
        versionSingle: any;
        key: string;
        summary: string;
    };
    _roles: string[];
    is_consists_in_default_role: boolean | undefined;
    enabled: boolean;
    action_id: string;
    value: any;
}

const DmFlags = () => {
    const request = new Request2({ requestConfigs: DM });
    const [filter, setFilter] = useState('');
    const [forAllRoles, setForAllRoles] = useState(false);
    const [showCreateFlagModal, setShowCreateFlagModal] = useState<boolean>(false);

    const requestOptions = React.useMemo(() => [
        {
            requestName: REQUESTS.GET_TASKS,
            requestOptions: {
                queryParams: {
                    query: 'Queue: DRIVEFRONT AND Components: "drivematics" AND DRIVEFRONT.flag: notEmpty()',
                },
            },
        },
        {
            requestName: REQUESTS.GET_ACTIONS,
            requestOptions: {
                queryParams: {},
            },
        },
        {
            requestName: REQUESTS.GET_ROLES,
            requestOptions: {
                queryParams: {},
            },
        },
    ], []);

    const [
        isLoading,
        response, ,
        makeResponse,
    ] = useRequestQueueHandler<Record<string, IFlagSettings>>(request, requestOptions
        , (_response) => {
            const [st_flags, actions, roles] = _response;

            const actionsMap = roles?.report?.reduce((p, c) => {
                c?.actions?.map(el => {
                    const compiledArray = [c?.role_id, ...c?.parent_role_ids];

                    if (p?.[el.action_id]) {
                        p[el.action_id].push(...compiledArray);
                    } else {
                        p[el.action_id] = compiledArray;
                    }
                });

                return p;
            }, {});

            const flagsMap = actions?.report?.reduce((p, c) => {
                if (c.action_type === 'flags') {
                    c?.action_meta?.flags.map(el => {
                        p[el.key] = {
                            value: el.value,
                            action_id: c.action_id,
                            action_description: c.action_description,
                            enabled: c.enabled,
                            is_consists_in_default_role: actionsMap[c.action_id]?.includes(DEFAULT_ROLE),
                            _st: null,
                            _roles: actionsMap[c.action_id],
                        };
                    });
                }

                return p;
            }, {});

            st_flags?.forEach(_st => {
                if (flagsMap?.[_st?.[KEY_FLAG]]) {
                    flagsMap[_st[KEY_FLAG]] = { ...flagsMap[_st[KEY_FLAG]], _st };
                }
            });

            return flagsMap;
        });

    useEffect(() => {
        makeResponse();
    }, []);

    const filterByFields = (flag_id, flag_setting, filter) => {
        return flag_id?.toUpperCase()?.includes(filter.toUpperCase())
            || flag_setting?._st?.versionSingle?.display?.includes(filter.toUpperCase())
            || flag_setting?._st?.key?.toUpperCase()?.includes(filter.toUpperCase())
            || flag_setting?._st?.summary?.toUpperCase()?.includes(filter.toUpperCase())
            || flag_setting?._roles?.some(role => role?.toUpperCase()?.includes(filter.toUpperCase()));
    };

    const openCreateFlagModal = () => {
        setShowCreateFlagModal(true);
    };

    const closeCreateFlagModal = (success?: boolean) => {
        setShowCreateFlagModal(false);
        if (success) {
            makeResponse();
        }
    };

    return <div>
        <div className={styles.controls}>
            <Input value={filter} onChange={setFilter.bind(null)} placeholder={'Фильтр'} className={styles.input}/>
            <Button onClick={openCreateFlagModal}
                    basic>
                Добавить новый флаг
            </Button>
            <div className={styles.checkbox}>Выкатили на всех: <Checkbox checked={forAllRoles}
                                                                         onChange={setForAllRoles}/>
            </div>
        </div>
        {
            isLoading ? <Spin/> :

                <table className={styleTable.table}>
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Flag</th>
                            <th>version</th>
                            <th>ST</th>
                            <th>actions/roles</th>
                        </tr>
                    </thead>
                    <tbody>
                        {
                            Object.entries(response || {})
                                ?.filter(([flag_id, flag_setting]) => {
                                    if (filter && !forAllRoles) {
                                        return filter && filterByFields(flag_id, flag_setting, filter);
                                    }

                                    if (forAllRoles && filter) {
                                        return filterByFields(flag_id, flag_setting, filter)
                                            && flag_setting?.is_consists_in_default_role;
                                    }

                                    if (forAllRoles && !filter) {
                                        return flag_setting?.is_consists_in_default_role;
                                    }

                                    if (!forAllRoles && !filter) {
                                        return true;
                                    }
                                })
                                ?.sort((a: any, b: any) => {
                                    return a[1]?.is_consists_in_default_role - b[1]?.is_consists_in_default_role;
                                })
                                ?.map((flag, index) =>
                                    <FlagItem flag={flag} key={index} index={index}/>)
                        }
                    </tbody>
                </table>
        }
        {showCreateFlagModal
            ? <CreateFlagModal onClose={closeCreateFlagModal}/>
            : null}
    </div>;
};

const FlagItem = (props) => {
    const { flag, index } = props || {};
    const [flag_id, flag_setting] = flag || [];
    const st = flag_setting?._st || {};
    const is_consists_in_default_role = flag_setting.is_consists_in_default_role;

    return <tr>
        <td>{index + 1}</td>
        <td title={`value: ${flag_setting.value}`} className={is_consists_in_default_role ? styles.done : ''}>
            {flag_id}
            <div className={styles.flag_description}>{flag_setting.action_description}</div>
        </td>
        <td>{st.versionSingle?.display}</td>
        <td>
            {st.key &&
            <>
                <div>
                    <Link href={`https://st.yandex-team.ru/${st.key}`}
                          target={'_blank'}>{st.key}</Link>
                </div>
                <div>
                    <div className={styles.summary}>{st.summary}</div>
                    <div className={`${styles.status} ${st.status.key === 'open' ? styles.st_open : ''}`}>
                        [{st.status?.display}]
                    </div>
                    <FormatDate className={styles.status} value={st.statusStartTime}/>
                </div>
            </>}
        </td>
        <td>
            <div className={`${styles.state} ${flag_setting.enabled ? styles.enabled : ''}`}/>
            {flag_setting.action_id
                ? <Link href={`#/settings/actions/${flag_setting.action_id}`} target={'_blank'}>
                    {flag_setting.action_id}
                </Link>
                : EMPTY_DATA}
            <div>{
                flag_setting?._roles?.join(', ')
            }</div>
        </td>
    </tr>;
};

export default DmFlags;
