import React from 'react';

import { Dict } from '../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { RoleItem, RoleProposition } from '../../../models/role';
import { Button, ButtonTypes } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { Confirm, Window } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import Select from '../../../ui/Select';
import { Tabs } from '../../../ui/Tabs';
import { IWeekCalendarPropsItem, WeekCalendar } from '../../../ui/WeekCalendar';
import { Request2 } from '../../../utils/request';
import { Copy } from '../../Copy';
import { FormConstructor } from '../../FormConstructor';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { ProposeCommentModal } from '../ProposeCommentModal';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import { IPropositions } from '../Roles/component';
import * as style from './index.css';
import { shemaRole } from './shema';

interface IEditRoleProps {
    onCLose: () => void;
    update: () => void;
    role_id: string;
    proposition_id: string;
    // report: RoleItem[]
    // propositions: IPropositions
}

enum RoleSettingsTabs {
    ACTIONS = 'actions',
    ROLES = 'roles',
}

interface IEditRoleState extends Dict<any> {
    error: any;
    selectedTab: RoleSettingsTabs;
    actions: IWeekCalendarPropsItem[];
    roles: IWeekCalendarPropsItem[];
    editRole: RoleItem & RoleProposition;
    _editedRole: any;

    allRoles: any[];
    rolesIsLoading: boolean;

    allActions: any[];
    actionsIsLoading: boolean;
    showProposeComment: boolean;
    isWorking: boolean;
    proposeError: any;
    propositions: RoleProposition[];

    confirmIsOpen: boolean;
    errorConfirm: any;
    confirmIsLoading: boolean;
    confirmQuestion: string;
    confirmAction: any;

    currentPropositions: IPropositions[];
    currentRole: RoleItem;
    roleIsLoading: boolean;
}

const actions = ({ actions, roles }) => [
    { name: `Экшены: ${actions}`, link: RoleSettingsTabs.ACTIONS },
    { name: `Роли: ${roles}`, link: RoleSettingsTabs.ROLES },
];

export class EditRole extends React.Component<IEditRoleProps, IEditRoleState> {
    state: IEditRoleState = {
        error: null,
        selectedTab: RoleSettingsTabs.ACTIONS,
        actions: [],
        roles: [],
        editRole: {} as RoleItem & RoleProposition,
        _editedRole: {},

        allRoles: [],
        rolesIsLoading: false,

        allActions: [],
        actionsIsLoading: false,

        showProposeComment: false,
        isWorking: false,
        proposeError: null,
        propositions: [],

        confirmIsOpen: false,
        errorConfirm: null,
        confirmIsLoading: false,
        confirmQuestion: '',
        confirmAction: null,

        currentPropositions: [] as IPropositions[],
        currentRole: {} as RoleItem,
        roleIsLoading: false,
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    onChange(data) {
        this.setState({ _editedRole: JSON.parse(JSON.stringify(data)) });
    }

    selectTab(selectedTab) {
        this.setState({
            selectedTab,
        });
    }

    getRole() {
        this.props.role_id && this.setState({
            roleIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_ROLES, { queryParams: { role_id: encodeURIComponent(this.props.role_id) } })
                .then(response => {
                    //this.setState({roleIsLoading: false, currentRole: response.report || []});
                    const currentRole = response?.report?.[0] || { role_id: this.props.role_id };
                    const currentPropositions = response?.propositions;
                    this.setState({
                        currentRole,
                        currentPropositions,
                        roleIsLoading: false,
                    }, () => {
                        this.setItems();
                    });
                })
                .catch(error => {
                    this.setState({ roleIsLoading: false });
                });
        });
    }

    getRolesActions() {
        this.setState({ actionsIsLoading: true }, () => {
            this.request.exec(REQUESTS.GET_ACTIONS, { queryParams: { report: 'compact' } })
                .then(response => {
                    this.setState({ actionsIsLoading: false, allActions: response.report || [] });
                })
                .catch(error => {
                    this.setState({ actionsIsLoading: false, allActions: [] });
                });
        });

        this.setState({ rolesIsLoading: true }, () => {
            this.request.exec(REQUESTS.GET_ROLES, { queryParams: { report: 'compact' } })
                .then(response => {
                    this.setState({ rolesIsLoading: false, allRoles: response.report || [] });
                })
                .catch(error => {
                    this.setState({ rolesIsLoading: false, allRoles: [] });
                });
        });
    }

    componentDidMount(): void {
        this.getRole();
        this.getRolesActions();
    }

    componentWillUnmount() {
        this.request.abort();
    }

    setItems() {
        /*let editRole: any = this.props.report.find(role => role.role_id === this.props.role_id)
            || {};*/
        let editRole: any = this.state.currentRole;

        /* if (this.props.proposition_id) {
             editRole = this.props.propositions[editRole.role_id]
                 && this.props.propositions[editRole.role_id].find(pr => pr.proposition_id === this.props.proposition_id)
                 || {};
         }*/

        if (this.props.proposition_id) {
            editRole = this.state?.currentPropositions
                ?.find((pr: any) => pr.proposition_id === this.props.proposition_id)
                || {} as RoleItem;
        }

        /* let propositions = this.props.propositions[editRole.role_id] || [];*/
        const propositions: any = this.state.currentPropositions?.filter(p => p.role_id === editRole?.role_id) || [];

        const actions = editRole?.actions && editRole.actions.map(item => {
            return {
                name: item.action_id,
                time_restrictions: item.role_action_meta && item.role_action_meta.time_restrictions,
            };
        }) || [];

        const roles = editRole?.slave_roles && editRole.slave_roles.map(item => {
            return {
                name: item.slave_role_id,
                time_restrictions: item.link_meta && item.link_meta.time_restrictions,
            };
        });

        this.setState({
            editRole,
            actions,
            roles,
            propositions,
        });
    }

    componentDidUpdate(prevProps: Readonly<IEditRoleProps>, prevState: Readonly<IEditRoleState>, snapshot?: any): void {
        /* if (!isObjectEqual(this.props.report, prevProps.report)
             || !isObjectEqual(this.props.propositions, prevProps.propositions)
             || this.props.role_id !== prevProps.role_id
             || this.props.proposition_id !== prevProps.proposition_id) {
             this.setItems();
         }*/
        if (
            this.props.role_id !== prevProps.role_id
            || this.props.proposition_id !== prevProps.proposition_id
        ) {
            this.getRole();
            //this.setItems();
        }
    }

    onDataChange(type: string, value) {
        this.setState({
            [type]: value,
        });
    }

    onSelect(type: string, items: string[]) {
        const entities = this.state[type] || [];
        items.forEach((item => {
            if (!entities.find(e => e.name === item)) {
                entities.push({ name: item, time_restriction: undefined });
            }
        }));
        this.setState({
            [type]: entities,
        });
    }

    propose() {
        this.setState({
            showProposeComment: true,
            isWorking: false,
        });
    }

    showComment(state) {
        this.setState({
            showProposeComment: state,
        });
    }

    setComment(comment, force = false) {
        const actions = this.state.actions && this.state.actions.map(action => {
            const meta = action.time_restrictions
                && action.time_restrictions.restrictions
                && action.time_restrictions.restrictions.length && {
                time_restrictions: action.time_restrictions,
            } || {};

            return {
                action: this.state.allActions.find(_action => _action.action_id === action.name),
                action_id: action.name,
                role_action_meta: meta,
                role_id: this.state._editedRole.role_id,
            };
        }) || [];
        const slave_roles = this.state.roles && this.state.roles.map(_role => {
            const meta = _role.time_restrictions
                && _role.time_restrictions.restrictions
                && _role.time_restrictions.restrictions.length && {
                time_restrictions: _role.time_restrictions,
            } || {};

            return {
                slave_role_id: _role.name,
                role_id: this.state._editedRole.role_id,
                link_meta: meta,
            };
        }) || [];

        const boolToStr = (param) => {
            return typeof param === 'string' ? param : (param ? '1' : '0');
        };

        const proposeObject = Object.assign({}, this.state._editedRole, {
            role_is_idm: boolToStr(this.state._editedRole.role_is_idm),
            role_is_public: boolToStr(this.state._editedRole.role_is_public),
            role_optional: boolToStr(this.state._editedRole.role_optional),
        }, {
            actions,
            slave_roles,
        });

        this.setState({
            isWorking: true,
            proposeError: null,
        }, () => {
            this.request.exec(
                !force
                    ? REQUESTS.ROLES_SNAPSHOT_PROPOSE
                    : REQUESTS.ROLES_SNAPSHOT_UPSERT, {
                    queryParams: {
                        comment: encodeURIComponent(comment) || `edit role: ${new Date().getTime()}`,
                    },
                    body: [proposeObject],
                })
                .then(() => {
                    this.setState({
                        isWorking: false,
                        proposeError: null,
                        showProposeComment: false,
                    }, () => {
                        this.props.update();
                    });
                })
                .catch((error) => {
                    this.setState({
                        isWorking: false,
                        proposeError: error,
                    });
                });
        });
    }

    confirmAction(props: { confirmQuestion: string; action: string }) {
        this.setState({
            confirmIsOpen: true,
            errorConfirm: null,
            confirmQuestion: props.confirmQuestion,
            confirmAction: () => {
                this.setState({
                    confirmIsLoading: true,
                    errorConfirm: null,
                }, () => {
                    const action = `ROLES_SNAPSHOT_${props.action.toUpperCase()}`;

                    REQUESTS[action] && this.request.exec(REQUESTS[action], {
                        body: { proposition_ids: this.props.proposition_id },
                    })
                        .then(() => {
                            this.setState({
                                confirmIsLoading: false,
                                errorConfirm: null,
                                confirmIsOpen: false,
                            }, () => {
                                this.props.update();
                                location.href = `#/settings/roles?role_id=${this.props.role_id}`;
                            });
                        })
                        .catch((error) => {
                            this.setState({
                                confirmIsLoading: false,
                                errorConfirm: error,
                            });
                        });
                });
            },
        });
    }

    closeConfirm() {
        this.setState({
            confirmIsOpen: false,
            errorConfirm: null,
        });
    }

    upsert() {
        this.setComment(null, true);
    }

    render() {
        const text = `Редактирование роли ${this.props.role_id}`;

        return <Window onClose={this.props.onCLose.bind(this)} title={text} error={this.state.error}>
            {
                this.state.confirmIsOpen && <Confirm error={this.state.errorConfirm}
                                                     question={this.state.confirmQuestion}
                                                     accept={this.state.confirmAction}
                                                     isWorking={this.state.confirmIsLoading}
                                                     onClose={this.closeConfirm.bind(this)}/>
            }
            {
                this.state.showProposeComment
                && <ProposeCommentModal isWorking={this.state.isWorking}
                                        error={this.state.proposeError}
                                        setComment={this.setComment.bind(this)}
                                        onClose={this.showComment.bind(this, false)}/>
            }
            <div className={style.component}>
                <div className={style.role_settings}>
                    {
                        this.state.roleIsLoading ? <Spin/> : null
                    }
                    <FormConstructor schema={shemaRole}
                                     initialData={this.state.editRole}
                                     onChange={this.onChange.bind(this)}/>
                </div>
                <div className={style.role_options}>
                    <Tabs currentTab={this.state.selectedTab}
                          selectTab={this.selectTab.bind(this)}
                          tabs={actions({
                              actions: this.state.actions && this.state.actions.length || 0,
                              roles: this.state.roles && this.state.roles.length || 0,
                          })}/>
                    <div className={style.tabs_content}>
                        {
                            this.state.selectedTab === RoleSettingsTabs.ACTIONS
                                ? <Actions data={this.state.actions}
                                           isLoading={this.state.actionsIsLoading}
                                           allData={this.state.allActions}
                                           onSelect={this.onSelect.bind(this, 'actions')}
                                           onChange={this.onDataChange.bind(this, 'actions')}/>
                                : <Roles data={this.state.roles}
                                         isLoading={this.state.rolesIsLoading}
                                         allData={this.state.allRoles}
                                         onSelect={this.onSelect.bind(this, 'roles')}
                                         onChange={this.onDataChange.bind(this, 'roles')}/>
                        }
                    </div>
                </div>
                <div className={style.propositions}>
                    <div>
                        <strong>Propositions: </strong>
                        <span> <Link href={`#/settings/roles?role_id=${encodeURIComponent(this.props.role_id)}`}>
                            Роль
                        </Link> </span>
                        {this.state.propositions.map((pr, index) => {
                            const href = `#/settings/roles?role_id=${encodeURIComponent(this.props.role_id)}&proposition_id=${pr.proposition_id}`;

                            return <span key={index}> <Link href={href}>{index}</Link> </span>;
                        })}
                        <Copy text={`${location.origin}${location.pathname}#/lite-confirmations?type=role&id=${encodeURIComponent(this.props.role_id)}`}>
                            Копировать ссылку на предложения
                        </Copy>
                    </div>
                    <div>
                        <strong>Description:</strong> {this.state.editRole?.proposition_description} <span> </span>
                        <div><strong>Date: </strong><FormatDate value={this.state.editRole?.confirmations
                        && this.state.editRole.confirmations[0]
                        && this.state.editRole.confirmations[0].history_instant * ONE_SECOND}
                                                                withSecond/></div>
                    </div>
                </div>
                <div className={style.error}>
                    {
                        this.state.proposeError && <SimpleError error={this.state.proposeError}/>
                    }
                </div>
                <div className={style.controls}>
                    {!this.props.proposition_id && <>
                        <Button colorType={ButtonTypes.positive}
                                onClick={this.upsert.bind(this)}
                                disabled={!(this.state._editedRole && this.state._editedRole.role_id)}
                                isLoading={this.state.isWorking}>
                            Сохранить
                        </Button>
                        <Button colorType={ButtonTypes.warning}
                                onClick={this.propose.bind(this)}
                                isLoading={this.state.isWorking}>
                            Предложить
                        </Button>
                    </>}
                    {
                        this.props.proposition_id && <>
                            <Button colorType={ButtonTypes.negative}
                                    onClick={this.confirmAction.bind(this, {
                                        confirmQuestion: 'Отклонить предложение?',
                                        action: 'reject',
                                    })}
                                    disabled={!this.state.propositions?.length}>
                                Отклонить предложение
                            </Button>
                            <Button colorType={ButtonTypes.positive}
                                    onClick={this.confirmAction.bind(this, {
                                        confirmQuestion: 'Принять предложение?',
                                        action: 'confirm',
                                    })}
                                    disabled={!this.state.propositions?.length}>
                                Принять предложение
                            </Button>
                        </>
                    }
                    <Button colorType={ButtonTypes.negative}
                            basic
                            className={style.right}
                            onClick={this.props.onCLose.bind(this)}>
                        Закрыть
                    </Button>
                    <Button basic
                            isLoading={this.state.actionsIsLoading || this.state.rolesIsLoading}
                            onClick={this.getRolesActions.bind(this)}>
                        Обновить список экшенов({this.state.allActions?.length}) / ролей({this.state.allRoles?.length})
                    </Button>
                </div>
            </div>
        </Window>;
    }
}

interface IDataProps {
    data: IWeekCalendarPropsItem[];
    isLoading: boolean;
    allData: any[];
    onChange: () => void;
    onSelect: (items: string[]) => void;
}

const Actions = React.memo((props: IDataProps) => {
    const itemsObj = props.data.reduce((_p, _c) => {
        if (!_p.hasOwnProperty(_c.name)) {
            _p[_c.name] = _c;
        }

        return _p;
    }, {});
    const options = props.allData && props.allData
        .map(item => {
            if (itemsObj.hasOwnProperty(item.action_id)) {
                itemsObj[item.action_id] = {
                    description: item.action_description,
                    type: item.action_type,
                    link: `#/settings/actions/${item.action_id}`,
                    enabled: item.enabled,
                };
            }

            return {
                text: item.action_id,
                value: item.action_id,
                description: `rev.${item.action_revision}: ${item.action_description || EMPTY_DATA}`,
            };
        });

    return <div>
        <Select onSelect={props.onSelect}
                options={options}
                multiSelect
                placeholder={'Экшены'}
                disabled={props.isLoading}/>
        <WeekCalendar items={props.data} onChange={props.onChange} itemsObj={itemsObj}/>
    </div>;
});

const Roles = React.memo((props: IDataProps) => {
    const itemsObj = props?.data?.reduce((_p, _c) => {
        if (!_p.hasOwnProperty(_c.name)) {
            _p[_c.name] = _c;
        }

        return _p;
    }, {}) || {};

    const options = props.allData && props.allData
        .map(item => {
            if (itemsObj.hasOwnProperty(item.role_id)) {
                itemsObj[item.role_id] = {
                    description: item.role_description,
                    type: '',
                    link: `#settings/roles?role_id=${item.role_id}`,
                    enabled: true,
                };
            }

            return {
                text: item.role_id,
                value: item.role_id,
                description: `${item.role_description || EMPTY_DATA}`,
            };
        });

    return <div>
        <Select onSelect={props.onSelect}
                options={options}
                multiSelect
                placeholder={'Роли'}
                disabled={props.isLoading}/>
        {props.data && itemsObj && <WeekCalendar items={props.data} onChange={props.onChange} itemsObj={itemsObj}/>}
    </div>;
});
