import * as React from 'react';

import { Dict } from '../../../../types';
import { BlockRuleType } from '../../../reducers/adminUserReducer';
import { Button, ButtonTypes } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import * as coreStyle from '../../../ui/index.css';
import { LabelStatus, TLabel } from '../../../ui/Table';
import * as styleTable from '../../../ui/Table/index.css';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { Translate } from '../../../utils/translate';
import { IStore } from '../../App/store';
import { FormConstructor } from '../../FormConstructor';
import { controlType } from '../../FormConstructor/types';
import { BLOCK_RULES_KEY_UI } from '../../InterfaceAdminConfig/adminConfigKeys';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { IPostSettingItem } from '../../UISettings/component';
import { GLOBAL_VARS_REQUESTS as requestConfigs, REQUESTS } from '../GlobalVarsView/request';
import * as style from './index.css';

interface IBlockRule {
    roles: string[];
    env: string[];
    description: string;
    active: boolean;
    type: BlockRuleType;
}

interface IEditRulesProps extends IStore {
    userId: string;
    BlockRules: Dict<any>;
}

interface IEditRulesState {
    blockRules: Dict<any>;
    isBlockRuleEditorOpen: boolean;
    blockRuleEntry: [string, IBlockRule] | null;
    isBlockRulesLoading: boolean;
    blockRulesError: Error | null;
}

export default class EditRules extends React.Component<IEditRulesProps, IEditRulesState> {
    state: IEditRulesState = {
        blockRules: {},
        isBlockRuleEditorOpen: false,
        blockRuleEntry: null,
        isBlockRulesLoading: false,
        blockRulesError: null,
    };
    request = new Request2({
        requestConfigs,
    });
    t = this.props.Lang && new Translate(this.props.Lang) || {} as Translate;

    componentDidMount(): void {
        this.getData();
    }

    getData() {
        this.setState({ isBlockRulesLoading: true }, () => {
            this.request.exec(REQUESTS.GET_GLOBAL_SETTINGS)
                .then(response => {
                    const blockRulesStringObj = response && response.settings && Array.isArray(response.settings)
                    && response.settings.length
                        ? response.settings.filter(setting => setting.setting_key === 'interface_admin.block_rules')[0]
                        : '';

                    const blockRules = blockRulesStringObj && isValidJSONString(blockRulesStringObj.setting_value)
                        ? JSON.parse(blockRulesStringObj.setting_value)
                        : {};

                    this.setState({ blockRules, isBlockRulesLoading: false });
                })
                .catch((blockRulesError) => {
                    this.setState({
                        blockRulesError,
                        isBlockRulesLoading: false,
                    });
                });
        });

    }

    openBlockRuleEditor(blockRuleEntry: [string, IBlockRule]) {
        if (blockRuleEntry) {
            this.setState({ isBlockRuleEditorOpen: true, blockRuleEntry });

        } else {
            this.setState({ isBlockRuleEditorOpen: true });
        }
    }

    closeBlockRuleEditor() {
        this.setState({ isBlockRuleEditorOpen: false, blockRuleEntry: null });
    }

    onUpdate(value: Dict<IBlockRule>) {
        const newValues = Object.assign({}, this.state.blockRules, value);
        const body: IPostSettingItem = {
            settings: [{
                setting_key: BLOCK_RULES_KEY_UI,
                setting_value: JSON.stringify(newValues),
            }],
        };

        this.request.exec(REQUESTS.UPSERT_SETTINGS, { body })
            .then(() => {
                this.getData();
                this.closeBlockRuleEditor();
            });
    }

    render() {
        if (this.state.blockRulesError) {
            return <SimpleError error={this.state.blockRulesError} data={{ label: 'Ошибка при загрузке правил' }}/>;
        }

        if (this.state.isBlockRulesLoading) {
            return <Spin/>;
        }

        return <div className={style.edit_rules}>
            <div className={style.create_button_container}>
                <Button basic
                        colorType={ButtonTypes.positive}
                        onClick={this.openBlockRuleEditor.bind(this)}>Создать правило</Button>
            </div>
            <div className={style.tableContainer}>
                <table className={`${styleTable.table} ${style.rules_table}`}>
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Правило</th>
                            <th>Описание</th>
                            <th/>
                            <th/>
                            <th>Тип</th>
                            <th>Роли</th>
                            <th>Окружение</th>
                        </tr>
                    </thead>
                    <tbody>
                        {this.state.blockRules
                            && Object.entries(this.state.blockRules).map((entry: [string, IBlockRule], index) => {
                                const [key, value] = entry;

                                return <tr key={key} onClick={this.openBlockRuleEditor.bind(this, entry)}>
                                    <td>{index + 1}</td>
                                    <td>{key}</td>
                                    <td>{value.description}</td>
                                    <td>
                                        <TLabel status={value.active ? LabelStatus.POSITIVE : LabelStatus.DEFAULT}
                                                text={value.active ? 'Активно' : 'Не активно'}/>
                                    </td>
                                    <td>
                                        {
                                            this.props.BlockRules[key]
                                                ? <TLabel status={LabelStatus.INFO} text={'Доступно'}/>
                                                : null
                                        }
                                    </td>
                                    <td>
                                        {value.type}
                                    </td>
                                    <td>
                                        {value?.roles?.join?.(', ')}
                                    </td>
                                    <td>
                                        {value?.env?.join?.(', ')}
                                    </td>
                                </tr>;
                            })}
                    </tbody>
                </table>
            </div>
            {this.state.isBlockRuleEditorOpen
                ? <BlockRuleEditor blockRuleEntry={this.state.blockRuleEntry}
                                   allBlockRulesList={this.state.blockRules}
                                   update={this.onUpdate.bind(this)}
                                   onClose={this.closeBlockRuleEditor.bind(this)}/>
                : null}
        </div>;
    }
}

class BlockRuleEditor extends React.Component<any, any> {
    state = {
        values: {},
    };

    mainSchema = {
        name: { type: controlType.string, display_name: 'Имя', order: 1, required: true },
        description: { type: controlType.string, display_name: 'Описание', order: 2 },
        type: {
            type: controlType.variants,
            multi_select: false,
            display_name: 'Тип',
            variants: ['show'],
            order: 3,
        },
        roles: {
            type: controlType.variants,
            multi_select: true,
            display_name: 'Роли',
            variants: [],
            order: 4,
            editable: true,
        },
        active: {
            type: controlType.bool,
            display_name: 'Активно',
            order: 6,
        },
    };

    onChange(values: any) {
        const newValue = { [values.name]: {} };

        Object.entries(values).reduce((result: Dict<any>, entry: [string, any]) => {
            const [key, value] = entry;

            if (key !== 'name') {
                result[values.name][key] = value;
            }

            return result;
        }, newValue);

        this.setState({ values: newValue });
    }

    render() {
        const initialData = this.props.blockRuleEntry
            ? Object.assign({}, { name: this.props.blockRuleEntry[0] }, this.props.blockRuleEntry[1])
            : {};

        return <Window title={initialData ? 'Редактирование правила' : 'Создание правила'}
                       onClose={this.props.onClose.bind(this)}>
            <FormConstructor initialData={initialData}
                             schema={this.mainSchema}
                             onChange={this.onChange.bind(this)}/>
            {Object.keys(this.props.allBlockRulesList).includes(Object.keys(this.state.values)[0])
            && Object.keys(this.state.values)[0] !== initialData.name
                ? <h3 className={style.warning}>Такое правило уже существует!</h3>
                : null}
            <div className={coreStyle.button_container}>
                <Button onClick={this.props.onClose.bind(this)} basic colorType={ButtonTypes.negative}>Отмена</Button>
                <Button onClick={this.props.update.bind(this, this.state.values)}>
                    {initialData ? 'Обновить' : 'Создать'}
                </Button>
            </div>
        </Window>;
    }
}
