import React from 'react';

import { Button, ButtonTypes } from '../../../ui/Button';
import Select from '../../../ui/Select';
import TextArea from '../../../ui/TextArea';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { download } from '../../../utils/utils';
import { initMap } from '../../MainMap/utils';
import Spin from '../../Spin';
import { IPolygonItem } from '../Polygons/component';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import style from './index.css';

enum PolygonQueryType {
    searchInside = 'searchInside',
    match_tag = 'match_tag',
    match_id = 'match_id'
}

interface IPolygonQueryGroupItem {
    type: PolygonQueryType;
    tag_name: string;
    polygon_id?: string;
    style: string;
    title: string;
}

interface IPolygonQuery {
    groups: IPolygonQueryGroupItem[];
    styles: {};
}

interface IExportPolygonStat {
    query: '';
    error: Error | null;
    isLoading: boolean;
    areas: IPolygonItem[];
    city: string;
}

declare let ymaps: any;

export class ExportPolygon extends React.Component<any, IExportPolygonStat> {
    state: IExportPolygonStat = {
        query: '',
        isLoading: true,
        error: null,
        areas: [],
        city: '',
    };
    map: any;
    polygonsCollection: any;
    prevPolygonsCollection: any[];
    queryCollection: any;

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    onTextAreaChanged(query) {
        this.setState({
            query,
        });
    }

    onChangeSelect(value) {
        this.setState({
            city: value,
            query: this.props.queries.find(el => el.name === value)?.query,
        });
    }

    getQueryObject() {
        let queryObject: IPolygonQuery | null = null;
        if (isValidJSONString(this.state.query)) {
            queryObject = JSON.parse(this.state.query);
        }

        return queryObject;
    }

    defaultColors() {
        return {
            fillColor: 'bb931e',
            fillOpacity: .4,
            strokeColor: 'bb931e',
        };
    }

    getPolygons() {
        this.request.exec(REQUESTS.AREAS_INFO)
            .then(response => {

                response.areas?.map(poly => {
                    this.polygonsCollection.add(new ymaps.Polygon([poly.area_coords], {
                        hintContent: `${poly.title || ''} ${poly.area_id} : ${poly.area_tags || ''}`,
                    }, {
                        ...this.defaultColors(),
                        area_tags: poly.area_tags,
                        area_id: poly.area_id,
                        area_size: poly.area_size,
                    }));
                });

                this.map.setBounds(this.polygonsCollection.getBounds());

                this.setState({
                    areas: response.areas || [],
                    isLoading: false,
                });
            })
            .catch(error => {
                this.setState({
                    areas: [],
                    isLoading: false,
                    error,
                });
            });
    }

    initMap() {
        initMap('poly_map', (map) => {
            this.map = map;
            this.polygonsCollection = new ymaps.GeoObjectCollection();
            this.prevPolygonsCollection = [];
            this.queryCollection = new ymaps.GeoObjectCollection();
            this.map.geoObjects.add(this.polygonsCollection);
            this.map.geoObjects.add(this.queryCollection);

            this.drawMap();
        });
    }

    drawMap() {
        this.getPolygons();
    }

    componentDidMount() {
        this.initMap();
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getMainPoly(el) {
        let mainPoly = ymaps.geoQuery();
        this.polygonsCollection?.each(poly => {
            if (poly.options.get('area_id') === el.polygon_id) {
                mainPoly = poly;
            }
        });

        return mainPoly;
    }

    addToResult(results, result, obj, el) {
        result.setOptions(obj?.styles[el.style]);
        result.setOptions({ description: el.description });
        result.each(el => results.push(el));
    }

    viewQueryResult() {
        this.clearViewQueryResult();

        this.polygonsCollection.each(el => {
            this.prevPolygonsCollection.push(el);
        });
        const results: any[] = [];

        const obj = this.getQueryObject();
        obj?.groups?.forEach((el) => {
            if (el.type === PolygonQueryType.match_id) {

                const mainPoly = this.getMainPoly(el);

                const result = ymaps.geoQuery(mainPoly);
                this.addToResult(results, result, obj, el);
            }

            if (el.type === PolygonQueryType.match_tag) {
                const result = ymaps.geoQuery(this.map.geoObjects).search((poly) => {
                    const tags = (poly.options.get('area_tags') || '').split(',').map(el => el.trim());
                    const regExp = new RegExp(`${el.tag_name}`);

                    return tags.some(tag => tag.match(regExp));
                });
                this.addToResult(results, result, obj, el);
            }

            if (el.type === PolygonQueryType.searchInside) {
                const mainPoly = this.getMainPoly(el);

                const result = ymaps.geoQuery(this.map.geoObjects)
                    .searchInside(mainPoly)
                    .search((poly) => {
                        const tags = (poly.options.get('area_tags') || '').split(',').map(el => el.trim());
                        const regExp = new RegExp(`${el.tag_name}`);

                        return tags.some(tag => tag.match(regExp));
                    });
                this.addToResult(results, result, obj, el);
            }
        });

        results.sort((a, b) => {
            return b.options.get('area_size') - a.options.get('area_size');
        })
            .forEach(el => {
                this.queryCollection.add(el);
            });

        this.polygonsCollection.removeAll();
        if (this.queryCollection && this.queryCollection.getLength()) {
            this.map.setBounds(this.queryCollection.getBounds());
        }
    }

    clearViewQueryResult() {
        this.queryCollection.removeAll();
        this.prevPolygonsCollection.length && this.prevPolygonsCollection.forEach(el => {
            el.options.set(this.defaultColors());
            this.polygonsCollection.add(el);
        });
        this.prevPolygonsCollection = [];
    }

    exportToGeoJson() {
        const obj: any = {
            'type': 'FeatureCollection',
            'features': [],
        };

        class Feature {
            type = 'Feature';
            id = '';

            'geometry' = {
                'type': 'Polygon',
                coordinates: [],
            };
            properties: any = {};

            constructor(props) {
                this.id = props.id;
                this.geometry.coordinates = props.coordinates;
                this.properties = props.properties;
                this.properties.fill = props.properties.fillColor;
                this.properties['fill-opacity'] = props.properties.fillOpacity;
                this.properties.stroke = props.properties.strokeColor;
                this.properties['stroke-width'] = props.properties.strokeWidth;
            }
        }

        this.queryCollection?.each(polygon => {
            obj.features.push(new Feature({
                id: polygon.options.get('area_id'),
                coordinates: polygon.geometry.getCoordinates(),
                properties: polygon.options.getAll(),
            }));
        });

        download(JSON.stringify(obj),
            `${this.state.city}-poly-${this.queryCollection?.getLength()}.json`,
            'text/json');
    }

    render() {
        const options = (this.props.queries || []).filter(el => el.enabled).map(el => {
            return {
                text: el.title,
                value: el.name,
            };
        });

        return <div className={style.content}>
            <div className={`${style.panel} ${style.controls}`}>
                <Select onSelect={this.onChangeSelect.bind(this)}
                        disabled={!this.state.areas.length}
                        containerClassName={`${style.select}`}
                        placeholder={'город'}
                        options={options}/>
                {this.state.isLoading ? <Spin size={'s'}/> : null}
            </div>
            <div className={`${style.panel} ${style.query}`}>
                <TextArea onChange={this.onTextAreaChanged.bind(this)}
                          disabled={!this.state.areas.length}
                          className={style.textarea}
                          value={this.state.query}
                          placeholder={'запрос'}/>
                <div className={style.query_controls}>
                    <Button onClick={this.viewQueryResult.bind(this)}
                            disabled={!this.state.query}
                            colorType={ButtonTypes.positive}>Просмотр</Button>
                    <Button disabled={!this.state.query}
                            onClick={this.clearViewQueryResult.bind(this)}
                            colorType={ButtonTypes.warning}>Сбросить</Button>
                    <Button onClick={this.exportToGeoJson.bind(this)}>geoJSON</Button>
                </div>
            </div>
            <div className={`${style.panel} ${style.map}`} id={'poly_map'}/>
        </div>;
    }
}
