import * as React from 'react';
import { RouteComponentProps } from 'react-router';

import { Dict } from '../../../../../types';
import { UserInfoHandler } from '../../../../models/user';
import { Button, ButtonTypes, CancelButton, DeleteButton, SaveButton } from '../../../../ui/Button';
import { Confirm, Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import ProposeExistWarningModal from '../../../../ui/ProposeExistWarningModal/ProposeExistWarningModal';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import { Copy } from '../../../Copy';
import { FormConstructor } from '../../../FormConstructor';
import { controlType, ISchemaItem } from '../../../FormConstructor/types';
import { ProposeCommentModal } from '../../ProposeCommentModal';
import { VARIABLE_SEPARATOR } from '../constants';
import { GLOBAL_VARS_REQUESTS, REQUESTS } from '../request';
import { IPropositionItem, ISettingsItem } from '../types';
import style from './index.css';

enum ProposeAction {
    REJECT = 'REJECT',
    CONFIRM = 'CONFIRM'
}

interface IAddGlobalVarModalProps extends RouteComponentProps<{
    setting_key: string;
    proposition: string;
}> {
    initialData: { setting_key?: string; setting_value?: string } | null;
    propositions?: IPropositionItem[];
    globalSettings?: ISettingsItem[];
    onClose: (isSuccessful: boolean) => {};
    getSettings: () => void;
    users: Record<string, any>;
}

interface IAddGlobalVarModalState {
    initialData: { setting_key?: string; setting_value?: string; revision?: string; propositions?: IPropositionItem[] }
    | IPropositionItem;
    settingData: { setting_key?: string; setting_value?: string };
    isWorking: boolean;
    error: Error | null;
    onConfirmClose: () => void;
    confirmRemoveVarModalIsOpen: boolean;
    confirmUpsertVarModalIsOpen: boolean;
    accept: () => void;
    question: string;
    showProposeComment: boolean;
    isWarningModalOpen: boolean;
    proposeError: Error | null;
    success: boolean;
    proposeAction: ProposeAction | null;
}

export default class AddGlobalVarModal extends React.Component<IAddGlobalVarModalProps, IAddGlobalVarModalState> {
    state: IAddGlobalVarModalState = {
        initialData: {},
        settingData: {},
        isWorking: false,
        error: null,
        onConfirmClose: () => {},
        confirmRemoveVarModalIsOpen: false,
        confirmUpsertVarModalIsOpen: false,
        accept: () => {},
        question: '',
        showProposeComment: false,
        isWarningModalOpen: false,
        proposeError: null,
        success: false,
        proposeAction: null,
    };
    request = new Request2({ requestConfigs: GLOBAL_VARS_REQUESTS });
    schema: Dict<ISchemaItem> = {
        setting_key: {
            display_name: 'Ключ',
            order: 1,
            required: true,
            type: controlType.string,
        },
        setting_value: {
            display_name: 'Значение',
            order: 2,
            type: controlType.text,
        },
    };

    constructor(props: IAddGlobalVarModalProps) {
        super(props);
        const initialData = this.props.initialData || {};

        this.state = { ...this.state, initialData };
    }

    componentDidMount() {
        const { initialData } = this.props;
        if (!initialData) {
            this.setInitialData();
        }
    }

    componentDidUpdate(prevProps: Readonly<IAddGlobalVarModalProps>): void {
        if (!isObjectEqual(this.props.initialData, prevProps.initialData)) {
            this.setState({ initialData: this.props.initialData || {} });
        }

        if (!isObjectEqual(this.props.propositions, prevProps.propositions)
            || !isObjectEqual(this.props.globalSettings, prevProps.globalSettings)
            || !isObjectEqual(this.props.match, prevProps.match)) {
            this.setInitialData();
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    setInitialData() {
        const { propositions, match, globalSettings } = this.props;
        const { setting_key, proposition } = match?.params;
        const filteredPropositions = propositions?.filter((el) => el.setting_key === setting_key) ?? [];

        if (setting_key) {
            const initialData = proposition === 'proposition'
                ? filteredPropositions?.[0] || {}
                : globalSettings?.filter((el) => el.setting_key === setting_key)?.[0] || {};
            this.setState({ initialData });
        }
    }

    onSaveClick() {
        const { settingData, initialData } = this.state;

        if (initialData?.setting_key
            && initialData?.setting_key?.[initialData?.setting_key.length - 1] !== VARIABLE_SEPARATOR
            && settingData.setting_key !== initialData?.setting_key) {
            this.setState({ confirmUpsertVarModalIsOpen: true });
        } else {
            this.saveGVar(settingData);
        }
    }

    onReplace() {
        const { settingData, initialData } = this.state;

        this.saveGVar(settingData);
        this.deleteGVar(initialData || {});
    }

    onCopy() {
        const { settingData } = this.state;
        this.saveGVar(settingData);
    }

    saveGVar(settingData: { setting_key?: string; setting_value?: string }) {
        this.setState({
            error: null,
            isWorking: true,
        }, () => {
            this.request.exec(REQUESTS.UPSERT_SETTINGS, { body: { settings: [settingData] } })
                .then(() => {
                    this.setState({ isWorking: false });
                    this.props.onClose(true);
                })
                .catch(error => {
                    this.setState({
                        error,
                        isWorking: false,
                    });
                });
        });
    }

    onDeleteClick() {
        const { settingData } = this.state;

        this.setState({
            confirmRemoveVarModalIsOpen: true,
            question: `Удалить переменную ${settingData.setting_key}?`,
            accept: () => {
                this.deleteGVar(settingData);
            },
            onConfirmClose: () => {
                this.setState({
                    isWorking: false,
                    confirmRemoveVarModalIsOpen: false,
                });
            },
        });
    }

    deleteGVar(settingData: { setting_key?: string; setting_value?: string }) {
        this.setState({
            error: null,
            isWorking: true,
        }, () => {
            this.request.exec(REQUESTS.REMOVE_SETTINGS, { body: { settings: [settingData] } })
                .then(() => {

                    this.setState({
                        confirmRemoveVarModalIsOpen: false,
                        isWorking: false,
                    });
                    this.props.onClose(true);
                })
                .catch(error => {
                    this.setState({
                        error,
                        isWorking: false,
                    });
                });
        });
    }

    onConfirmClose() {
        this.setState({
            confirmRemoveVarModalIsOpen: false,
            confirmUpsertVarModalIsOpen: false,
        });
    }

    onFormChange(settingData: Dict<ISettingsItem>) {
        this.setState({ settingData });
    }

    propose() {
        if (!this.checkIfPropositionExist()) {
            this.setState({
                showProposeComment: true,
                isWorking: false,
            });
        } else {
            this.setState({ isWarningModalOpen: true, isWorking: false });
        }
    }

    checkIfPropositionExist() {
        const { propositions } = this.props;
        const { settingData } = this.state;

        return propositions?.find((el) => el.setting_key === settingData?.setting_key);
    }

    showComment(state) {
        this.setState({
            showProposeComment: state,
        });
    }

    closeCopyModal(proposeError) {
        this.setState({ showProposeComment: false }, () => {
            if (proposeError) {
                this.setState({ proposeError: null });
            } else {
                this.props.onClose(true);
                this.props.getSettings();
            }
        });
    }

    setComment(comment) {
        this.setState({
            isWorking: true,
            proposeError: null,
        }, () => {
            const { settingData, initialData } = this.state;
            const proposeObject = Object.assign({}, settingData, {
                revision: initialData?.revision,
            });

            this.request.exec(REQUESTS.SETTINGS_PROPOSE, {
                queryParams: {
                    comment: encodeURIComponent(comment),
                },
                body: { settings: [proposeObject] },
            })
                .then(() => {
                    this.setState({
                        isWorking: false,
                        proposeError: null,
                        success: true,
                        showProposeComment: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        isWorking: false,
                        proposeError: error,
                    });
                });
        });
    }

    showConfirm(proposeAction: ProposeAction, proposeError) {
        this.setState({
            proposeAction,
            proposeError,
        });
    }

    proposeAccept() {
        this.setState({
            proposeError: null,
            isWorking: true,
        }, () => {
            const { onClose, getSettings } = this.props;
            const initialData = this.state.initialData as IPropositionItem;
            const action = `SETTINGS_${this.state.proposeAction}`;

            this.request.exec(REQUESTS?.[action], {
                queryParams: {
                    comment: initialData.proposition_description || action,
                },
                body: {
                    proposition_ids: [initialData?.proposition_id],
                },
            })
                .then(() => {
                    this.setState({
                        proposeError: null,
                        isWorking: false,
                        proposeAction: null,
                    }, () => {
                        onClose(true);
                        getSettings();
                    });
                })
                .catch((proposeError) => {
                    let error = proposeError;
                    if (error?.data?.error_details?.special_info?.session_info
                        ?.hasOwnProperty('self confirm is denied')) {
                        error = new Error('Нельзя подтверждать свои же предложения :(');
                    }

                    this.setState({
                        proposeError: error,
                        isWorking: false,
                    });
                });
        });
    }

    render() {
        const {
            initialData, error, confirmRemoveVarModalIsOpen, confirmUpsertVarModalIsOpen, settingData,
            accept, isWorking, question, proposeError, showProposeComment, proposeAction, success, isWarningModalOpen,
        } = this.state;
        const { onClose, users, match } = this.props;
        const title = initialData.setting_value ? 'Обновить значение' : 'Добавить значение';
        const authorInfo = users[(initialData as IPropositionItem).proposition_author];
        const author = UserInfoHandler.getPrintName.call(authorInfo);
        const href = `#/settings/global-vars/${(initialData as ISettingsItem).propositions?.[0]?.setting_key}/proposition`;
        const showPropositions = (initialData as ISettingsItem).propositions
            ? false
            : match?.params?.setting_key !== undefined && match?.params?.proposition === 'proposition';

        return <Window title={title}
                       error={error}
                       className={style.modal}
                       onClose={onClose.bind(this, false)}>
            <div>
                {showPropositions
                    && <div className={style.loc_item}>
                        <div>
                            <span>
                            Автор
                            </span>: <span className={style.loc_item_value}>
                                <Link href={`#/clients/${(initialData as IPropositionItem).proposition_author}/info`}
                                      target={'_blank'}>
                                    {author}
                                </Link>
                            </span>
                        </div>
                        <div>
                            <span>
                            Комментарий
                            </span>: <span className={style.loc_item_value}>
                                {(initialData as IPropositionItem).proposition_description}
                            </span>
                        </div>
                    </div>}
                <FormConstructor onChange={this.onFormChange.bind(this)}
                                 initialData={initialData}
                                 schema={this.schema}/>
            </div>
            {
                showProposeComment
                && <ProposeCommentModal isWorking={isWorking}
                                        error={proposeError}
                                        isRequired={true}
                                        setComment={this.setComment.bind(this)}
                                        onClose={this.showComment.bind(this, false)}/>
            }
            {proposeAction && <Confirm error={proposeError}
                                       isWorking={isWorking}
                                       title={proposeAction}
                                       question={`Выполнить действие?`}
                                       accept={this.proposeAccept.bind(this)}
                                       onClose={this.showConfirm.bind(this, null)}/>
            }
            <div>
                {(initialData as ISettingsItem).propositions?.length
                    ? (initialData as ISettingsItem).propositions?.[0]
                        && <div>
                            <Link href={href}>Предложение</Link>
                                &nbsp;—&nbsp;
                            <Copy text={`${location.origin}${location.pathname}${href}`}>
                                Копировать ссылку на предложение
                            </Copy>
                        </div>
                    : ''}
                {showPropositions
                    && <Link href={`#/settings/global-vars/${initialData.setting_key}`}>
                        Базовое значение
                    </Link>}
            </div>
            {showPropositions
                ? <div className={coreStyle.button_container}>
                    <Button colorType={ButtonTypes.negative}
                            onClick={this.showConfirm.bind(this, ProposeAction.REJECT, null)}
                            isLoading={isWorking}>
                        Отклонить
                    </Button>
                    <Button colorType={ButtonTypes.positive}
                            onClick={this.showConfirm.bind(this, ProposeAction.CONFIRM, null)}
                            isLoading={isWorking}>
                        Принять
                    </Button>
                </div>
                : <div className={coreStyle.button_container}>
                    <CancelButton onClick={onClose.bind(this, false)}/>
                    <DeleteButton isLoading={isWorking} onClick={this.onDeleteClick.bind(this)}/>
                    <Button colorType={ButtonTypes.warning}
                            onClick={this.propose.bind(this)}
                            isLoading={isWorking}>
                        Предложить
                    </Button>
                    <SaveButton isLoading={isWorking} onClick={this.onSaveClick.bind(this)}/>
                </div>}
            {
                confirmRemoveVarModalIsOpen
                    ? <Confirm onClose={this.onConfirmClose.bind(this)}
                               error={error}
                               accept={accept}
                               isWorking={isWorking}
                               question={question}/>
                    : null
            }
            {success || proposeError
                ? <Window onClose={this.closeCopyModal.bind(this, proposeError)}
                          title={'Предложение GVars'}
                          error={proposeError}>
                    {success
                        ? <div>
                            <h3>Предложение успешно создано</h3>
                            <Copy text={`${location.origin}${location.pathname}#/settings/global-vars/${settingData?.setting_key}/proposition`}>
                                Копировать ссылку на предложение
                            </Copy>
                        </div>
                        : null
                    }
                </Window>
                : null
            }
            {
                confirmUpsertVarModalIsOpen
                    ? <Window title={'Изменение переменной'} onClose={this.onConfirmClose.bind(this)}>
                        <div>
                            <span>
                                Кажется, сейчас изменится ключ переменной {initialData?.setting_key}. Что сделать?
                            </span>
                            <div className={coreStyle.button_container}>
                                <CancelButton onClick={this.onConfirmClose.bind(this)}/>
                                <Button basic onClick={this.onReplace.bind(this)}>Заменить</Button>
                                <Button onClick={this.onCopy.bind(this)}>Копировать</Button>
                            </div>
                        </div>
                    </Window>
                    : null
            }
            {isWarningModalOpen
                ? <ProposeExistWarningModal onClose={() => this.setState({ isWarningModalOpen: false })}/>
                : null}
        </Window>;
    }
}
