import * as React from 'react';

import { ONE_SECOND } from '../../../../constants';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import { LabelStatus, TLabel } from '../../../../ui/Table';
import * as styleTable from '../../../../ui/Table/index.css';
import { Request2 } from '../../../../utils/request';
import Spin from '../../../Spin';
import { ISettingItem } from '../../../UISettings/component';
import { VARIABLE_SEPARATOR } from '../constants';
import { MAX_VALUE_LENGTH } from '../GlobalVarsTreeListItem/component';
import { GLOBAL_VARS_REQUESTS, REQUESTS } from '../request';
import * as style from './index.css';

interface IGVarsHistoryItem extends ISettingItem{
    history_action: string;
    history_timestamp: string;
    history_user_id: string;
}

interface IGVarsHistoryModalProps {
    onClose: () => void;
}

interface IGVarsHistoryModalState {
    history: IGVarsHistoryItem[];
    historyIsLoading: boolean;
    error: Error | null;
    isFullValueModalOpen: boolean;
    fullValueItem: IGVarsHistoryItem | null;
}

export class GVarsHistoryModal extends React.Component<IGVarsHistoryModalProps, IGVarsHistoryModalState> {
    state: IGVarsHistoryModalState = {
        history: [],
        historyIsLoading: false,
        error: null,
        isFullValueModalOpen: false,
        fullValueItem: null,
    };
    request = new Request2({ requestConfigs: GLOBAL_VARS_REQUESTS });

    componentDidMount(): void {
        this.getHistory();
    }

    getHistory() {
        this.setState({
            historyIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_SETTINGS_HISTORY)
                .then(response => {
                    this.setState({
                        history: response?.settings_history || [],
                        historyIsLoading: false,
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        historyIsLoading: false,
                    });
                });
        });
    }

    openFullValueModal(item) {
        this.setState({ fullValueItem: item, isFullValueModalOpen: true });
    }

    closeFullValueModal() {
        this.setState({ fullValueItem: null, isFullValueModalOpen: false });
    }

    render() {
        const { historyIsLoading, error, history } = this.state;
        const { onClose } = this.props;

        return <Window error={error} onClose={onClose.bind(null)} title={'История изменений'}>
            {historyIsLoading
                ? <Spin/>
                : <div className={style.gvars_history}>
                    <table className={styleTable.table}>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Дата</th>
                                <th>Ключ</th>
                                <th>Действие</th>
                                <th>Значение</th>
                                <th/>
                            </tr>
                        </thead>
                        <tbody>
                            {
                                history.map((item, index) => {
                                    const {
                                        setting_key,
                                        setting_value,
                                        history_action,
                                        history_timestamp,
                                        history_user_id,
                                    } = item;

                                    const path = setting_key.split(VARIABLE_SEPARATOR);

                                    const idField = <div>
                                        <span>{path.slice(0, path.length - 1).join(VARIABLE_SEPARATOR)}</span>
                                        {VARIABLE_SEPARATOR}
                                        <span className={style.id_field}>
                                            {path[path.length - 1]}
                                        </span>
                                    </div>;

                                    let actionStatus = LabelStatus.DEFAULT;
                                    switch (history_action) {
                                    case 'add':
                                        actionStatus = LabelStatus.POSITIVE;
                                        break;
                                    case 'remove':
                                        actionStatus = LabelStatus.NEGATIVE;
                                        break;
                                    case 'update_data':
                                        actionStatus = LabelStatus.INFO;
                                        break;
                                    }

                                    return <tr key={`${setting_key}_${history_timestamp}`}>
                                        <td>{index + 1}</td>
                                        <td>
                                            <FormatDate value={+history_timestamp * ONE_SECOND} withSecond={true}/>
                                        </td>
                                        <td>{idField}</td>
                                        <td>
                                            <TLabel status={actionStatus} text={history_action}/>
                                        </td>
                                        <td>
                                            <Link onClick={this.openFullValueModal.bind(this, item)}>
                                                {setting_value.length > MAX_VALUE_LENGTH
                                                    ? `${setting_value?.slice(0, MAX_VALUE_LENGTH)}...`
                                                    : setting_value}
                                            </Link>
                                        </td>
                                        <td>
                                            <Link href={`#/clients/${history_user_id}/info`}>Пользователь</Link>
                                        </td>
                                    </tr>;
                                })
                            }
                        </tbody>
                    </table>
                </div>}
            {this.state.isFullValueModalOpen
                ? <GVarsHistoryValueModal historyItem={this.state.fullValueItem}
                                          onClose={this.closeFullValueModal.bind(this)}/>
                : null}
        </Window>;
    }
}

interface IGVarsHistoryValueModalProps {
    historyItem: IGVarsHistoryItem | null;
    onClose: () => void;
}

const GVarsHistoryValueModal = React.memo((props: IGVarsHistoryValueModalProps) => {
    const { historyItem, onClose } = props;
    const { setting_key = '', setting_value = '' } = historyItem || {};

    return <Window title={`Значение переменной ${setting_key}`} onClose={onClose.bind(null)}>
        <pre>
            {setting_value}
        </pre>
    </Window>;
});
