import * as React from 'react';
import { Route, Switch } from 'react-router-dom';
import VirtualList from 'react-tiny-virtual-list';

import { Dict } from '../../../../types';
import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import AddGlobalVarModal from './AddGlobalVarModal';
import { LIST_ITEM_SIZE, VARIABLE_SEPARATOR } from './constants';
import { GlobalVarsTreeListItem } from './GlobalVarsTreeListItem/component';
import { GVarsHistoryModal } from './GVarsHistoryModal/component';
import * as style from './index.css';
import { GLOBAL_VARS_REQUESTS, REQUESTS } from './request';
import { IPropositionItem, ISettingsItem, ISettingsTree } from './types';

interface IGlobalVarsViewState {
    isLoading: boolean;
    filterValue: string;
    planeTree: ISettingsTree[];
    propositions: IPropositionItem[];
    globalSettings: ISettingsItem[];
    windowSize: Dict<number>;
    collapsedIds: Dict<boolean>;
    isOpenAddGlobalVarModal: boolean;
    isOpenGVarsHistoryModal: boolean;
    settingModalInitialData: null | ISettingsItem;
    error: Error | null;
    propositionsPlaneTree: ISettingsTree[];
    showPropositions: boolean;
    users: Record<string, any>;
}

export default class GlobalVarsView extends React.Component<{}, IGlobalVarsViewState> {
    state: IGlobalVarsViewState = {
        isLoading: false,
        filterValue: '',
        planeTree: [],
        propositions: [],
        globalSettings: [],
        windowSize: { width: window.innerWidth, height: window.innerHeight },
        collapsedIds: {},
        isOpenAddGlobalVarModal: false,
        isOpenGVarsHistoryModal: false,
        settingModalInitialData: null,
        error: null,
        propositionsPlaneTree: [],
        showPropositions: false,
        users: {},
    };
    listRef: React.RefObject<HTMLDivElement> | null = null;
    request = new Request2({ requestConfigs: GLOBAL_VARS_REQUESTS });

    constructor(props: {}) {
        super(props);
        this.listRef = React.createRef();
    }

    onResize() {
        this.setState({
            windowSize: { width: window.innerWidth, height: window.innerHeight },
        });
    }

    componentDidMount() {
        this.getGlobalSettings();
    }

    getGlobalSettings() {
        this.setState({ isLoading: true, error: null }, () => {
            this.request.exec(REQUESTS.GET_GLOBAL_SETTINGS)
                .then(response => {
                    const globalSettings: ISettingsItem[] = response?.settings || [];
                    const propositions: IPropositionItem[] = response?.propositions || [];

                    propositions.map((proposition) => {
                        const index = globalSettings.findIndex((el) => {
                            return el.setting_key === proposition.setting_key;
                        });

                        if (index !== -1) {
                            globalSettings[index] = {
                                ...globalSettings[index],
                                propositions: globalSettings[index].propositions
                                    ? [...globalSettings[index].propositions, proposition]
                                    : [proposition],
                            };
                        }
                    });

                    const users = response?.users || {};
                    let tree = this.getTree(globalSettings);
                    const planeTree = this.getPlaneTree(tree);
                    tree = this.getTree(propositions);
                    const propositionsPlaneTree = this.getPlaneTree(tree);
                    this.setState({
                        isLoading: false,
                        planeTree,
                        propositions,
                        globalSettings,
                        propositionsPlaneTree,
                        users,
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    getTree(settings: ISettingsItem[]): ISettingsTree[] {
        return settings.reduce((result: ISettingsTree[], curr) => {

            const fullPath = curr?.setting_key?.split?.(VARIABLE_SEPARATOR);
            const parentsPath = fullPath.slice(0, curr?.setting_key.split(VARIABLE_SEPARATOR).length - 1);

            //Go inside tree's nodes to find place for current item
            const currentItem = parentsPath.reduce((result: ISettingsTree[], pathItem, index) => {
                const key = fullPath.slice(0, index + 1).join(VARIABLE_SEPARATOR);
                const isEqualKey = (item: ISettingsTree) => {
                    return item.setting_key === key;
                };

                //if no needed item, add it
                if (!result.some(isEqualKey)) {
                    result.push({ setting_key: key, setting_value: null, children: [] });
                }

                const currentParent = result.filter(isEqualKey)[0];

                if (!currentParent.hasOwnProperty('children')) {
                    currentParent.children = [];
                }

                return currentParent.children || [];
            }, result);

            currentItem.push(curr);

            return result;
        }, []);
    }

    getPlaneTree(children: ISettingsTree[], nestingLevel = 0, parentsIds: string[] = []): ISettingsTree[] {
        return children.reduce((result: ISettingsTree[], child) => {
            child.meta = { nestingLevel, parentsIds };
            result.push(child);
            if (child.children) {
                result.push(...this.getPlaneTree(child.children, nestingLevel + 1,
                    [...parentsIds, child.setting_key]));
            }

            return result;
        }, []);
    }

    onFilterChange(filterValue: string) {
        this.setState({ filterValue });
    }

    openAddGlobalVarModal(settingModalInitialData: ISettingsItem, shouldOpen = true) {
        this.setState({
            settingModalInitialData: settingModalInitialData || null,
            isOpenAddGlobalVarModal: shouldOpen,
        });
    }

    closeAddGlobalVarModal(isSuccessful: boolean) {
        this.setState({ settingModalInitialData: null, isOpenAddGlobalVarModal: false }, () => {
            if (isSuccessful) {
                this.getGlobalSettings();
            }
        });

        location.hash = '/settings/global-vars/';
    }

    unCollapseAll() {
        this.setState({ collapsedIds: {} });
    }

    collapseAll() {
        const { showPropositions, planeTree, propositionsPlaneTree } = this.state;
        const collapsedIds = (showPropositions ? propositionsPlaneTree : planeTree)
            ?.reduce((result: Dict<boolean>, planeTreeItem) => {
                result[planeTreeItem.setting_key] = true;

                return result;
            }, {});
        this.setState({ collapsedIds });
    }

    getListHeight() {
        const topPosition = this.listRef?.current?.offsetTop || 0;
        const BOTTOM_MARGIN = 35;

        return this.state.windowSize.height - topPosition - BOTTOM_MARGIN;
    }

    onItemClick(item_id: string) {
        const collapsedIds = Object.assign({}, this.state.collapsedIds);
        collapsedIds[item_id] = !collapsedIds[item_id];

        this.setState({ collapsedIds });
    }

    getFilteredTree(): ISettingsTree[] {
        const { showPropositions, planeTree: settingsPlaneTree, propositionsPlaneTree } = this.state;
        let planeTree = showPropositions ? propositionsPlaneTree : settingsPlaneTree;

        let nodesFilteredById: string[] | null = this.state.filterValue ? [] : null;
        const filterValue = this.state.filterValue?.toLowerCase() || '';

        planeTree.forEach((node: ISettingsTree) => {
            if (filterValue) {
                const filter = filterValue.toLowerCase();
                const includesKey = node.setting_key.toLowerCase().includes(filter);
                const includesValue = node.setting_value ? node.setting_value.toLowerCase().includes(filter) : null;

                if (includesKey || includesValue) {
                    nodesFilteredById && nodesFilteredById.push(node.setting_key, ...(node?.meta?.parentsIds ?? []));
                }
            }
        });
        nodesFilteredById = nodesFilteredById
            ? Array.from(new Set(nodesFilteredById))
            : null;

        planeTree = planeTree.reduce((result: ISettingsTree[], node) => {

            const isCollapsed = node?.meta?.parentsIds?.some(parentsId => this.state.collapsedIds[parentsId]);
            const isFilteredById = nodesFilteredById && !showPropositions
                ? node.setting_key && nodesFilteredById.includes(node.setting_key)
                : true;

            if (!isCollapsed && isFilteredById) {
                result.push(node);
            }

            return result;
        }, []);

        return planeTree;
    }

    openGVarsHistoryModal() {
        this.setState({ isOpenGVarsHistoryModal: true });
    }

    closeGVarsHistoryModal() {
        this.setState({ isOpenGVarsHistoryModal: false });
    }

    onPropositionsChange(showPropositions: boolean) {
        this.setState({ showPropositions });
    }

    render() {
        const {
            isLoading, isOpenGVarsHistoryModal, showPropositions, filterValue, collapsedIds,
            isOpenAddGlobalVarModal, settingModalInitialData, error, users, propositions, globalSettings,
        } = this.state;
        const planeTree = this.getFilteredTree();
        const listHeight = +Math.floor(this.getListHeight());

        if (error) {
            return <SimpleError error={error}
                                data={{ label: 'Ошибка при загрузке глобальных переменных' }}/>;
        }

        return <div className={style.global_vars_view}>
            <div className={style.controls}>
                <div className={style.filters}>
                    <Input onChange={this.onFilterChange.bind(this)}
                           value={filterValue}
                           className={style.filter_input}
                           placeholder={'Фильтр'}
                           disabled={showPropositions}/>
                    <Button basic
                            onClick={this.openAddGlobalVarModal.bind(this, {})}
                            disabled={showPropositions}>Добавить</Button>
                    <Link className={style.open_history_link} onClick={this.openGVarsHistoryModal.bind(this)}>
                        История изменений
                    </Link>
                    <Checkbox checked={showPropositions}
                              className={style.propositions_checkbox}
                              onChange={this.onPropositionsChange.bind(this)}/>
                    <span>Показать предложения</span>
                </div>
                <div className={style.collapse_controls}>
                    <Link onClick={this.unCollapseAll.bind(this)} className={style.collapse_control}>
                        Разложить все
                    </Link>
                    <Link onClick={this.collapseAll.bind(this)} className={style.collapse_control}>
                        Сложить все
                    </Link>
                </div>
            </div>
            <div ref={this.listRef}>
                {isLoading
                    ? <Spin/>
                    :
                    <VirtualList height={listHeight}
                                 itemCount={planeTree.length}
                                 itemSize={LIST_ITEM_SIZE}
                                 renderItem={({ index, style }) => {
                                     return <GlobalVarsTreeListItem key={index}
                                                                    _style={style}
                                                                    onItemClick={this.onItemClick.bind(this)}
                                                                    openAddGlobalVarModal={this.openAddGlobalVarModal
                                                                        .bind(this)}
                                                                    filterValue={filterValue}
                                                                    index={index}
                                                                    collapsedIds={collapsedIds}
                                                                    item={planeTree[index]}
                                                                    showPropositions={showPropositions}/>;
                                 }}/>
                }
            </div>
            {isOpenAddGlobalVarModal
                ? <AddGlobalVarModal initialData={settingModalInitialData}
                                     onClose={this.closeAddGlobalVarModal.bind(this)}
                                     getSettings={this.getGlobalSettings.bind(this)}
                                     users={users}/>
                : null}
            {isOpenGVarsHistoryModal
                ? <GVarsHistoryModal onClose={this.closeGVarsHistoryModal.bind(this)}/>
                : null}
            <Switch>
                <Route path="/settings/global-vars/:setting_key/:proposition?"
                       render={() => <AddGlobalVarModal initialData={settingModalInitialData}
                                                        onClose={this.closeAddGlobalVarModal.bind(this)}
                                                        getSettings={this.getGlobalSettings.bind(this)}
                                                        propositions={propositions}
                                                        globalSettings={globalSettings}
                                                        users={users}/>}/>
            </Switch>
        </div>;
    }
}
