import * as React from 'react';

import { Dict } from '../../../../../types';
import { isValidJSONString } from '../../../../utils/isValidJSONString';
import { Request2 } from '../../../../utils/request';
import { ITag } from '../../../TagModal/component';
import { IChat } from '../../ChatsEditor/types';
import { USER_LANDING_TAG_TYPE } from '../../Landings/constants';
import { ILanding } from '../../Landings/types';
import { REQUESTS, SETTINGS_REQUESTS } from '../requests';

interface IISDecoratorState {
    dataIsLoading: boolean;
    savingIsLoading: boolean;
    landings: ILanding[];
    propositions: ILanding[];
    chats: IChat[];
    landingsLoadingError: Error | null;
    userLandingTags: Dict<any>;
    parentChat: Dict<any>;
    ISError: Error | null;
    error: Error | null;
}

export interface IISDecoratorExport {
    getFullData?: () => void;
    clearErrors?: () => void;
    saveIntroscreen?: (chat, landing) => void;
    dataIsLoading?: boolean;
    savingIsLoading?: boolean;
    landings?: ILanding[];
    propositions?: ILanding[];
    chats?: IChat[];
    landingsLoadingError?: Error | null;
    userLandingTags?: Dict<any>;
    ISError?: Error | null;
    error?: Error | null;
}

export const ISDecorator = () => (Component): any => {
    return class extends React.Component {
        state: IISDecoratorState = {
            dataIsLoading: false,
            savingIsLoading: false,
            landings: [],
            propositions: [],
            chats: [],
            landingsLoadingError: null,
            userLandingTags: {},
            parentChat: {},

            ISError: null,
            error: null,
        };
        request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

        componentWillUnmount() {
            this.request.abort();
        }

        getFullData(selectedLandingId?: string) {
            this.setState({
                dataIsLoading: true,
                landingsLoadingError: null,
            }, () => {
                Promise.all([
                    this.request.exec(REQUESTS.GET_NOTIFIERS),
                    this.request.exec(REQUESTS.GET_LANDINGS),
                    this.request.exec(REQUESTS.GET_TAGS),
                ])
                    .then(response => {
                        let landings: ILanding[] = response?.find(el => el.landings)?.landings ?? [];
                        const propositions: ILanding[] = response?.find(el => el.propositions)?.propositions ?? [];
                        landings = landings.concat(propositions);
                        const notifiers: IChat[] = response?.find(el => el.objects)?.objects ?? [];
                        const chats: IChat[] = notifiers?.filter(notifier => notifier.type === 'native_chat') ?? [];
                        const parentChat = chats?.find(el => {
                            return el.meta?.chat_id === 'static_announcements';
                        });
                        const TAGS_RESPONSE_INDEX = 2;
                        const tags: ITag[] = response?.[TAGS_RESPONSE_INDEX]?.records ?? [];

                        const userLandingTagsArray = tags.filter(tag => {
                            return tag.type === USER_LANDING_TAG_TYPE;
                        });

                        const userLandingTags = userLandingTagsArray.reduce((result: Dict<ITag[]>, userLandingTag) => {
                            const meta: any = userLandingTag?.meta ?? {};
                            userLandingTag.meta = isValidJSONString(meta)
                                ? JSON.parse(meta)
                                : typeof(meta) === 'object' ? meta : {};
                            const objectMeta: any = userLandingTag.meta;
                            const landing_id: string = objectMeta.landing_id;

                            if (result?.[landing_id]) {
                                result[landing_id].push(userLandingTag);
                            } else {
                                result[landing_id] = [userLandingTag];
                            }

                            return result;
                        }, {});

                        if (selectedLandingId) {
                            const selectedLanding = landings
                                .filter(landing => landing.landing_id === selectedLandingId)?.[0] ?? null;

                            this.setState({
                                dataIsLoading: false,
                                landings,
                                propositions,
                                selectedLanding,
                                chats,
                                landingsLoadingError: null,
                                userLandingTags,
                                parentChat,
                            });
                        } else {
                            this.setState({
                                dataIsLoading: false,
                                landings,
                                propositions,
                                chats,
                                userLandingTags,
                                parentChat,
                            });
                        }
                    })
                    .catch(landingsLoadingError => {
                        this.setState({ landingsLoadingError, dataIsLoading: false });
                    });
            });
        }

        saveIntroscreen(chat, landing) {
            if (this.state.parentChat?.meta?.chat_script?.items) {
                const parentChat = this.state.parentChat;
                const node = this.makeFullNode(chat);

                const currentItemIndex = parentChat.meta.chat_script.items?.findIndex(el => {
                    return el.id === chat.id;
                });

                if (currentItemIndex !== -1) {
                    parentChat.meta.chat_script.items.splice(currentItemIndex, 1);
                }

                parentChat.meta.chat_script.items.push(node);

                this.updateScreens({ objects: [parentChat] }, landing, chat);
            } else {
                this.setState({
                    ISError: { message: 'Не удалось найти родительский чат' },
                });
            }
        }

        updateScreens(data, landingData, chat) {
            this.setState({ savingIsLoading: true }, () => {
                this.request.exec(REQUESTS.UPSERT_NOTIFIERS, { body: data })
                    .then(() => {
                        this.setState({ setISsuccess: true }, () => {
                            const landing = this.makeFullLanding(landingData, chat);
                            this.updateLandings(landing);
                        });
                    })
                    .catch(error => {
                        this.setState({ error, savingIsLoading: false });
                    });
            });
        }

        updateLandings(data) {
            this.request.exec(REQUESTS.UPSERT_LANDING, { body: data })
                .then(() => {
                    this.setState({ setLandingSuccess: true, savingIsLoading: false }, () => {
                        this.getFullData();
                    });
                })
                .catch(error => {
                    this.setState({ error, savingIsLoading: false });
                });
        }

        makeFullNode(chat) {
            const node = chat;
            node.put_tag_on_entry = false;
            node.suppress_support_call = false;
            node.use_classifier = false;
            node.next_step = null;

            return node;
        }

        makeFullLanding(landing, chat) {
            const {
                landing_chat_deadline,
                landing_chat_enabled,
                landing_enabled,
                landing_chat_title,
                landing_deadline,
                landing_preview,
                landing_priority,
                landing_timestamp_override,
                android_min_app_build,
                ios_min_app_build,
                landing_geo_tags,
                landing_id,
            } = landing;

            return {
                landing_auditory_condition: {
                    'type': 'or',
                    'subconditions': [
                        {
                            'type': 'and',
                            'subconditions': [
                                {
                                    'name': 'android_client',
                                    'type': 'platform',
                                },
                                {
                                    'min_value': android_min_app_build,
                                    'type': 'min_app_build',
                                },
                            ],
                        },
                        {
                            'type': 'and',
                            'subconditions': [
                                {
                                    'name': 'ios',
                                    'type': 'platform',
                                },
                                {
                                    'min_value': ios_min_app_build,
                                    'type': 'min_app_build',
                                },
                            ],
                        },
                    ],
                },
                landing_chat_id: 'static_announcements',
                landing_chat_messages_group: landing_id,
                landing_json: {},
                landing_json_substitute: true,
                patch_payload: {},
                revision: 0,
                landing_geo_tags: Array.isArray(landing_geo_tags) ? landing_geo_tags?.join() : landing_geo_tags,
                landing_id: landing_id ?? chat.id,
                landing_chat_deadline,
                landing_enabled,
                landing_chat_enabled,
                landing_chat_title,
                landing_deadline,
                landing_preview,
                landing_priority,
                landing_timestamp_override,
            };
        }

        clearErrors() {
            this.setState({
                landingsLoadingError: null,
                ISError: null,
                error: null,
            });
        }

        render() {
            const {
                dataIsLoading,
                landings,
                propositions,
                chats,
                landingsLoadingError,
                userLandingTags,
                ISError,
                error,
                savingIsLoading,
            } = this.state;

            return <Component {...this.props}
                              getFullData={this.getFullData.bind(this)}
                              dataIsLoading={dataIsLoading}
                              savingIsLoading={savingIsLoading}
                              landings={landings}
                              propositions={propositions}
                              chats={chats}
                              landingsLoadingError={landingsLoadingError}
                              userLandingTags={userLandingTags}
                              saveIntroscreen={this.saveIntroscreen.bind(this)}
                              ISError={ISError}
                              error={error}
                              clearErrors={this.clearErrors.bind(this)}/>;
        }
    };
};
