import React from 'react';

import { Button, ButtonTypes, SaveButton } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import { SimpleError } from '../../../SimpleError';
import { ChatIntroScreenPreview } from '../../ChatsEditor/ChatIntroscreenPreview/component';
import { IChat, INode } from '../../ChatsEditor/types';
import LandingsList from '../../Landings/LandingsList/component';
import { PreviewPreview } from '../../Landings/PreviewPreview/component';
import { ILanding } from '../../Landings/types';
import { IISDecoratorExport, ISDecorator } from '../IntroscreenDecorator/ISDecorator';
import { InfoModal } from '../IntroscreenModals/InfoModal';
import { TestingModal } from '../IntroscreenModals/TestingModal';
import { SETTINGS_REQUESTS } from '../requests';
import { SimplifiedForms } from '../SimplifiedForms/SimplifiedForms';
import style from '../style.css';

type IIntroscreenTestProps = IISDecoratorExport

interface IIntroscreenTestState {
    selectedLanding: Record<string, any>;
    selectedChat: Record<string, any>;

    initialLanding: Record<string, any>;
    initialChat: Record<string, any>;

    isTestModalOpen: boolean;
    isInfoModalOpen: boolean;
}

@ISDecorator()
export class IntroscreenTest extends React.Component<IIntroscreenTestProps, IIntroscreenTestState> {
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    state: IIntroscreenTestState = {
        selectedLanding: {},
        selectedChat: {},

        initialLanding: {},
        initialChat: {},

        isTestModalOpen: false,
        isInfoModalOpen: false,
    };

    componentDidMount() {
        this.props.getFullData?.();
    }

    setContentFormValues(selectedChat) {
        if (!isObjectEqual(selectedChat, this.state.selectedChat)) {
            this.setState({ selectedChat });
        }
    }

    setOptionsFormValues(selectedLanding) {
        if (!isObjectEqual(selectedLanding, this.state.selectedLanding)) {
            this.setState({ selectedLanding });
        }
    }

    setTestModalOpen(isTestModalOpen) {
        this.setState({ isTestModalOpen });
    }

    setInfoModalOpen(isInfoModalOpen) {
        this.setState({ isInfoModalOpen });
    }

    onSelectItem(landing) {
        const PARENT_CHAT_ID = 'static_announcements';
        const PARENT_CHAT = this.props.chats?.find(el => {
            return el.meta?.chat_id === PARENT_CHAT_ID;
        });

        const CHAT = PARENT_CHAT?.meta?.chat_script?.items?.find(el => {
            return el.id === landing.landing_id;
        }) ?? {};

        this.setState({
            selectedChat: CHAT,
            selectedLanding: landing,
            initialLanding: landing,
            initialChat: CHAT,
        });
    }

    clearInitial() {
        this.setState({
            selectedLanding: {},
            selectedChat: {},
            initialChat: {},
            initialLanding: {},
        });
    }

    render() {
        const {
            landings = [],
            chats = [],
            userLandingTags = {},
            dataIsLoading,
            getFullData,
            error,
            ISError,
            landingsLoadingError,
            savingIsLoading,
        } = this.props;

        const {
            selectedLanding,
            selectedChat,
            initialLanding,
            initialChat,
            isTestModalOpen,
            isInfoModalOpen,
        } = this.state;

        return <div className={style.landings}>
            <div className={style.landing_list}>
                <LandingsList landings={landings}
                              chats={chats ?? []}
                              userLandingTags={userLandingTags}
                              dataIsLoading={Boolean(dataIsLoading)}
                              getData={getFullData?.bind(this)}
                              onClear={this.clearInitial.bind(this)}
                              onLandingListItemSelect={this.onSelectItem.bind(this)}
                              selectedLanding={selectedLanding as ILanding}/>
            </div>

            <div className={style.form}>
                <SimplifiedForms setContentFormValues={this.setContentFormValues.bind(this)}
                                 setOptionsFormValues={this.setOptionsFormValues.bind(this)}
                                 landing={initialLanding}
                                 chat={initialChat}/>

                <div className={style.buttons}>
                    <SaveButton onClick={this.props.saveIntroscreen?.bind(this, selectedChat, selectedLanding)}
                                isLoading={savingIsLoading}/>

                    <Button onClick={this.setTestModalOpen.bind(this, true)}>
                        Протестировать
                    </Button>

                    <Button onClick={this.setInfoModalOpen.bind(this, true)} basic={true}>
                        Подсказки
                    </Button>

                    <Button onClick={this.clearInitial.bind(this)} colorType={ButtonTypes.warning} basic={true}>
                        Очистить
                    </Button>
                </div>
            </div>

            <div className={style.preview}>
                <ChatIntroScreenPreview chat={{} as IChat} node={selectedChat as INode}/>
                <PreviewPreview landing={selectedLanding as ILanding}/>
            </div>

            {isTestModalOpen
                ? <TestingModal id={selectedChat.id}
                                onClose={this.setTestModalOpen.bind(this, false)}
                                landingInfo={selectedLanding as ILanding}/>
                : null
            }

            {isInfoModalOpen
                ? <InfoModal onClose={this.setInfoModalOpen.bind(this, false)}/>
                : null
            }

            {error || ISError || landingsLoadingError
                ? <Window onClose={this.props.clearErrors?.bind(this)}>
                    <SimpleError error={error || ISError || landingsLoadingError}/>
                </Window>
                : null
            }
        </div>;
    }
}
