import moment from 'moment';
import React from 'react';

import { Dict } from '../../../../types';
import { CITIES, ONE_SECOND } from '../../../constants';
import DatePicker from '../../../ui/DatePicker';
import { NoInformation } from '../../../ui/NoInformation';
import Select, { IOptionInfo } from '../../../ui/Select';
import { TabItem, Tabs } from '../../../ui/Tabs';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { INTROSCREENS_TYPES } from '../Landings/LandingsList/component';
import { LANDINGS_REQUESTS, REQUESTS } from '../Landings/request';
import { ILanding } from '../Landings/types';
import * as style from './index.css';

interface IIntroScreensDeadlineProps {
}

interface IIntroScreensDeadlineState {
    isDataLoading: boolean;
    error: Error | null;
    data: ILanding[];
    sortedData: ILanding[];
    chartsSize: Dict<any>;
    maxValue: number;
    points: number[];
    selectOptions: IOptionInfo[];
    currentCity: string;
    currentTab: string;
}

const POINTS_COUNT = 9;
const DEMICAL_PLACE = 3;
const PERCENTS = 100;
const MIN_DATE_WIDTH = 0.05;
const ADDITIONAL_COLUMNS = 2;
const MARGIN = 290;
const ID_WIDTH = `16vw`;
const ALL_OPTIONS = 'all';
const NO_CITIES = 'no_cities';
let intervals = POINTS_COUNT;

export default class IntroScreensDeadline extends
    React.Component<IIntroScreensDeadlineProps, IIntroScreensDeadlineState> {
    state: IIntroScreensDeadlineState = {
        isDataLoading: false,
        error: null,
        data: [],
        sortedData: [],
        chartsSize: {},
        maxValue: 0,
        points: [],
        selectOptions: [],
        currentCity: ALL_OPTIONS,
        currentTab: INTROSCREENS_TYPES.all,
    };
    request = new Request2({ requestConfigs: LANDINGS_REQUESTS });
    timeNow = +new Date() / ONE_SECOND;
    tabs: TabItem[] = Object.values(INTROSCREENS_TYPES)?.map(option => {
        return { name: option, link: option };
    });

    componentDidMount(): void {
        this.getData();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getData() {
        this.setState({ isDataLoading: true, error: null }, () => {
            this.request.exec(REQUESTS.GET_LANDINGS)
                .then(response => {
                    const landings = response?.landings || [];
                    let maxValue = 0;
                    const relevantLandings: ILanding[] = [];

                    landings.forEach(landing => {
                        if (landing?.landing_deadline && landing.landing_deadline > this.timeNow) {
                            relevantLandings.push(landing);
                            maxValue = Math.max(landing.landing_deadline, maxValue);
                        }
                    });

                    const selectOptions = this.generateSelectOptions();

                    this.setState({
                        isDataLoading: false,
                        data: relevantLandings,
                        sortedData: relevantLandings,
                        selectOptions,
                        maxValue,
                    }, () => this.sortLandings());
                })
                .catch(error => {
                    this.setState({ error, isDataLoading: false });
                });
        });
    }

    getChartData(data: ILanding[], maxValue?: number): Dict<any> {
        const maxDate = maxValue ?? this.state.maxValue;
        const interval = (maxDate - this.timeNow) / (intervals - 1);
        const points = this.generatePoints(interval);
        const chartsSize = this.generateChartsSize(data, interval);

        return { chartsSize, points };
    }

    generateSelectOptions() {
        const selectOptions = Object.entries(CITIES).map(opt => {
            return { value: opt[0], text: opt[1]?.name };
        });
        selectOptions.push({ value: ALL_OPTIONS, text: 'Все города' }, { value: NO_CITIES, text: 'Не определены' });

        return selectOptions;
    }

    generatePoints(interval: number): number[] {
        const points: any[] = [];
        for (let i = 1; i < intervals; i++) {
            points.push((this.timeNow + interval * i) * ONE_SECOND);
        }

        // first item will be empty (id row), second item is always today
        points.unshift(0, this.timeNow * ONE_SECOND);

        return points;
    }

    generateChartsSize(data: ILanding[], interval: number): Dict<any> {
        const chartsSize = {};

        data?.forEach(item => {
            const date = item?.landing_deadline || 0;
            const size = ((date - this.timeNow) / (interval * (intervals + 1))).toFixed(DEMICAL_PLACE);
            chartsSize[item?.landing_id] = { size, date };
        });

        return chartsSize;
    }

    onSelectChange(currentCity: string) {
        this.setState({ currentCity }, () => {
            this.sortLandings();
        });
    }

    onTabSelect(currentTab: string) {
        this.setState({ currentTab }, () => {
            this.sortLandings();
        });
    }

    sortLandings() {
        const { currentTab, currentCity } = this.state;
        const allData = this.state.data?.slice() || [];
        let sortedData: ILanding[] = [];

        if (currentCity === ALL_OPTIONS) {
            sortedData = allData;
        } else if (currentCity === NO_CITIES) {
            allData.forEach(el => {
                if (!el.landing_geo_tags) {
                    sortedData.push(el);
                }
            });
        } else if (currentCity !== NO_CITIES) {
            allData.forEach(el => {
                if (el.landing_geo_tags && el.landing_geo_tags.split(',').includes(currentCity)) {
                    sortedData.push(el);
                }
            });
        }

        if (currentTab === INTROSCREENS_TYPES.warnings) {
            sortedData = sortedData.filter(item => !item.landing_chat_id && Object.keys(item.landing_json)?.length);
        } else if (currentTab === INTROSCREENS_TYPES.introscreens) {
            sortedData = sortedData.filter(item => item.landing_chat_id);
        }

        const { chartsSize, points } = this.getChartData(sortedData);

        this.setState({ chartsSize, points, sortedData });
    }

    onDateChange(newDate) {
        const maxDate = +moment().add(POINTS_COUNT, 'days');

        if (newDate >= maxDate) {
            intervals = POINTS_COUNT;
            const { chartsSize, points } = this.getChartData(this.state.sortedData.slice(), newDate / ONE_SECOND);

            this.setState({
                maxValue: newDate / ONE_SECOND,
                chartsSize,
                points,
            });
        } else {
            const daysCount = newDate.diff(moment(), 'days');

            if (daysCount > 0) {
                intervals = daysCount + ADDITIONAL_COLUMNS;
                const { chartsSize, points } = this.getChartData(this.state.sortedData.slice(), newDate / ONE_SECOND);

                this.setState({
                    maxValue: newDate / ONE_SECOND,
                    chartsSize,
                    points,
                });
            }
        }
    }

    render() {
        const { error, isDataLoading, selectOptions, maxValue, currentTab, chartsSize, points } = this.state;
        const initialOption = selectOptions.find(el => el.value === ALL_OPTIONS);

        return <div className={style.component}>
            {error
                ? <SimpleError error={error}/>
                : isDataLoading
                    ? <Spin/>
                    : <>
                        <div className={style.controls}>
                            <Select options={selectOptions}
                                    className={style.select}
                                    initialValues={[initialOption?.value || '']}
                                    onSelect={this.onSelectChange.bind(this)}
                                    placeholder={'Город'}/>

                            <DatePicker value={maxValue * ONE_SECOND}
                                        placeholder={'Предельная дата'}
                                        onChange={this.onDateChange.bind(this)}
                                        className={style.datepicker}/>
                        </div>
                        <Tabs tabs={this.tabs}
                              currentTab={currentTab}
                              selectTab={this.onTabSelect.bind(this)}/>

                        <DeadlineChart chartsSize={chartsSize}
                                       points={points}
                                       maxDate={maxValue}/>
                    </>

            }
        </div>;
    }

}

interface IDeadlineChartProps {
    chartsSize: Dict<any>;
    points: number[];
    maxDate: number;
}

const DeadlineChart = React.memo((props: IDeadlineChartProps) => {
    const { chartsSize, points, maxDate } = props;
    const sizesArray = Object.entries(chartsSize);

    return <div className={style.chart_container} style={{ height: window.innerHeight - MARGIN }}>
        {sizesArray.length && points.length
            ? <div className={style.main_block}>
                <table className={style.table}>
                    <thead>
                        <tr>
                            {points?.length
                                ? points.map((p, i) => {
                                    return <td className={style.time_header} key={i}>
                                        {i === 0
                                            ? null
                                            : <>
                                                <span className={style.header_title}>
                                                    {moment(+p).format('DD.MM.YY')}
                                                </span>
                                                <div className={style.border}/>
                                            </>
                                        }
                                    </td>;
                                })
                                : null
                            }
                        </tr>
                    </thead>
                    <tbody>
                        {sizesArray?.length
                            ? sizesArray.map(chart => {
                                const chartWidth = `${chart[1].size * PERCENTS * (intervals + 1)}%`;
                                const date = moment(chart[1].date * ONE_SECOND).format('DD.MM.YY');
                                const chartStyle = `${style.chart} ${chart[1].date > maxDate ? style.overChart : ''}`;

                                return <tr className={style.table_row} key={chart[0]}>
                                    <td className={style.screen_id}
                                        style={{ 'width': ID_WIDTH }}>{chart[0]}
                                    </td>
                                    <td className={style.screen_chart}>
                                        <div className={chartStyle} style={{ width: chartWidth }}>
                                            <span className={chart[1].size > MIN_DATE_WIDTH
                                                ? style.date_text
                                                : style.after_text}
                                                  data-date={date}>
                                                {chart[1].size > MIN_DATE_WIDTH ? date : '|'}
                                            </span>
                                        </div>
                                    </td>
                                </tr>;

                            })
                            : null
                        }
                    </tbody>
                </table>
            </div>
            : <NoInformation/>
        }
    </div>;
});
