import React, { useEffect, useState } from 'react';

import { useRequestHandler } from '../../../../hooks/useRequestHandler';
import { Button, CancelButton } from '../../../../ui/Button';
import { Confirm, Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import { Request2 } from '../../../../utils/request';
import { Copy } from '../../../Copy';
import Spin from '../../../Spin';
import { LANDINGS_REQUESTS, REQUESTS } from '../request';

interface IAddActionButtonProps {
    landing: string;
    disabled: boolean;
    setExtraAction: (id: string) => void;
}

export const AddActionButton = React.memo((props: IAddActionButtonProps) => {
    const { landing, disabled } = props;

    const [isCreateActionModalOpen, openCreateActionModal] = useState<boolean>(false);

    const ACTION_TYPE = 'simple';

    return <>
        <Button onClick={openCreateActionModal.bind(null, true)}
                disabled={disabled}
                basic>
            Создать action
        </Button>

        {isCreateActionModalOpen
            ? <AddActionModal onClose={openCreateActionModal.bind(null, false)}
                              landing={[landing]}
                              action_id={`${landing}_screen`}
                              setExtraAction={props.setExtraAction.bind(null)}
                              action_type={ACTION_TYPE}
                              deprecated={false}
                              enabled={true}/>
            : null
        }
    </>;
});

interface IAddActionModalProps {
    onClose: () => void;
    landing: string[];
    action_id: string;
    action_type: string;
    deprecated: boolean;
    enabled: boolean;
    setExtraAction: (id: string) => void;
}

export const AddActionModal = React.memo((props: IAddActionModalProps) => {
    const { onClose, landing, action_id, action_type, deprecated, enabled } = props;

    const [isConfirmed, confirm] = useState<boolean>(false);

    const request = new Request2({ requestConfigs: LANDINGS_REQUESTS });
    const requestOptions = React.useMemo(() => {
        return {
            requestName: REQUESTS.ADD_ACTION,
            requestOptions: {
                body: {
                    action_id,
                    action_type,
                    deprecated,
                    enabled,
                    action_meta: { landing },
                },
            },
        };
    }, [landing[0]]);

    const [isLoading, , error, makeRequest] = useRequestHandler(request, requestOptions);

    useEffect(() => {
        if (isConfirmed) {
            makeRequest();

            props.setExtraAction(action_id);
        }
    }, [isConfirmed]);

    const href = `#/settings/actions/${action_id}`;

    return !isConfirmed
        ? <Confirm question={'Создать action?'}
                   error={null}
                   accept={confirm.bind(null, true)}
                   onClose={onClose.bind(null)}/>

        : <Window onClose={onClose.bind(null)} error={error}>
            {!error
                ? <div>
                    {isLoading
                        ? <Spin />
                        : <span>
                            <div>Готово!</div>

                            <div>
                                Экшен <Copy text={action_id}> {action_id} </Copy> создан
                            </div>

                            <div>
                                <Link href={href} target={'_blank'}>Перейти </Link>
                            </div>
                        </span>
                    }
                </div>
                : null
            }

            <div className={coreStyle.button_container}>
                <CancelButton onClick={onClose.bind(null)}/>
            </div>
        </Window>;
});
