import * as React from 'react';

import { Dict } from '../../../../../types';
import { EMPTY_DATA } from '../../../../constants';
import { UserInfoHandler } from '../../../../models/user';
import { Button, CancelButton } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import Select, { IOptionInfo } from '../../../../ui/Select';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import { FormConstructorSchemaWorker } from '../../../FormConstructor/FormConstructorSchemaWorker';
import { FormConstructorValuesWorker } from '../../../FormConstructor/FormConstructorValuesWorker';
import { controlType, ISchemaItem } from '../../../FormConstructor/types';
import Spin from '../../../Spin';
import TagEditModal from '../../Tags/TagEditModal';
import { AddActionButton } from '../AddActionButton/AddActionButton';
import { USER_LANDING_TAG_TYPE } from '../constants';
import * as style from '../index.css';
import { LANDINGS_REQUESTS, REQUESTS as LANDING_REQUESTS } from '../request';
import { ILanding } from '../types';

const ADD_INTRO_SCREEN_MODAL_DESCRIPTION = `Здесь можно навесить тег, к которому привязан интроскрин, `
    + `на своих связанных пользователей. Навешивание вызовет очистку истории показов интроскрина`;

const DRIVE_TEAM_USERS_PAGE_SIZE = 100;
const SCREEN_TAG_TYPE_PREFIX = '_screen';

export interface IConstantsFaceTag {
    constantsIFaceTag: Promise<{ iface_tag_implementations: Dict<any> }>;
}

interface IAddIntroScreenToUserModalProps extends IConstantsFaceTag {
    userLandingTags: Dict<any>;
    onClose: () => void;
    updateData: () => void;
    clearHistoryInitialFormData: Dict<any>;
    landings: ILanding[];
}

interface IAddIntroScreenToUserModalState {
    isTagModalOpen: boolean;
    selectedTagName: string | null;

    clearHistoryFormData: Dict<any>;
    clearHistorySchema: Dict<ISchemaItem>;
    clearHistoryError: Error | null;

    attachingError: Error | null;

    isHistoryClearing: boolean;
    isTagAttaching: boolean;

    schemaLoadingError: Error | null;

    isActionLoading: boolean;
    actionError: Error | null;
    actions: any[];
    actualActions: any[];
}

export default class AddIntroScreenToUserModal extends
    React.Component<IAddIntroScreenToUserModalProps, IAddIntroScreenToUserModalState> {
    state = {
        isTagModalOpen: false,
        selectedTagName: this.props.clearHistoryInitialFormData?.id ?? null,

        clearHistoryFormData: {},
        clearHistorySchema: {},
        clearHistoryError: null,

        attachingError: null,

        isHistoryClearing: false,
        isTagAttaching: false,

        schemaLoadingError: null,

        isActionLoading: false,
        actionError: null,
        actions: [],
        actualActions: [],
    };
    requestLandings = new Request2({ requestConfigs: LANDINGS_REQUESTS });

    componentDidMount(): void {
        this.getSchema();
        this.getActions();
    }

    componentDidUpdate(
        prevProps: Readonly<IAddIntroScreenToUserModalProps>,
        prevState: Readonly<IAddIntroScreenToUserModalState>,
        snapshot?: any,
    ): void {
        if (this.state.clearHistoryFormData !== prevState.clearHistoryFormData) {
            const clearHistoryFormData: Dict<any> = this.state.clearHistoryFormData;

            if (clearHistoryFormData?.id) {
                const { userLandingTags } = this.props;
                const introScreenTags = clearHistoryFormData && userLandingTags?.[clearHistoryFormData.id] || null;
                const selectedTagName = introScreenTags?.[0]?.name;
                this.setState({ selectedTagName });
            }
        }

        if (this.state.selectedTagName !== prevState.selectedTagName) {
            this.setActualActions();
        }
    }

    getSchema() {
        const { landings } = this.props;

        this.requestLandings.exec(LANDING_REQUESTS.GET_USERS_BY_ROLE, {
            queryParams: {
                roles: 'drive_team_user',
                page_size: DRIVE_TEAM_USERS_PAGE_SIZE,
            },
        })
            .then(response => {
                const users = response?.report ?? [];

                const userIdOptions = users.map(user => {
                    return { text: UserInfoHandler.getPrintName.call(user), value: user.id };
                });

                const introScreensOptions = landings.map(landing => {
                    const { landing_id } = landing;

                    return { value: landing_id };
                });

                const clearHistorySchema: Dict<ISchemaItem> = {
                    user_id: {
                        type: controlType.variants, variants: userIdOptions, display_name: 'Аккаунт пользователя',
                        required: true,
                        multi_select: true,
                    },
                    id: {
                        type: controlType.variants,
                        variants: introScreensOptions,
                        display_name: 'Id интроскрина',
                        required: true,
                    },
                };

                this.setState({ clearHistorySchema });

            })
            .catch(schemaLoadingError => {
                this.setState({ schemaLoadingError });
            });
    }

    getActions() {
        this.setState({ isActionLoading: true }, () => {
            this.requestLandings.exec(LANDING_REQUESTS.GET_ACTIONS)
                .then(response => {
                    this.setState({
                        isActionLoading: false,
                        actionError: null,
                        actions: response.report,
                    }, () => this.setActualActions());
                })
                .catch(actionError => this.setState({
                    actionError,
                    isActionLoading: false,
                }));
        });
    }

    setActualActions() {
        const currentLanding = this.state.selectedTagName;
        const actualActions = this.state.actions?.filter((el: any) => {
            return el.action_meta?.landing?.some(landing => landing === currentLanding);
        });

        this.setState({ actualActions });
    }

    setExtraAction(action_id) {
        this.setState({
            actualActions: [
                ...this.state.actualActions,
                { action_id },
            ],
        });
    }

    openTagModal() {
        this.setState({ isTagModalOpen: true });
    }

    closeTagModal(isSuccessful: boolean) {
        this.setState({ isTagModalOpen: false }, () => {
            if (isSuccessful) {
                this.props.updateData();
            }
        });
    }

    onTagSelect(selectedTagName: string) {
        this.setState({ selectedTagName });
    }

    onFormChange(clearHistoryFormData: Dict<any>) {
        this.setState({ clearHistoryFormData });
    }

    testIntroScreen() {
        const { onClose } = this.props;
        const { selectedTagName } = this.state;
        const clearHistoryFormData: Dict<any> = this.state.clearHistoryFormData;

        let { user_id } = clearHistoryFormData;

        if (!Array.isArray(user_id)) {
            user_id = [user_id];
        }

        this.setState({ isTagAttaching: true }, () => {
            this.clearHistory()
                .then(() => {

                    this.props.constantsIFaceTag
                        .then(response => {
                            const iface_tag_implementations = response.iface_tag_implementations ?? {};

                            let schema: Dict<ISchemaItem>;
                            if (iface_tag_implementations?.[selectedTagName]) {
                                schema = iface_tag_implementations?.[selectedTagName];
                            } else {
                                schema = iface_tag_implementations[USER_LANDING_TAG_TYPE];
                            }

                            let values = FormConstructorSchemaWorker.constructValuesBySchema(schema);
                            values = FormConstructorValuesWorker.formatValues({
                                valuesFull: values,
                                schemaFull: schema,
                                parents: [],
                            });

                            Promise.all(
                                user_id.map(user_id => {
                                    const data = Object.assign({}, {
                                        priority: 0,
                                        tag: selectedTagName,
                                        object_id: user_id,
                                    }, values);

                                    return this.requestLandings.exec(LANDING_REQUESTS.ATTACH_USER_TAG,
                                        { body: data });
                                }),
                            )
                                .then(() => {
                                    this.setState({ isTagAttaching: false });
                                    onClose();
                                })
                                .catch(attachingError => {
                                    this.setState({ attachingError, isTagAttaching: false });
                                });

                        });
                });
        });

    }

    clearHistory() {
        return new Promise((resolve, reject) => {
            const clearHistoryFormData: Dict<any> = this.state.clearHistoryFormData;
            let { user_id, id } = clearHistoryFormData;

            if (!Array.isArray(user_id)) {
                user_id = [user_id];
            }

            this.setState({ isHistoryClearing: true }, () => {
                Promise.all(
                    user_id.map(userId => {
                        return this.requestLandings.exec(LANDING_REQUESTS.CLEAR_LANDING_HISTORY, {
                            queryParams: {
                                user_id: userId,
                                id,
                            },
                        });
                    }),
                )
                    .then(() => {
                        this.setState({ isHistoryClearing: false }, () => {
                            resolve();
                        });
                    })
                    .catch(clearHistoryError => {
                        this.setState({ clearHistoryError, isHistoryClearing: false }, () => {
                            reject();
                        });
                    });
            });
        });
    }

    render() {
        const { userLandingTags, onClose, clearHistoryInitialFormData, updateData } = this.props;
        const {
            isTagModalOpen, selectedTagName, attachingError, clearHistoryError,
            isHistoryClearing, isTagAttaching, schemaLoadingError, isActionLoading, actualActions, actionError,
        } = this.state;
        const clearHistorySchema: Dict<any> = this.state.clearHistorySchema;
        const clearHistoryFormData: Dict<any> = this.state.clearHistoryFormData;

        const introScreenTags = clearHistoryFormData && userLandingTags?.[clearHistoryFormData.id] || null;
        const introScreenTagsOptions: IOptionInfo[] = introScreenTags
            ? introScreenTags.map(introScreenTag => {
                return { value: introScreenTag.name, description: introScreenTag.display_name };
            })
            : null;

        const isFormValid = clearHistorySchema.user_id && clearHistorySchema.id && selectedTagName;

        const currentLandingId = clearHistoryInitialFormData.id || clearHistoryFormData.id;

        return <>
            <Window title={'Добавить интроскрин пользователю'}
                    onClose={onClose.bind(this)}
                    description={ADD_INTRO_SCREEN_MODAL_DESCRIPTION}
                    error={attachingError || clearHistoryError || schemaLoadingError || actionError}>
                <div className={style.part_key}>Тег:</div>
                {introScreenTags
                    ? <div>
                        <Select options={introScreenTagsOptions}
                                initialValues={introScreenTagsOptions?.[0] ? [introScreenTagsOptions[0].value] : []}
                                placeholder={'Тег'}
                                onSelect={this.onTagSelect.bind(this)}
                                description={'Тег, который будет навешан на пользователя для показа интроскрина'}/>
                    </div>
                    : <div>
                        Не найдено подходящего тега
                    </div>}
                <div className={style.create_item_link}>
                    <Link onClick={this.openTagModal.bind(this)}>Создать тег</Link>
                </div>
                <div className={style.create_item_link}>
                    <Link onClick={updateData.bind(this)}>Обновить список тегов</Link>
                </div>
                <div className={style.part_key}>Пользователь:</div>
                <FormConstructor schema={clearHistorySchema}
                                 initialData={clearHistoryInitialFormData}
                                 onChange={this.onFormChange.bind(this)}/>

                <div className={style.part_key}>Экшен:</div>
                <div>
                    {isActionLoading
                        ? <Spin size={'s'} />
                        : <div>
                            {!actualActions.length ? EMPTY_DATA : null}

                            {actualActions.map((el: any) => {
                                return <div key={el.action_id}>{el.action_id}</div>;
                            })}
                        </div>
                    }
                </div>

                <div className={coreStyle.button_container}>
                    <CancelButton onClick={onClose.bind(this)}/>
                    <Button disabled={!isFormValid}
                            isLoading={isTagAttaching || isHistoryClearing}
                            onClick={this.testIntroScreen.bind(this)}>Навесить интроскрин</Button>
                </div>
                <div className={coreStyle.button_container}>
                    <Button basic
                            isLoading={isTagAttaching || isHistoryClearing}
                            disabled={!clearHistoryFormData.user_id || !clearHistoryFormData.id}
                            onClick={this.clearHistory.bind(this)}>Очистить историю показа</Button>

                    <AddActionButton disabled={!currentLandingId}
                                     setExtraAction={this.setExtraAction.bind(this)}
                                     landing={currentLandingId}/>
                </div>
            </Window>
            {isTagModalOpen
                ? <TagEditModal tagType={USER_LANDING_TAG_TYPE}
                                initialData={{
                                    need_history_deep: true,
                                    deep_history_deep: '3d',
                                    landing_id: clearHistoryFormData.id,
                                }}
                                tagName={`${clearHistoryFormData.id}${SCREEN_TAG_TYPE_PREFIX}`}
                                displayName={`${clearHistoryFormData.id}${SCREEN_TAG_TYPE_PREFIX}`}
                                onClose={this.closeTagModal.bind(this)}/>
                : null}
        </>;
    }
}
