import * as React from 'react';

import { Dict } from '../../../../../types';
import { ONE_SECOND } from '../../../../constants';
import { Button, ButtonTypes, ClearButton, SaveButton } from '../../../../ui/Button';
import { Confirm } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import { deepCopy } from '../../../../utils/utils';
import { FormConstructor } from '../../../FormConstructor';
import { controlType } from '../../../FormConstructor/types';
import { SimpleError } from '../../../SimpleError';
import { ProposeCommentModal } from '../../ProposeCommentModal';
import {
    LANDING_CONFIG_CLEAR,
    LANDING_CONFIG_CLEAR_JSON,
    LANDING_CONFIG_SCHEMA_JSON,
    LANDING_CONFIG_SCHEMA_MAIN,
} from '../constants';
import { LANDINGS_REQUESTS, REQUESTS } from '../request';
import { ILanding } from '../types';

interface ILandingFormProps {
    selectedLanding: ILanding;
    onClear: () => {};
    getData: (selectedLandingId?: string) => {};
    onChange: (updatedLanding: Dict<any>) => {};
}

interface ILandingFormState {
    landingFormValue: Dict<any> | null;
    landingValueInitJSON: Dict<any> | null;
    landingFormValueJSON: Dict<any> | null;
    currentJSONEditor: 'form' | 'json';
    saveError: Error | null;
    isWorking: boolean;

    showProposeComment: boolean;
    isWorkingPropose: boolean;
    proposeError: Error | null;
    landingProposeAction: LandingProposeAction | null;

}

enum LandingProposeAction {
    REJECT = 'REJECT',
    CONFIRM = 'CONFIRM'
}

export default class LandingForm extends React.Component<ILandingFormProps, ILandingFormState> {
    state: ILandingFormState = {
        landingFormValue: {},
        landingValueInitJSON: {},
        landingFormValueJSON: {},
        currentJSONEditor: 'form',
        saveError: null,
        isWorking: false,

        showProposeComment: false,
        isWorkingPropose: false,
        proposeError: null,
        landingProposeAction: null,
    };
    request = new Request2({ requestConfigs: LANDINGS_REQUESTS });

    componentDidMount(): void {
        const { selectedLanding } = this.props;

        const landingValueInitJSON = Object.assign({ landing_json: {} }, deepCopy(LANDING_CONFIG_CLEAR_JSON),
            deepCopy({ landing_json: selectedLanding?.landing_json || {} }));

        this.setState({ landingValueInitJSON });
    }

    shouldComponentUpdate(
        nextProps: Readonly<ILandingFormProps>,
        nextState: Readonly<ILandingFormState>,
        nextContext: any,
    ): boolean {
        return !isObjectEqual(nextProps, this.props) || !isObjectEqual(nextState, this.state);
    }

    componentDidUpdate(prevProps: Readonly<ILandingFormProps>): void {
        if (!isObjectEqual(prevProps.selectedLanding, this.props.selectedLanding)) {
            const { selectedLanding } = this.props;

            const landingValueInitJSON = Object.assign({ landing_json: {} }, deepCopy(LANDING_CONFIG_CLEAR_JSON),
                deepCopy({ landing_json: selectedLanding?.landing_json || {} }));

            this.setState({ landingValueInitJSON, saveError: null });
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    onFormChange(landingFormValue: Dict<any>) {
        this.setState({ landingFormValue, saveError: null }, () => {
            this.onAllFormChange();
        });
    }

    onFormChangeJSON(landingFormValueJSON: Dict<any>) {
        const landingValueInitJSON = this.state.landingValueInitJSON ?? { landing_json: {} };
        const newLandingFormValue = Object.assign({}, landingValueInitJSON?.landing_json ?? {},
            landingFormValueJSON?.landing_json ?? {});

        this.setState({
            landingFormValueJSON: { landing_json: deepCopy(newLandingFormValue) },
            saveError: null,
        }, () => {
            this.onAllFormChange();
        });
    }

    onEdit(landingFormValueJSON: Dict<any>) {
        this.setState({ landingFormValueJSON: deepCopy(landingFormValueJSON) }, () => {
            this.onAllFormChange();
        });
    }

    onAllFormChange() {
        const { landingFormValue, landingFormValueJSON } = this.state;
        const fullForm = Object.assign({}, landingFormValue, landingFormValueJSON);
        this.props.onChange(fullForm);
    }

    changeJSONEditor() {
        let { currentJSONEditor } = this.state;
        const isForm = currentJSONEditor === 'form';
        currentJSONEditor = isForm ? 'json' : 'form';

        this.setState({ currentJSONEditor, landingValueInitJSON: deepCopy(this.state.landingFormValueJSON ?? {}) });
    }

    prepareToSave() {
        const landingFormValue = this.state.landingFormValue ?? {};
        const landingFormValueJSON = this.state.landingFormValueJSON ?? {};

        if (landingFormValue?.landing_auditory_condition
            && !Object.keys(landingFormValue.landing_auditory_condition).length) {
            landingFormValue.landing_auditory_condition = null;
        }

        //https://st.yandex-team.ru/DRIVEFRONT-554
        if (landingFormValue?.landing_geo_tags && Array.isArray(landingFormValue?.landing_geo_tags)) {
            landingFormValue.landing_geo_tags = landingFormValue.landing_geo_tags.join(',');
        }

        landingFormValue.landing_json = landingFormValueJSON.landing_json;

        return landingFormValue;
    }

    onSaveClick() {
        const { onClear, getData } = this.props;
        const landingFormValue = this.prepareToSave();

        this.setState({ isWorking: true, saveError: null }, () => {
            this.request.exec(REQUESTS.UPSERT_LANDING, { body: landingFormValue })
                .then(() => {
                    this.setState({ isWorking: false });
                    onClear();
                    getData(landingFormValue?.landing_id);
                })
                .catch(saveError => {
                    this.setState({ saveError, isWorking: false });
                });
        });
    }

    showProposeComment(state) {
        this.setState({
            showProposeComment: state,
        });
    }

    propose(comment) {
        this.setState({
            proposeError: null,
            isWorkingPropose: true,
        }, () => {
            const landingFormValue = this.prepareToSave();
            this.request.exec(REQUESTS.PROPOSE_LANDING, {
                queryParams: { comment },
                body: landingFormValue,
            })
                .then(() => {
                    this.setState({
                        proposeError: null,
                        isWorkingPropose: false,
                        showProposeComment: false,
                    }, () => {
                        setTimeout(() => {
                            this.props.getData();
                        }, ONE_SECOND);
                    });
                })
                .catch((proposeError) => {
                    this.setState({
                        proposeError,
                        isWorkingPropose: false,
                    });
                });
        });
    }

    showConfirm(landingProposeAction: LandingProposeAction) {
        this.setState({
            landingProposeAction,
        });
    }

    landingProposeAccept() {
        this.setState({
            proposeError: null,
            isWorkingPropose: true,
        }, () => {
            const action = `PROPOSITION_${this.state.landingProposeAction}`;
            this.request.exec(REQUESTS?.[action], {
                queryParams: {
                    comment: action,
                },
                body: {
                    proposition_ids: [this.props.selectedLanding.proposition_id],
                },
            })
                .then(() => {
                    this.props.onClear();
                    this.setState({
                        proposeError: null,
                        isWorkingPropose: false,
                        landingProposeAction: null,
                    }, () => {
                        setTimeout(() => {
                            this.props.getData();
                        }, ONE_SECOND);
                    });
                })
                .catch((proposeError) => {
                    this.setState({
                        proposeError,
                        isWorkingPropose: false,
                    });
                });
        });
    }

    render() {
        const { selectedLanding, onClear } = this.props;

        //https://st.yandex-team.ru/DRIVEFRONT-554
        if (selectedLanding?.landing_geo_tags) {
            selectedLanding.landing_geo_tags = selectedLanding?.landing_geo_tags?.toString().split(',');
        }

        const { saveError, isWorking, currentJSONEditor, showProposeComment, landingProposeAction } = this.state;
        const landingValueInitJSON = this.state.landingValueInitJSON ?? {};

        const initialData = Object.assign({}, deepCopy(LANDING_CONFIG_CLEAR),
            deepCopy(selectedLanding || {}));

        const isProposition = selectedLanding?.proposition_id;

        const propositionLabel = isProposition && <div>
            <hr/>
            <div>Предложение к интроскрину: <b>{selectedLanding.landing_id}</b></div>
            <div>комментарий: <b>{selectedLanding.proposition_description}</b></div>
            <div>id: <i>{selectedLanding.proposition_id}</i></div>
            <hr/>
        </div>;

        return <>
            {
                landingProposeAction && <Confirm error={this.state.proposeError}
                                                 isWorking={this.state.isWorkingPropose}
                                                 title={landingProposeAction}
                                                 question={`Выполнить действие?`}
                                                 accept={this.landingProposeAccept.bind(this)}
                                                 onClose={this.showConfirm.bind(this, null)}/>
            }
            {propositionLabel}
            <FormConstructor initialData={initialData}
                             schema={LANDING_CONFIG_SCHEMA_MAIN}
                             onChange={this.onFormChange.bind(this)}/>
            <Link onClick={this.changeJSONEditor.bind(this)}>
                {currentJSONEditor === 'form' ? 'Редактировать в JSON' : 'Редактировать в Форме'}
            </Link>
            {currentJSONEditor === 'form'
                ? <FormConstructor initialData={landingValueInitJSON ?? { landing_json: {} }}
                                   schema={LANDING_CONFIG_SCHEMA_JSON}
                                   onChange={this.onFormChangeJSON.bind(this)}/>
                : <FormConstructor initialData={landingValueInitJSON ?? { landing_json: {} }}
                                   schema={{ landing_json: { type: controlType.json, display_name: 'JSON' } }}
                                   onChange={this.onEdit.bind(this)}/>}
            {saveError
                ? <SimpleError error={saveError} data={{ label: 'Ошибка при обновлении лендингов' }}/>
                : null}
            {
                showProposeComment
                && <ProposeCommentModal setComment={this.propose.bind(this)}
                                        onClose={this.showProposeComment.bind(this, false)}
                                        error={this.state.proposeError}
                                        isRequired={true}
                                        isWorking={this.state.isWorkingPropose}/>
            }
            {propositionLabel}
            <div className={coreStyle.button_container}>
                <ClearButton isLoading={isWorking} onClick={onClear.bind(this)}/>
                {!isProposition ? <>
                    <Button onClick={this.showProposeComment.bind(this, true)}
                            disabled={!selectedLanding}>Заявка</Button>
                    <SaveButton isLoading={isWorking} onClick={this.onSaveClick.bind(this)}/>
                </> : <>
                    <Button colorType={ButtonTypes.negative}
                            onClick={this.showConfirm.bind(this, LandingProposeAction.REJECT)}>Отклонить</Button>
                    <Button colorType={ButtonTypes.positive}
                            onClick={this.showConfirm.bind(this, LandingProposeAction.CONFIRM)}>Принять</Button>
                </>}
            </div>
        </>;
    }
}
