import * as React from 'react';

import { Dict } from '../../../../../types';
import { EMPTY_DATA } from '../../../../constants';
import Checkbox from '../../../../ui/Checkbox';
import { Cross } from '../../../../ui/Cross';
import { Confirm, IWithConfirmState } from '../../../../ui/FullModal';
import { Input } from '../../../../ui/Input';
import { TabItem, Tabs } from '../../../../ui/Tabs';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import Spin from '../../../Spin';
import { IChat } from '../../ChatsEditor/types';
import { LANDINGS_REQUESTS, REQUESTS } from '../request';
import { ILanding } from '../types';
import * as style from './index.css';

interface ILandingsListProps {
    landings: ILanding[];
    chats: IChat[];
    dataIsLoading: boolean;
    selectedLanding: ILanding | null;
    onLandingListItemSelect: () => void;
    getData: () => void;
    onClear: () => void;
    userLandingTags: Dict<any>;
}

interface ILandingsListState extends IWithConfirmState {
    filterValue: string;
    currentTab: string;
    isActiveSwitcher: boolean;
}

export const INTROSCREENS_TYPES = {
    all: 'Все',
    introscreens: 'Интроскрины',
    warnings: 'Ворнинги',
    proposition: 'Предл.',
};

export default class LandingsList extends React.Component<ILandingsListProps, ILandingsListState> {
    state: ILandingsListState = {
        filterValue: '',
        currentTab: INTROSCREENS_TYPES.all,
        isActiveSwitcher: true,
        confirmIsOpen: false,
        question: '',
        accept: () => {},
        isWorking: false,
        error: null,
    };
    request = new Request2({ requestConfigs: LANDINGS_REQUESTS });
    tabs: TabItem[] = Object.values(INTROSCREENS_TYPES)?.map(option => {
        return { name: option, link: option };
    });

    shouldComponentUpdate(
        nextProps: Readonly<ILandingsListProps>,
        nextState: Readonly<ILandingsListState>,
        nextContext: any,
    ): boolean {
        return !isObjectEqual(nextProps, this.props) || !isObjectEqual(nextState, this.state);
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    onFilterChange(filterValue: string) {
        this.setState({ filterValue });
    }

    onActiveSwitcherChange(isActiveSwitcher: boolean) {
        this.setState({ isActiveSwitcher });
    }

    onSelectTab(currentTab: string) {
        this.setState({ currentTab });
    }

    getFilteredLandings(): ILanding[] {
        let { landings = [] } = this.props;
        const { filterValue, isActiveSwitcher, currentTab } = this.state;

        if (isActiveSwitcher) {
            landings = landings.filter(landing => landing.landing_enabled);
        }

        if (currentTab === INTROSCREENS_TYPES.introscreens) {
            landings = landings.filter(landing => landing.landing_chat_id);
        } else if (currentTab === INTROSCREENS_TYPES.warnings) {
            landings = landings
                .filter(landing => !landing.landing_chat_id && Object.keys(landing.landing_json ?? {})?.length);
        } else if (currentTab === INTROSCREENS_TYPES.proposition) {
            landings = landings.filter(landing => landing.proposition_id);
        }

        if (filterValue) {
            const finalSearchValue = filterValue.toLowerCase();
            const filteredChatInfo = this.filterChats(finalSearchValue) ?? [];

            landings = landings.filter(landing => landing.landing_id?.toLowerCase()?.includes(finalSearchValue)
                || landing.landing_json?.title?.toLowerCase().includes(finalSearchValue)
                || landing.landing_json?.points?.some(string => string?.toLowerCase()?.includes(finalSearchValue))
                || landing.landing_json?.button?.toLowerCase()?.includes(finalSearchValue)
                || landing.landing_json?.other_buttons
                    ?.some(button => button.button?.toLowerCase()?.includes(finalSearchValue))
                || filteredChatInfo
                    .some(item => item.chat_id === landing.landing_chat_id && item.node_id === landing.landing_id));
        }

        return landings
            .sort((a, b) => a?.landing_id?.localeCompare(b?.landing_id));
    }

    filterChats(filterValue: string): Dict<any> [] {
        const { chats = [] } = this.props;
        const filteredChatInfo: { chat_id: string; node_id: string }[] = [];

        chats.forEach(chat => {
            chat.meta?.chat_script?.items?.forEach(item => {
                if (item.action_button_text?.toLowerCase()?.includes(filterValue)
                    || item.pre_action_messages?.some(message => message?.text?.toLowerCase()?.includes(filterValue))) {
                    filteredChatInfo.push({ chat_id: chat.name, node_id: item.id });
                }
            });
        });

        return filteredChatInfo;
    }

    onDeleteClick(id: string) {
        const { getData, onClear } = this.props;

        this.setState({
            confirmIsOpen: true,
            question: `Удалить лендинг ${id}?`,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    this.request.exec(REQUESTS.REMOVE_LANDING, { body: { id: [id] } })
                        .then(() => {
                            this.closeModal();
                            getData();
                            onClear();
                        })
                        .catch(error => {
                            this.setState({
                                error,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    closeModal() {
        this.setState({ isWorking: false, confirmIsOpen: false });
    }

    render() {
        const { dataIsLoading, selectedLanding, onLandingListItemSelect, userLandingTags } = this.props;
        const introScreensWithTags = Object.keys(userLandingTags);
        const {
            filterValue,
            isActiveSwitcher,
            confirmIsOpen,
            accept,
            error,
            isWorking,
            question,
            currentTab,
        } = this.state;

        const filteredLandings = this.getFilteredLandings();

        return <>
            <div>
                <div className={style.list_controls}>
                    <Input className={style.filter_input}
                           value={filterValue}
                           onChange={this.onFilterChange.bind(this)}
                           placeholder={'Фильтр'}/>
                    <Checkbox checked={isActiveSwitcher}
                              onChange={this.onActiveSwitcherChange.bind(this)}
                              placeholder={'Только активные'}/>
                </div>

                <Tabs tabs={this.tabs}
                      currentTab={currentTab}
                      selectTab={this.onSelectTab.bind(this)}/>
            </div>

            <div className={style.list}>
                {dataIsLoading
                    ? <Spin/>
                    : filteredLandings
                        ? filteredLandings.map(landing => {
                            return <LandingListItem key={`${landing.landing_id}_${landing.proposition_id}`}
                                                    landing={landing}
                                                    introScreensWithTags={introScreensWithTags}
                                                    selectedLanding={selectedLanding}
                                                    onSelect={onLandingListItemSelect.bind(this)}
                                                    onDeleteClick={this.onDeleteClick.bind(this, landing.landing_id)}/>;
                        })
                        : null}
            </div>
            {
                confirmIsOpen
                    ? <Confirm accept={accept.bind(this)}
                               error={error}
                               onClose={this.closeModal.bind(this)}
                               isWorking={isWorking}
                               question={question}/>
                    : null
            }
        </>;
    }
}

interface ILandingListItemProps {
    landing: ILanding;
    selectedLanding: ILanding | null;
    onSelect: () => void;
    onDeleteClick: () => void;
    introScreensWithTags: string[];
}

const LandingListItem = React.memo((props: ILandingListItemProps) => {
    const { landing, selectedLanding, onSelect, onDeleteClick, introScreensWithTags } = props;
    const { landing_id = '', landing_json = {}, landing_chat_id, landing_chat_title, proposition_id } = landing;

    const isIncludesChatId = !!landing_chat_id;
    const isProposition = landing?.proposition_id;

    let isSelected = false;
    let description = landing_chat_title || landing_json?.title || EMPTY_DATA;

    if (isProposition) {
        description = `Предложениие: ${landing.proposition_description}` || EMPTY_DATA;
    }

    if (selectedLanding?.proposition_id) {
        isSelected = !!(landing?.proposition_id && (selectedLanding?.proposition_id === proposition_id));
    } else {
        isSelected = (!landing?.proposition_id && (selectedLanding?.landing_id === landing_id));
    }

    return <div title={isProposition ? 'Предложение' : 'Лендинг'}
                className={`${style.landings_list_item} `
    + `${isSelected ? style.selected : ''} `
    + `${isProposition ? style.is_proposition : ''}`}
                onClick={onSelect?.bind(null, landing)}>
        <div className={style.is_active_icon_container}>
            <div className={`${style.is_active_icon} ${landing.landing_enabled ? style.enabled : ''}`}/>
        </div>
        <div className={style.name} title={landing_id || EMPTY_DATA}>
            <span className={style.id}>{landing_id || EMPTY_DATA}</span>
            {isIncludesChatId
                ? <span className={style.is_chat}>Чат</span>
                : null}
            {introScreensWithTags.includes(landing_id)
                ? <span className={style.is_includes_tag}>Тег</span>
                : null}
        </div>

        <div className={style.description}
             title={landing_chat_title || landing_json?.title || EMPTY_DATA}>
            {description}
        </div>
        {
            !isProposition && <div className={style.close_icon}>
                <Cross onClick={onDeleteClick.bind(null, landing_id)}/>
            </div>
        }
    </div>;
});
