import React from 'react';

import { Dict } from '../../../../../types';
import { EMPTY_DATA } from '../../../../constants';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { SCREEN_PREVIEW_WIDTH } from '../types';
import * as style from './index.css';

const REG_EXP = (/\[\s*url[^\]]*\](.*?)\[\s*\/\s*url\]/ig);
const FONT_SIZE_COEFFICIENT = 28.5;
const COLOR_CHANGE_DELAY = 3;

enum BUTTON_STYLES {
    solid = 'solid',
    bordered = 'bordered',
}

interface IScreenPreviewProps {
    landingJSON: Dict<any>;
}

interface IScreenPreviewState {
}

export default class WarningPreview extends React.Component<IScreenPreviewProps, IScreenPreviewState> {
    state: IScreenPreviewState = {};
    screenPreviewDiv: any;

    shouldComponentUpdate(nextProps: Readonly<IScreenPreviewProps>): boolean {
        return !isObjectEqual(this.props, nextProps);
    }

    editNewLines(textLine: string) {
        return textLine?.replace(/\n/ig, '<br>');
    }

    editLinks(textLine: string) {
        const matches = textLine?.match(REG_EXP);
        matches?.forEach((match: string) => {
            const linkText = match
                .replace(/\[.*url.[^\]]*\]/ig, '')
                .replace(/\[.*url.*]/ig, '');

            const linkHref = match
                .replace(/\[.*url=/ig, '')
                .replace(/\].*/ig, '');
            textLine = textLine.replace(match, `<a target="_blank" href="${linkHref}">${linkText}</a>`);
        });

        return textLine;
    }

    editButton(buttonText: string) {
        return buttonText?.replace(/\n/ig, '<br>');
    }

    generateButtonStyle(json: Dict<any> ) {
        const buttonColor = json.button_color ?? 'fff';
        const buttonColorString = buttonColor[0] === '#' ? buttonColor : `#${buttonColor}`;
        if (json?.button_style === BUTTON_STYLES.bordered) {
            return {
                backgroundColor: 'transparent',
                border: `1px solid ${buttonColorString}`,
            };
        }

        return {
            backgroundColor: buttonColorString,
        };
    }

    generateButtonColor(json: Dict<any> ) {
        const buttonColor = json.button_color ?? 'fff';
        const buttonColorString = buttonColor[0] === '#' ? buttonColor : `#${buttonColor}`;

        return {
            color: `${json?.button_style === BUTTON_STYLES.bordered ? buttonColorString : 'inherit'}`,
        };
    }

    render() {
        const json = this.props.landingJSON;
        let {
            title = EMPTY_DATA,
            button = EMPTY_DATA,
            color1 = '000',
            color2 = '000',
        } = json;

        const backgroundKeyframes =
            `@keyframes backgroundColorChange
                {
                  0%   {background: ${color1[0] === '#' ? color1 : `#${color1}`};}
                  50%  {background: ${color2[0] === '#' ? color2 : `#${color2}`};}
                  100% {background: ${color1[0] === '#' ? color1 : `#${color1}`};}
             }`;

        const styleSheet: any = document.styleSheets[0];
        for (let i = 0; i < styleSheet.cssRules.length; i++) {
            if (styleSheet?.cssRules?.[i]?.name === 'backgroundColorChange') {
                styleSheet.deleteRule(i);
            }
        }

        styleSheet?.insertRule(backgroundKeyframes);

        const fontSize = SCREEN_PREVIEW_WIDTH / FONT_SIZE_COEFFICIENT;
        const div_style = {
            width: SCREEN_PREVIEW_WIDTH,
            backgroundColor: color1[0] === '#' ? color1 : `#${color1}`,
            animation: `backgroundColorChange ${COLOR_CHANGE_DELAY}s infinite`,
            fontSize: `${fontSize}px`,
        };

        title = this.editNewLines(title);
        title = this.editLinks(title);

        return <div className={style.main} style={div_style}>
            <div>
                <div className={style.title}>
                    <h1>
                        {title}
                    </h1>
                </div>
                <div className={style.screen_preview_text}>
                    {json && Array.isArray(json.points)
                    && json.points.map((point: string, index: number) => {
                        if (!point) {
                            return <div key={index}>
                                {point === null && 'NULL'
                                || point === undefined && 'UNDEFINED'
                                || ''}
                            </div>;
                        }

                        point = this.editNewLines(point);
                        point = this.editLinks(point);

                        return <div key={index}>{point}</div>;
                    })}
                </div>
            </div>

            <div className={style.button_container}>
                {json.other_buttons?.length
                    ? json.other_buttons.map((button, ind) => {
                        return <div className={style.screen_preview_button}
                                    style={this.generateButtonStyle(button)}
                                    key={ind}>
                            <div style={this.generateButtonColor(button)}>
                                {this.editButton(button.button || EMPTY_DATA)}
                            </div>
                        </div>;
                    })
                    : null
                }
                <div className={style.screen_preview_button} style={this.generateButtonStyle(json)}>
                    <div style={this.generateButtonColor(json)}>
                        {this.editButton(button)}
                    </div>
                </div>
            </div>
        </div>;
    }
}
