import * as React from 'react';

import { Dict } from '../../../../types';
import { Button } from '../../../ui/Button';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import { ITag } from '../../TagModal/component';
import { ChatIntroScreenPreview } from '../ChatsEditor/ChatIntroscreenPreview/component';
import { IChat } from '../ChatsEditor/types';
import { NOTIFIERS_REQUESTS, REQUESTS as NOTIFIER_REQUESTS } from '../Notifiers/request';
import AddIntroScreenToUserModal from './AddIntroScreenToUserModal';
import { USER_LANDING_TAG_TYPE } from './constants';
import * as style from './index.css';
import LandingForm from './LandingForm';
import LandingsList from './LandingsList';
import PreviewPreview from './PreviewPreview';
import { LANDINGS_REQUESTS, REQUESTS as LANDING_REQUESTS } from './request';
import { ILanding, SCREEN_PREVIEW_WIDTH } from './types';
import WarningPreview from './WarningPreview/component';

const TAGS_RESPONSE_INDEX = 2;

interface ILandingsState {
    dataIsLoading: boolean;
    landings: ILanding[];
    propositions: ILanding[];
    chats: IChat[];
    landingsLoadingError: Error | null;
    selectedLanding: ILanding | null;
    formData: ILanding | null;
    clearHistoryInitialFormData: Dict<any>;
    userLandingTags: Dict<any>;
    isAddIntroScreenToUserModal: boolean;
}

export default class Landings extends React.Component<{}, ILandingsState> {
    state: ILandingsState = {
        dataIsLoading: false,
        landings: [],
        propositions: [],
        chats: [],
        landingsLoadingError: null,
        selectedLanding: null,
        formData: null,
        clearHistoryInitialFormData: {},
        userLandingTags: {},
        isAddIntroScreenToUserModal: false,
    };
    requestNotifier = new Request2({ requestConfigs: NOTIFIERS_REQUESTS });
    requestLandings = new Request2({ requestConfigs: LANDINGS_REQUESTS });

    componentDidMount() {
        this.getData();
    }

    componentWillUnmount(): void {
        this.requestNotifier.abort();
        this.requestLandings.abort();
    }

    getData(selectedLandingId?: string) {
        this.setState({
            dataIsLoading: true,
            landingsLoadingError: null,
        }, () => {
            Promise.all([
                this.requestNotifier.exec(NOTIFIER_REQUESTS.GET_NOTIFIERS),
                this.requestLandings.exec(LANDING_REQUESTS.GET_LANDINGS),
                this.requestLandings.exec(LANDING_REQUESTS.GET_TAGS),
            ])
                .then(response => {
                    let landings: ILanding[] = response?.find(el => el.landings)?.landings ?? [];
                    const propositions: ILanding[] = response?.find(el => el.propositions)?.propositions ?? [];
                    landings = landings.concat(propositions);
                    const notifiers: IChat[] = response?.find(el => el.objects)?.objects ?? [];
                    const chats: IChat[] = notifiers?.filter(notifier => notifier.type === 'native_chat') ?? [];
                    const tags: ITag[] = response?.[TAGS_RESPONSE_INDEX]?.records ?? [];

                    const userLandingTagsArray = tags.filter(tag => {
                        return tag.type === USER_LANDING_TAG_TYPE;
                    });

                    const userLandingTags = userLandingTagsArray.reduce((result: Dict<ITag[]>, userLandingTag) => {
                        const meta: any = userLandingTag?.meta ?? {};
                        userLandingTag.meta = isValidJSONString(meta)
                            ? JSON.parse(meta)
                            : typeof(meta) === 'object' ? meta : {};
                        const objectMeta: any = userLandingTag.meta;
                        const landing_id: string = objectMeta.landing_id;

                        if (result?.[landing_id]) {
                            result[landing_id].push(userLandingTag);
                        } else {
                            result[landing_id] = [userLandingTag];
                        }

                        return result;
                    }, {});

                    if (selectedLandingId) {
                        const selectedLanding = landings
                            .filter(landing => landing.landing_id === selectedLandingId)?.[0] ?? null;

                        this.setState({
                            dataIsLoading: false,
                            landings,
                            propositions,
                            selectedLanding,
                            chats,
                            landingsLoadingError: null,
                            userLandingTags,
                        });
                    } else {
                        this.setState({
                            dataIsLoading: false,
                            landings,
                            propositions,
                            chats,
                            userLandingTags,
                        });
                    }
                })
                .catch(landingsLoadingError => {
                    this.setState({ landingsLoadingError, dataIsLoading: false });
                });
        });
    }

    onLandingListItemSelect(selectedLanding: ILanding) {
        this.setState({ selectedLanding });
    }

    onClear() {
        this.setState({ selectedLanding: null });
    }

    onFormChange(formData: ILanding) {
        const landing_id = formData?.landing_id ?? '';
        const clearHistoryInitialFormData = {
            id: landing_id,
        };
        this.setState({ formData, clearHistoryInitialFormData });
    }

    openAddIntroScreenToUserModal() {
        this.setState({ isAddIntroScreenToUserModal: true });
    }

    closeAddIntroScreenToUserModal() {
        this.setState({ isAddIntroScreenToUserModal: false });
    }

    render() {
        const {
            landingsLoadingError, landings, dataIsLoading,
            selectedLanding, formData, chats, clearHistoryInitialFormData,
            userLandingTags, isAddIntroScreenToUserModal,
        } = this.state;
        if (landingsLoadingError) {
            return <SimpleError error={landingsLoadingError} data={{ label: 'Ошибка при загрузке лендингов' }}/>;
        }

        const chat: IChat | null = selectedLanding
            ? chats.find(chat => chat.name === selectedLanding?.landing_chat_id) ?? null
            : null;
        const node = chat?.meta?.chat_script?.items
            ?.find(item => item.id === selectedLanding?.landing_chat_messages_group) ?? null;

        const introscreenPreviewStyle = {
            width: SCREEN_PREVIEW_WIDTH,
        };

        return <div className={style.landings}>
            <div className={style.landings_list}>
                <LandingsList landings={landings}
                              chats={chats}
                              userLandingTags={userLandingTags}
                              dataIsLoading={dataIsLoading}
                              selectedLanding={selectedLanding}
                              onClear={this.onClear.bind(this)}
                              onLandingListItemSelect={this.onLandingListItemSelect.bind(this)}
                              getData={this.getData.bind(this)}/>
            </div>
            <div className={style.landing_form}>
                <LandingForm selectedLanding={selectedLanding}
                             onChange={this.onFormChange.bind(this)}
                             onClear={this.onClear.bind(this)}
                             getData={this.getData.bind(this)}/>
                <div className={style.clear_history_controls}>
                    <Button onClick={this.openAddIntroScreenToUserModal.bind(this)}>Протестировать интроскрин</Button>
                </div>
            </div>
            <div className={style.landings_preview}>
                <PreviewPreview landing={formData}/>
                {Object.values(formData?.landing_json || {})?.length && !formData?.landing_chat_id
                    ? <WarningPreview landingJSON={formData?.landing_json ?? {}}/>
                    : formData?.landing_chat_id && formData?.landing_id
                        ? <div style={introscreenPreviewStyle}>
                            <ChatIntroScreenPreview chat={chat} node={node}/>
                        </div>
                        : null
                }
            </div>
            {isAddIntroScreenToUserModal
                ?
                <AddIntroScreenToUserModal userLandingTags={userLandingTags}
                                           landings={landings}
                                           clearHistoryInitialFormData={clearHistoryInitialFormData}
                                           updateData={this.getData.bind(this)}
                                           onClose={this.closeAddIntroScreenToUserModal.bind(this)}/>
                : null}
        </div>;
    }
}
