import * as React from 'react';
import VirtualList from 'react-tiny-virtual-list';
import ReactTooltip from 'react-tooltip';

import { Dict } from '../../../../types';
import { Button } from '../../../ui/Button';
import { Confirm } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import RadioGroup from '../../../ui/Radio/RadioGroup';
import { Request2 } from '../../../utils/request';
import { PictureGallery } from '../../PictureGallery';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import * as styles from './index.css';
import MdsTableRow, { IMdsItem } from './MdsTableRow';
import { MdsUploadDialog } from './MdsUploadDialog';

interface IMDSState {
    [key: string]: any;

    isLoading: boolean;
    images: IMdsItem[];
    isWorking: boolean;
    error: Error | null;
    fileWindowState: boolean;
    selectedImg: IMdsItem | null;
    windowSize: Dict<number>;
    filter: string;
    startIndex: number;
    deletedImg: IMdsItem | null;
    removeError: Error | null;
    isRemoveLoading: boolean;
    selectedRadioValue: string;
}

const MARGIN = 295;
const mdsTooltipId = 'mdsTooltipId';

const SORTS = {
    BY_DATE: 'BY_DATE',
    BY_ALPHABET: 'BY_ALPHABET',
};

export class MDS extends React.Component<IMDSState> {
    state: IMDSState = {
        isLoading: false,
        images: [],
        isWorking: false,
        error: null,
        fileWindowState: false,
        selectedImg: null,
        windowSize: { width: window.innerWidth, height: window.innerHeight },
        filter: '',
        startIndex: 0,
        deletedImg: null,
        removeError: null,
        isRemoveLoading: false,
        selectedRadioValue: `${SORTS.BY_DATE}`,
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });
    onResize = () => {
        this.setState({
            windowSize: { width: window.innerWidth, height: window.innerHeight },
        });
    };

    getData() {
        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_MDS_RESOURCES)
                .then(response => {
                    this.setState({
                        images: (response.keys || [])
                            .sort(this?.[`${this.state.selectedRadioValue}`]),
                        isLoading: false,
                    });
                });
        });
    }

    onChange(key: string, value: string) {
        this.setState({
            [key]: value,
        });
    }

    showImg(selectedImg: string) {
        this.setState({
            selectedImg,
        });
    }

    onCloseModal() {
        this.setState({
            selectedImg: null,
        });
    }

    showChoseFileDialog(show: boolean) {
        this.setState({
            fileWindowState: show,
            filesError: null,
        });
    }

    componentDidMount() {
        this.getData();
        window.addEventListener('resize', this.onResize);
    }

    componentWillUnmount() {
        window.removeEventListener('resize', this.onResize);
    }

    showRemoveModal(deletedImg) {
        this.setState({
            deletedImg,
        });
    }

    remove() {
        this.setState({
            isRemoveLoading: true,
        }, () => {
            this.request.exec(REQUESTS.REMOVE_MDS_RESOURCES, {
                queryParams: {
                    key: this.state.deletedImg?.key,
                },
            }).then(() => {
                this.setState({
                    isRemoveLoading: false,
                    deletedImg: null,
                });
                this.getData();
            }).catch(removeError => {
                this.setState({
                    isRemoveLoading: false,
                    removeError,
                });
            });

        });
    }

    [`${SORTS.BY_DATE}`](a: IMdsItem, b: IMdsItem) {
        return b.modification_time - a.modification_time;
    }

    [`${SORTS.BY_ALPHABET}`](a: IMdsItem, b: IMdsItem) {
        return a.key.localeCompare(b.key);
    }

    onChangeSort(selectedRadioValue) {
        this.setState({
            selectedRadioValue,
            images: this.state.images.sort(this?.[`${selectedRadioValue}`]),
        });
    }

    render() {
        let {
            isLoading,
            isWorking,
            images,
            selectedImg,
            deletedImg,
            windowSize,
            error,
            filter,
        } = this.state;

        images = images.filter((el: IMdsItem) => {
            if (!filter) {
                return true;
            }

            const filterArr = filter.split(' ');

            return filterArr.every(arr_el => el?.key?.toUpperCase()?.includes(arr_el?.toUpperCase()));

        });

        return (
            <>
                {error && <SimpleError error={error}/>}
                <div>
                    <div className={styles.filter}>
                        <Input value={filter}
                               className={styles.input}
                               placeholder={'фильтр'}
                               onChange={this.onChange.bind(this, 'filter')}/>
                        <RadioGroup id={'group'}
                                    className={styles.radio_group}
                                    initialSelectedValue={this.state.selectedRadioValue}
                                    options={[{ label: 'по дате', value: SORTS.BY_DATE }, {
                                        label: 'по алфавиту',
                                        value: SORTS.BY_ALPHABET,
                                    }]}
                                    onSelect={this.onChangeSort.bind(this)}/>
                    </div>
                    <div className={styles.controls}>
                        <Button isLoading={isWorking}
                                onClick={this.showChoseFileDialog.bind(this, true)}
                                basic>Выбрать файл</Button>
                    </div>
                    {
                        this.state.fileWindowState
                        && <MdsUploadDialog onClose={this.showChoseFileDialog.bind(this, false)}
                                            error={this.state.filesError}
                                            getData={this.getData.bind(this)}/>
                    }
                    <ReactTooltip id={mdsTooltipId} html/>
                    <div>
                        {isLoading
                            ? <Spin/>
                            : <VirtualList width={'100%'}
                                           height={windowSize.height - MARGIN}
                                           itemCount={images.length}
                                           itemSize={30}
                                           overscanCount={0}
                                           onItemsRendered={({ startIndex }) => {
                                               if (this.state.startIndex !== startIndex) {
                                                   this.setState({
                                                       startIndex,
                                                   });
                                               }
                                           }}
                                           renderItem={({ index, style }) => {
                                               const img = images[index];

                                               return (
                                                   <div key={index}
                                                        className={styles.row}
                                                        style={style}>
                                                       <MdsTableRow key={index}
                                                                    img={img}
                                                                    style={style}
                                                                    index={index}
                                                                    mdsTooltipId={mdsTooltipId}
                                                                    startIndex={this.state.startIndex}
                                                                    showImg={this.showImg.bind(this, img)}
                                                                    showRemoveModal={this.showRemoveModal
                                                                        .bind(this, img)}/>
                                                   </div>
                                               );
                                           }}/>
                        }
                        {selectedImg &&
                        <PictureGallery withSize
                                        onClose={this.onCloseModal.bind(this)}
                                        pictures={[{ link: selectedImg?.link }]}
                                        video={!!selectedImg?.key?.match('.mp4|.ogv|.webm')}/>}
                        {
                            deletedImg && <Confirm error={this.state.removeError}
                                                   accept={this.remove.bind(this)}
                                                   isWorking={this.state.isRemoveLoading}
                                                   question={<div className={styles.remove}>
                                                       <h4>Удалить фотку?</h4>
                                                       <div>
                                                           {
                                                               deletedImg?.key?.includes('.mov')
                                                                   ? <video src={deletedImg.link} autoPlay controls/>
                                                                   : <img src={deletedImg.link}/>
                                                           }
                                                       </div>
                                                       <pre>{deletedImg.link}</pre>
                                                   </div>}
                                                   onClose={this.showRemoveModal.bind(this, '')}/>
                        }
                    </div>
                </div>
            </>
        );
    }
}
