import * as React from 'react';

import { Button } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import Spin from '../../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../../request';
import * as styles from './index.css';
import { ADD_MODEL_SCHEMA } from './schema';

interface IAddModelModalProps {
    onClose: () => void;
    reloadData: () => void;
    modelId: string;
}

interface IAddModelModalState {
    error: Error | null;
    isLoading: boolean;
    isWorking: boolean;
    meta: string;
    data: any;
    file: File | null;
    blob: Blob | null;
}

export class AddModelModal extends React.Component<IAddModelModalProps, IAddModelModalState> {
    state: IAddModelModalState = {
        error: null,
        isLoading: false,
        isWorking: false,
        meta: '',
        data: {},
        file: null,
        blob: null,
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    getData() {
        const { modelId } = this.props;
        if (modelId) {
            this.setState({
                isLoading: true,
            }, () => {
                this.request.exec(REQUESTS.GET_ML_MODEL, {
                    queryParams: {
                        name: modelId,
                    },
                })
                    .then((response) => {
                        this.setState({
                            meta: JSON.stringify(response),
                            isLoading: false,
                        });
                    })
                    .catch((error) => {
                        this.setState({
                            error,
                            isLoading: false,
                        });
                    });
            });
        }
    }

    sendData() {
        this.setState({
            isWorking: true,
        }, () => {
            const meta = this.state?.data?.meta && encodeURIComponent(this.state.data.meta) || '';
            const separator = this.state?.data?.separator && encodeURIComponent(this.state.data.separator) || '';
            this.request.exec(REQUESTS.SEND_ML_MODEL, {
                file: this.state.blob,
                queryParams: {
                    meta,
                    separator,
                },
                headers: {
                    'Content-Type': 'audio/basic',
                },
            })
                .then(() => {
                    this.setState({
                        isWorking: false,
                    });
                    this.props.onClose();
                    this.props.reloadData();
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isWorking: false,
                    });
                });
        });
    }

    onChange(value) {
        this.setState({
            data: value,
        });
    }

    file(file: FileList) {
        const blob = new Blob([file[0]], { type: file[0].type });
        this.setState({
            file: file[0],
            blob,
        });
    }

    componentDidMount() {
        this.getData();
    }

    render() {
        const { error, isLoading, meta } = this.state;
        const { onClose, modelId } = this.props;

        const initialData = {
            name: modelId,
            meta,
        };

        return (
            <Window className={styles.modal}
                    onClose={onClose}
                    error={error}
                    title={'Добавить/обновить модель'}>
                {isLoading ? <Spin/> : (
                    <>
                        <FormConstructor schema={ADD_MODEL_SCHEMA}
                                         initialData={initialData}
                                         onChange={this.onChange.bind(this)}/>
                        <Button basic
                                className={styles.uploadButton}
                                onClick={this.file.bind(this)}
                                file>Добавить файл</Button>
                        <div className={styles.button_container}>
                            <Button onClick={this.props.onClose.bind(this)}
                                    basic>Отмена</Button>
                            <Button isLoading={false}
                                    onClick={this.sendData.bind(this)}>{modelId ? 'Обновить' : 'Добавить'}</Button>
                        </div>
                    </>
                )}
            </Window>
        );
    }
}
