import * as React from 'react';

import { Dict } from '../../../../../types';
import { EMPTY_DATA } from '../../../../constants';
import { Button, ButtonTypes } from '../../../../ui/Button';
import { Confirm } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import * as styleTable from '../../../../ui/Table/index.css';
import { deepCopy } from '../../../../utils/utils';
import { FormConstructor } from '../../../FormConstructor';
import { SimpleError } from '../../../SimpleError';
import { MODEL_DEFAULT_TAG_SCHEMA } from '../constants/constants';
import * as style from '../index.css';

interface IModelDefaultTagsProps {
    selectedModelCode: string | null;
    setModel: (obj: Dict<any>) => Promise<any>;
    defaultTags: any[];
    defaultTagsList: any[];
}

interface IModelDefaultTagsState {
    isWorking: boolean;
    addError: Error | null;
    removeError: Error | null;
    formValues: {
        tag_name: string;
        priority: number;
        comment: string;
    };
    selectedDefaultTagItem: number | null;
    confirmRemoveDefaultTagIsOpen: boolean;
}

export class ModelDefaultTags extends React.Component<IModelDefaultTagsProps, IModelDefaultTagsState> {
    state: IModelDefaultTagsState = {
        isWorking: false,
        addError: null,
        removeError: null,
        formValues: {
            tag_name: '',
            priority: 0,
            comment: '',
        },
        selectedDefaultTagItem: null,
        confirmRemoveDefaultTagIsOpen: false,
    };

    changeDefaultTagName(defaultTagName: string) {
        const { formValues } = this.state;
        formValues.tag_name = defaultTagName;
        this.setState({ formValues });
    }

    changeFormValue(formValues: { tag_name: string; priority: number; comment: string }) {
        this.setState({ formValues });
    }

    addDefaultTag() {
        const { formValues } = this.state;
        const { defaultTags } = this.props;

        this.setState({
            isWorking: true,
            addError: null,
        }, () => {
            const obj = [...defaultTags, formValues];

            this.props.setModel({ default_tags: obj })
                .then(() => {
                    this.setState({ isWorking: false });
                })
                .catch(addError => {
                    this.setState({ addError, isWorking: false });
                });
        });
    }

    removeTag(selectedDefaultTagItem: number) {
        this.setState({
            selectedDefaultTagItem,
        }, () => {
            this.setState({
                confirmRemoveDefaultTagIsOpen: true,
                removeError: null,
            });
        });
    }

    onCloseConfirm() {
        this.setState({ confirmRemoveDefaultTagIsOpen: false, isWorking: false, removeError: null });
    }

    removeModelDefaultTag() {
        const selectedDefaultTagItem = this.state.selectedDefaultTagItem ?? 0;

        this.setState({
            isWorking: true,
        }, () => {

            const obj = [...this.props.defaultTags.slice(0, selectedDefaultTagItem ?? 0),
                ...this.props.defaultTags.slice(selectedDefaultTagItem + 1),
            ];

            this.props.setModel({ default_tags: obj })
                .then(() => {
                    this.setState({ isWorking: false });
                    this.onCloseConfirm();
                })
                .catch(removeError => {
                    this.setState({ removeError, isWorking: false });
                });
        });
    }

    render() {
        const {
            confirmRemoveDefaultTagIsOpen,
            isWorking, selectedDefaultTagItem, addError, removeError,
        } = this.state;
        const { defaultTagsList, defaultTags = [], selectedModelCode } = this.props;

        const schema = deepCopy(MODEL_DEFAULT_TAG_SCHEMA);
        schema.tag_name.variants = defaultTagsList.map(defaultTag => defaultTag?.value);

        return <div className={style.model_default_tags}>
            <h3>Теги по умолчанию</h3>
            {selectedModelCode
                ? <>
                    {
                        defaultTags.length
                            ? <table className={`${styleTable.table} ${style.specifications_table}`}>
                                <thead>
                                    <tr>
                                        <th>#</th>
                                        <th>Тег</th>
                                        <th>Приоритет</th>
                                        <th>Комментарий</th>
                                        <th/>
                                    </tr>
                                </thead>
                                <tbody>
                                    {defaultTags
                                        .map((item, index) => {
                                            return <tr key={`${item.tag_name}_${index}`}>
                                                <td>{index + 1}</td>
                                                <td>{item.tag_name || EMPTY_DATA}</td>
                                                <td>{item.priority ?? EMPTY_DATA}</td>
                                                <td>{item.comment || EMPTY_DATA}</td>
                                                <td>
                                                    <Link onClick={this.removeTag.bind(this, index)}>Удалить</Link>
                                                </td>
                                            </tr>;
                                        })}
                                </tbody>
                            </table>
                            : null
                    }

                    <FormConstructor schema={schema} onChange={this.changeFormValue.bind(this)}/>
                    <div className={coreStyle.button_container}>
                        <Button colorType={ButtonTypes.positive}
                                basic
                                onClick={this.addDefaultTag.bind(this)}>Добавить тег по умолчанию</Button>
                    </div>
                    {addError
                        ? <SimpleError error={addError}
                                       data={{ label: 'Ошибка при добавлении тега по умолчанию ' }}/>
                        : null}
                </>
                : <div className={style.empty}>Выберите модель</div>}
            {
                confirmRemoveDefaultTagIsOpen
                    ? <Confirm error={removeError}
                               accept={this.removeModelDefaultTag.bind(this)}
                               onClose={this.onCloseConfirm.bind(this)}
                               isWorking={isWorking}
                               question={`Удалить тег ${defaultTags[selectedDefaultTagItem ?? 0]?.tag_name} у модели `
                        + `${selectedModelCode}?`}/>
                    : null
            }
        </div>;
    }
}
