import * as React from 'react';

import { EMPTY_DATA } from '../../../../constants';
import { Button, ButtonTypes } from '../../../../ui/Button';
import { Confirm } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import * as styleTable from '../../../../ui/Table/index.css';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import { SimpleError } from '../../../SimpleError';
import { IModelSpecification } from '../component';
import { MODEL_SPECIFICATION_SCHEMA } from '../constants/constants';
import * as style from '../index.css';
import { MODEL_EDITOR_REQUESTS, REQUESTS } from '../request';

interface IModelSpecificationForm {
    sp_name: string;
    sp_value: string;
    sp_position: number;
    icon?: string | null;
    id?: string | null;
}

interface IModelSpecificationsProps {
    specifications: IModelSpecification[];
    selectedModelCode: string | null;
    updateSpecification: () => void;
}

interface IModelSpecificationsState {
    specificationFields: IModelSpecificationForm;
    editSpecificationFields: any;
    isWorking: boolean;
    addError: Error | null;
    removeError: Error | null;
    confirmRemoveSpIsOpen: boolean;
    specificationDeletedId: string | null;
}

export class ModelSpecifications extends React.Component<IModelSpecificationsProps, IModelSpecificationsState> {
    state: IModelSpecificationsState = {
        specificationFields: {
            sp_name: '',
            sp_value: '',
            sp_position: 0,
            icon: null,
            id: null,
        },
        editSpecificationFields: {},
        isWorking: false,
        addError: null,
        removeError: null,
        confirmRemoveSpIsOpen: false,
        specificationDeletedId: null,
    };
    request = new Request2({ requestConfigs: MODEL_EDITOR_REQUESTS });

    componentDidUpdate(
        prevProps: Readonly<IModelSpecificationsProps>,
        prevState: Readonly<IModelSpecificationsState>,
        snapshot?: any,
    ) {
        if (this.props.selectedModelCode !== prevProps.selectedModelCode) {
            this.clearEditSpecification();
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    onSpecificationInputChange(data: IModelSpecificationForm) {
        this.setState({ specificationFields: data });
    }

    addSpecification() {
        const { specificationFields } = this.state;
        const { selectedModelCode, updateSpecification } = this.props;
        const { sp_name, sp_value, sp_position, icon, id } = specificationFields;

        this.setState({
            isWorking: true,
            addError: null,
        }, () => {

            const data = {
                name: sp_name,
                position: +sp_position,
                value: sp_value,
                model_id: selectedModelCode,
                icon,
                ...id && { id },
            };

            this.request.exec(REQUESTS.ADD_MODEL_SPECIFICATION, { body: data })
                .then(() => {
                    this.setState({ isWorking: false });
                    updateSpecification();
                })
                .catch(addError => {
                    this.setState({ addError, isWorking: false });
                });
        });
    }

    deleteSpecificationItem(specificationDeletedId: string) {
        this.setState({
            confirmRemoveSpIsOpen: true,
            specificationDeletedId,
        });
    }

    onCloseConfirm() {
        this.setState({ confirmRemoveSpIsOpen: false, isWorking: false, removeError: null });
    }

    removeModelCp() {
        const { specificationDeletedId } = this.state;
        const { updateSpecification } = this.props;

        this.setState({
            isWorking: true,
        }, () => {

            const objectModel = { id: specificationDeletedId };

            this.request.exec(REQUESTS.REMOVE_MODEL_SPECIFICATION, { body: objectModel })
                .then(() => {
                    this.onCloseConfirm();
                    this.clearFieldsSpecification();
                    updateSpecification();
                })
                .catch(removeError => {
                    this.setState({ removeError, isWorking: false });
                });
        });
    }

    getEmpty() {
        return {
            sp_name: '',
            sp_value: '',
            sp_position: 0,
            icon: null,
        };
    }

    clearFieldsSpecification() {
        const specificationFields = { ...this.getEmpty() };

        this.setState({ specificationFields });
    }

    clearEditSpecification() {
        this.setState({
            editSpecificationFields: {
                ...this.getEmpty(),
                id: null,
            },
        });
    }

    editSpecificationItem(editSpecificationFields) {
        this.setState({
            editSpecificationFields: {
                ...editSpecificationFields,
                sp_name: editSpecificationFields.name,
                sp_value: editSpecificationFields.value,
                sp_position: editSpecificationFields.position,
                icon: editSpecificationFields.icon,
            },
        });
    }

    render() {
        const {
            isWorking, confirmRemoveSpIsOpen,
            addError, removeError, specificationDeletedId,
        } = this.state;
        const { specifications, selectedModelCode } = this.props;

        return <div className={style.specifications}>
            <h3>Спецификации модели</h3>

            {selectedModelCode
                ? <>
                    {specifications?.length
                        ? <table className={`${styleTable.table} ${style.specifications_table}`}>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Название</th>
                                    <th>Значение</th>
                                    <th>Позиция</th>
                                    <th>icon</th>
                                    <th colSpan={2}/>
                                </tr>
                            </thead>
                            <tbody>
                                {specifications
                                    .map((item, index) => {
                                        return <tr key={item.id}>
                                            <td>{index + 1}</td>
                                            <td>{item.name ?? EMPTY_DATA}</td>
                                            <td>{item.value ?? EMPTY_DATA}</td>
                                            <td>{item.position ?? EMPTY_DATA}</td>
                                            <td>{item.icon ? <img src={item.icon} className={style.icon}/> : ''}</td>
                                            <td>
                                                <Link onClick={this.deleteSpecificationItem.bind(this, item.id)}>
                                                    Del
                                                </Link>
                                            </td>
                                            <td>
                                                <Link onClick={this.editSpecificationItem.bind(this, item)}>Edit</Link>
                                            </td>
                                        </tr>;
                                    })}
                            </tbody>
                        </table>
                        : <div className={style.empty}>Спецификации отсутствуют</div>}
                    <div className={style.specifications_input}>
                        <FormConstructor schema={MODEL_SPECIFICATION_SCHEMA}
                                         initialData={this.state.editSpecificationFields}
                                         onChange={this.onSpecificationInputChange.bind(this)}/>
                    </div>
                    <div className={coreStyle.button_container}>
                        <Button colorType={ButtonTypes.positive}
                                basic
                                isLoading={isWorking}
                                onClick={this.addSpecification.bind(this)}>
                            {this.state.editSpecificationFields?.id ? 'Обновить' : 'Добавить'} спецификацию
                        </Button>
                        <Button onClick={this.clearEditSpecification.bind(this)}>Очистить</Button>
                    </div>
                    {addError
                        ? <SimpleError error={addError} data={{ label: 'Ошибка при добавлении спецификации' }}/>
                        : null}
                </>
                : <div className={style.empty}>Выберите модель</div>}
            {
                confirmRemoveSpIsOpen
                    ? <Confirm accept={this.removeModelCp.bind(this)}
                               onClose={this.onCloseConfirm.bind(this)}
                               error={removeError}
                               isWorking={isWorking}
                               question={`Удалить спецификации модели ${selectedModelCode}: ${
                                   specifications
                                       .filter(item => item.id === specificationDeletedId)?.[0]?.name}?`}/>
                    : null
            }
        </div>;
    }
}
