import * as React from 'react';

import { Dict } from '../../../../types';
import { EMPTY_DATA } from '../../../constants';
import { Button, ButtonTypes } from '../../../ui/Button';
import { Confirm } from '../../../ui/FullModal';
import * as coreStyle from '../../../ui/index.css';
import { Request2 } from '../../../utils/request';
import { FormConstructor } from '../../FormConstructor';
import { controlType } from '../../FormConstructor/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { ITag } from '../../TagModal/component';
import { CLEAR_SCHEMA, MODEL_SCHEMA } from './constants/constants';
import * as style from './index.css';
import { ModelDefaultTags } from './ModelDefaultTags/component';
import { ModelSpecifications } from './ModelSpecifications/component';
import { MODEL_EDITOR_REQUESTS, REQUESTS } from './request';

export interface IModelSpecification {
    name: string;
    position: number;
    value: string;
    id: string;
    icon?: string;
}

interface IModel {
    image_map_url_2x: string;
    name: string;
    fuel_distance_enabled: boolean;
    z_index?: number | number;
    code: string;
    manufacturer: string;
    image_angle_url?: string;
    image_map_url_3x: string;
    registry_manufacturer: string;
    short_name: string;
    image_large_url: string;
    meta?: Dict<any>;
    visual?: {
        background?: {
            gradient?: {
                bottom_color?: string;
                top_color?: string;
            };
        };
        title?: {
            color?: string;
        };
    };
    default_tags: { comment: string; tag_name: string; priority: number }[];
    image_pin_url_3x: string;
    image_small_url: string;
    model_specifications: IModelSpecification[];
    rank?: number;
    fuel_type: string;
    fuel_cap_side: string;
    image_pin_url_2x: string;
    registry_model: string;
    fuel_icon_url?: string;
    image_background_url?: string;
    segment?: string;
    maintenance_period: number;
    maintenance_mileage: number;
    intermediate_maintenance_mileage: number;
    first_maintenance_mileage: number;
}

interface IModelEditorState {
    dataIsLoading: boolean;
    models: IModel[];
    loadingError: Error | null;
    defaultTagsList: { text: string; value: string; description: string }[];
    selectedModelCode: string | null;
    selectedItem: IModel | null;
    configFormData: Dict<any>;
    isWorking: boolean;
    specifications: IModelSpecification[];
    error: Error | null;
    confirmRemoveIsOpen: boolean;
    default_tags: any[];
    isFormValid: boolean;

}

export default class ModelEditor extends React.Component<any, IModelEditorState> {
    state: IModelEditorState = {
        dataIsLoading: false,
        models: [],
        loadingError: null,
        defaultTagsList: [],
        selectedModelCode: null,
        selectedItem: null,
        configFormData: {},
        isWorking: false,
        specifications: [],
        error: null,
        confirmRemoveIsOpen: false,
        default_tags: [],
        isFormValid: false,
    };
    request = new Request2({ requestConfigs: MODEL_EDITOR_REQUESTS });

    componentDidMount() {
        this.getData();
        this.getDefaultTagsList();
        this.updateSpecification();
    }

    getData() {
        this.setState({
            dataIsLoading: true,
            loadingError: null,
        }, () => {
            this.request.exec(REQUESTS.GET_MODELS)
                .then(response => {
                    const models: IModel[] = response.car_models || [];

                    this.setState({
                        models: models,
                        dataIsLoading: false,
                        default_tags: this.state.selectedModelCode
                            && response.car_models
                                .filter(item => item.code === this.state.selectedModelCode)?.[0]?.default_tags || [],
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError, dataIsLoading: false });
                });
        });
    }

    getDefaultTagsList() {
        this.request.exec(REQUESTS.GET_TAGS)
            .then(response => {
                const tags: ITag[] = response.records
                    ? response.records.sort((a: ITag, b: ITag) => a.ts - b.ts)
                    : [];

                const defaultTagsList = tags.map(item => {
                    return { text: item.display_name || item.name, value: item.name, description: item.comment };
                });

                this.setState({ defaultTagsList });
            });
    }

    onSelect(item: IModel) {
        const { code: selectedModelCode } = item;
        this.setState({
            selectedModelCode,
            selectedItem: item,
            specifications: item?.model_specifications
                ?.sort((a, b) => a.position - b.position) || [],
            default_tags: item?.default_tags,
        });
    }

    onChangeForm(configFormData: Dict<any>, isFormValid: boolean) {
        this.setState({ configFormData: configFormData, isFormValid });
    }

    clearFields() {
        const selectedItem: any = CLEAR_SCHEMA;

        this.setState({
            selectedModelCode: '',
            selectedItem,
            specifications: [],
        });
    }

    handleModel() {
        this.setState({
            isWorking: true,
        }, () => {
            this.setModel(this.state.configFormData);
        });
    }

    setModel(objectModel: Dict<any>) {
        const data = Object.assign({}, this.state.configFormData, objectModel);

        return this.request.exec(REQUESTS.UPDATE_MODEL, { body: data })
            .then(() => {
                this.setState({ isWorking: false, selectedModelCode: data.code });
                this.getData();
                this.getDefaultTagsList();
                this.updateSpecification();
            })
            .catch(error => {
                this.setState({ error, isWorking: false });
                throw error;
            });
    }

    openConfirmRemove() {
        this.setState({ confirmRemoveIsOpen: true });
    }

    removeModel() {
        this.setState({
            isWorking: true,
        }, () => {
            const objectModel = { id: [this.state.selectedModelCode] };

            this.request.exec(REQUESTS.REMOVE_MODEL, { body: objectModel })
                .then(() => {
                    this.onCloseModal();
                    this.clearFields();
                    this.getData();
                })
                .catch(error => {
                    this.setState({ error, isWorking: false });
                });
        });
    }

    onCloseModal() {
        this.setState(
            {
                confirmRemoveIsOpen: false,
                isWorking: false,
            },
        );
    }

    updateSpecification() {
        this.state.selectedModelCode
        && this.request
            .exec(REQUESTS.GET_MODELS, { queryParams: { id: this.state.selectedModelCode } })
            .then(response => {
                const spec = response?.car_models?.[0]?.model_specifications;

                this.setState({
                    specifications:
                        spec?.sort((a: IModelSpecification, b: IModelSpecification) => a.position - b.position) || [],
                }, () => {
                    this.getData();
                });
            });
    }

    render() {
        const {
            loadingError,
            dataIsLoading,
            models,
            selectedModelCode,
            selectedItem,
            isWorking,
            error,
            specifications,
            defaultTagsList,
            isFormValid,
        } = this.state;
        if (loadingError) {
            return <SimpleError error={loadingError} data={{ label: 'Ошибка при загрузке моделей' }}/>;
        }

        const schema = Object.assign({}, MODEL_SCHEMA, {
            segment: {
                type: controlType.variants,
                variants: this.props?.modelSegments,
                display_name: 'segment (#DRIVEFRONT-851)',
                required: true,
                order: 23,
            },
        });

        return <div className={style.model_editor}>
            <div className={style.models_list}>
                {
                    dataIsLoading
                        ? <Spin/>
                        : models.map((item) => {
                            return <ModelItem key={item.code}
                                              item={item}
                                              selectedModelCode={selectedModelCode}
                                              onSelect={this.onSelect.bind(this)}/>;
                        })
                }
            </div>
            <div className={style.model_config}>
                <h3>Конфигурация модели</h3>
                <FormConstructor schema={schema}
                                 initialData={selectedItem ?? {}}
                                 onChange={this.onChangeForm.bind(this)}/>
                <div className={coreStyle.button_container}>
                    <Button basic
                            isLoading={isWorking}
                            colorType={ButtonTypes.warning}
                            onClick={this.clearFields.bind(this)}>Очистить</Button>
                    <Button colorType={ButtonTypes.positive}
                            isLoading={this.state.isWorking}
                            disabled={!isFormValid}
                            onClick={this.handleModel.bind(this)}>
                        {this.state.selectedModelCode
                        && this.state.selectedModelCode === this.state.configFormData?.code
                            ? 'Обновить модель'
                            : 'Добавить модель'}</Button>
                    <Button colorType={ButtonTypes.negative}
                            isLoading={isWorking}
                            disabled={
                                !this.state.selectedModelCode
                                || this.state.selectedModelCode !== this.state.configFormData?.code
                            }
                            onClick={this.openConfirmRemove.bind(this)}>Удалить модель</Button>
                </div>
                {
                    this.state.confirmRemoveIsOpen
                    && <Confirm accept={this.removeModel.bind(this)}
                                onClose={this.onCloseModal.bind(this)}
                                error={error}
                                isWorking={isWorking}
                                question={`Удалить модель ${selectedModelCode}?`}/>
                }
            </div>
            <div className={style.model_specifications}>
                <ModelSpecifications specifications={specifications}
                                     selectedModelCode={selectedModelCode}
                                     updateSpecification={this.updateSpecification.bind(this)}/>
                <ModelDefaultTags selectedModelCode={selectedModelCode}
                                  setModel={this.setModel.bind(this)}
                                  defaultTags={this.state.default_tags}
                                  defaultTagsList={defaultTagsList}/>
            </div>
        </div>;
    }
}

interface IModelItemProps {
    item: IModel;
    selectedModelCode: string | null;
    onSelect: () => void;
}

const ModelItem = React.memo((props: IModelItemProps) => {
    const { item, onSelect, selectedModelCode } = props;
    const { code = '', name = '' } = item;

    return <div onClick={onSelect?.bind(null, item)}
                className={`${style.models_editor_item} ${selectedModelCode === code
                    ? style.models_editor_item_selected
                    : ''}`}>
        <div className={style.name} title={name || EMPTY_DATA}>{name || EMPTY_DATA}</div>
        <div className={style.code} title={code || EMPTY_DATA}>{code || EMPTY_DATA}</div>
    </div>;
});
