import * as React from 'react';
import { RouteComponentProps } from 'react-router';

import { Dict } from '../../../../../types';
import { UserInfoHandler } from '../../../../models/user';
import { Button, ButtonTypes, CancelButton, DeleteButton, SaveButton } from '../../../../ui/Button';
import { Confirm, IWithConfirmState, Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import Select from '../../../../ui/Select';
import { ConstantsKey } from '../../../../utils/fetchConstants';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import Spin from '../../../Spin';
import { ProposeCommentModal } from '../../ProposeCommentModal';
import * as style from '../index.css';
import { NOTIFIERS_REQUESTS, REQUESTS } from '../request';

enum ProposeAction {
    REJECT = 'REJECT',
    CONFIRM = 'CONFIRM'
}

interface IConstants {
    constants: Promise<any>;
}

interface INotifierModalModalProps extends IConstants, RouteComponentProps<{ notifier_name: string }> {
    notifierName?: string;
    onClose: (success: boolean, isPropositions?: boolean) => void;
    showPropositions: boolean;
    users: Array<Record<string, any>>;
    getPropositions: () => void;
}

interface INotifierModalModalState extends IWithConfirmState {
    schemas: Dict<any> | null;
    notifier: Dict<any> | null;
    formData: Dict<any> | null;
    isFormValid: boolean;
    isFormChanged: boolean;
    notifierType: string;
    isLoading: boolean;
    loadingError: Error | null;
    showProposeComment: boolean;
    proposeError: Error | null;
    proposeAction: ProposeAction | null;
}

export default class NotifierModal extends React.Component<INotifierModalModalProps, INotifierModalModalState> {
    state: INotifierModalModalState = {
        schemas: null,
        notifier: null,
        formData: null,
        isFormValid: false,
        isFormChanged: false,
        notifierType: '',
        isLoading: false,
        loadingError: null,
        confirmIsOpen: false,
        question: '',
        error: null,
        accept: () => {
        },
        isWorking: false,
        showProposeComment: false,
        proposeError: null,
        proposeAction: null,
    };
    request = new Request2({ requestConfigs: NOTIFIERS_REQUESTS });

    componentDidMount(): void {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<INotifierModalModalProps>): void {
        const notifierName = this.props?.match?.params?.notifier_name
            && decodeURIComponent(this.props?.match?.params?.notifier_name);
        const notifierNamePrev = prevProps?.match?.params?.notifier_name
            && decodeURIComponent(prevProps?.match?.params?.notifier_name);

        if (notifierName !== notifierNamePrev) {
            this.getData();
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getData() {
        const { match, constants, showPropositions } = this.props;
        const notifierName = match?.params?.notifier_name
            && decodeURIComponent(match?.params?.notifier_name);
        this.setState({ isLoading: true, loadingError: null }, () => {
            constants
                .then(response => {
                    const schemas = response[ConstantsKey.IFACE_NOTIFIERS];
                    this.setState({ schemas }, () => {

                        //Get notifier data only if have it's name
                        if (notifierName) {
                            this.request.exec(REQUESTS[showPropositions ? 'GET_PROPOSITIONS' : 'GET_NOTIFIERS'],
                                { queryParams: { ids: [notifierName] } })
                                .then(notifierResponse => {
                                    const allResources = showPropositions
                                        ? notifierResponse?.propositions || []
                                        : notifierResponse?.objects || [];
                                    const notifier = !showPropositions
                                        ? allResources[0]
                                        : allResources.filter((el) => el.name === notifierName)[0];
                                    const notifierType = notifier?.type;
                                    this.setState({ isLoading: false, notifier, notifierType });
                                })
                                .catch(loadingError => {
                                    this.setState({ loadingError, isLoading: false });
                                });
                        } else {
                            this.setState({ isLoading: false });
                        }
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError, isLoading: false });
                });
        });
    }

    onTypeChange(notifierType: string) {
        this.setState({ notifierType });
    }

    onFormChange(formData: Dict<any>, isFormValid: boolean, isFormChanged: boolean) {
        this.setState({ formData, isFormValid, isFormChanged });
    }

    updateNotifier() {
        this.setState({ isWorking: true }, () => {
            const { formData, notifierType } = this.state;
            const data = { objects: [Object.assign({}, formData, { type: notifierType })] };

            this.request.exec(REQUESTS.UPSERT_NOTIFIERS, { body: data })
                .then(() => {
                    this.props.onClose(true);
                })
                .catch(error => {
                    this.setState({
                        error,
                        isWorking: false,
                    });
                });
        });
    }

    onDeleteClick() {
        const name = this.state.notifier?.name;
        this.setState({
            confirmIsOpen: true,
            question: `Удалить оповещатель <b>${name}</b>?`,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    this.request.exec(REQUESTS.REMOVE_NOTIFIERS, { body: { ids: [name] } })
                        .then(() => {
                            this.props.onClose(true);
                        })
                        .catch(error => {
                            this.setState({
                                error,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    closeDeleteModal() {
        this.setState({ confirmIsOpen: false });
    }
    propose() {
        this.setState({
            showProposeComment: true,
            isWorking: false,
        });
    }

    showComment(state) {
        this.setState({
            showProposeComment: state,
        });
    }

    setComment(comment) {
        this.setState({
            isWorking: true,
            proposeError: null,
        }, () => {
            const { formData, notifierType } = this.state;
            const proposeObject = Object.assign({}, formData, { type: notifierType });

            this.request.exec(REQUESTS.NOTIFIERS_PROPOSE, {
                queryParams: {
                    comment: encodeURIComponent(comment),
                },
                body: proposeObject,
            })
                .then(() => {
                    this.setState({
                        isWorking: false,
                        proposeError: null,
                        showProposeComment: false,
                    }, () => {
                        this.props.onClose(true, true);
                    });
                })
                .catch((error) => {
                    this.setState({
                        isWorking: false,
                        proposeError: error,
                    });
                });
        });
    }

    showConfirm(proposeAction: ProposeAction, proposeError) {
        this.setState({
            proposeAction,
            proposeError,
        });
    }

    proposeAccept() {
        this.setState({
            proposeError: null,
            isWorking: true,
        }, () => {
            const { notifier, proposeAction } = this.state;
            const { onClose, getPropositions } = this.props;
            const action = `NOTIFIERS_${proposeAction}`;

            this.request.exec(REQUESTS?.[action], {
                queryParams: {
                    comment: notifier?.proposition_description || action,
                },
                body: {
                    proposition_ids: [notifier?.proposition_id],
                },
            })
                .then(() => {
                    this.setState({
                        proposeError: null,
                        isWorking: false,
                        proposeAction: null,
                    }, () => {
                        onClose(true);
                        getPropositions();
                    });
                })
                .catch((proposeError) => {
                    let error = proposeError;
                    if (error?.data?.error_details?.special_info?.session_info
                        ?.hasOwnProperty('self confirm is denied')) {
                        error = new Error('Нельзя подтверждать свои же предложения :(');
                    }

                    this.setState({
                        proposeError: error,
                        isWorking: false,
                    });
                });
        });
    }

    render() {
        const { onClose, showPropositions, users } = this.props;
        const {
            isLoading, confirmIsOpen, schemas, notifier, notifierType, question,
            error, accept, isWorking, isFormChanged, loadingError, showProposeComment, proposeError,
            proposeAction,
        } = this.state;

        const authorInfo = users.find((el) => el.id === notifier?.proposition_author);
        const author = UserInfoHandler.getPrintName.call(authorInfo);
        const schemaExist = !!schemas?.[notifierType];
        const schema = schemaExist ? schemas?.[notifierType] : null;

        const schemaOptions = schemas && Object.keys(schemas)
            .map(schemaKey => ({ text: schemaKey, value: schemaKey }))
            || [];

        return <Window error={loadingError || error}
                       title={'Изменение нотификатора'}
                       onClose={onClose.bind(this, false)}
                       closeWithConfirm={isFormChanged}>
            {isLoading
                ? <Spin/>
                : <>
                    {showPropositions && <div className={style.loc_item}>
                        <div>
                            <span>
                                Автор
                            </span>: <span className={style.loc_item_value}>
                                <Link href={`#/clients/${notifier?.proposition_author}/info`}
                                      target={'_blank'}>
                                    {author}
                                </Link>
                            </span>
                        </div>
                        <div>
                            <span>
                                Комментарий
                            </span>: <span className={style.loc_item_value}>
                                {notifier?.proposition_description}
                            </span>
                        </div>
                    </div>}
                    {
                        schemas
                        && <Select options={schemaOptions}
                                   placeholder={'Тип оповещателя'}
                                   onSelect={this.onTypeChange.bind(this)}
                                   initialValues={notifierType ? [notifierType] : []}/>
                    }
                    {schema
                        ? <FormConstructor schema={schema}
                                           initialData={notifier ?? null}
                                           onChange={this.onFormChange.bind(this)}/>
                        : null}
                    {
                        showProposeComment
                        && <ProposeCommentModal isWorking={isWorking}
                                                error={proposeError}
                                                isRequired={true}
                                                setComment={this.setComment.bind(this)}
                                                onClose={this.showComment.bind(this, false)}/>
                    }
                    {proposeAction && <Confirm error={proposeError}
                                               isWorking={isWorking}
                                               title={proposeAction}
                                               question={`Выполнить действие?`}
                                               accept={this.proposeAccept.bind(this)}
                                               onClose={this.showConfirm.bind(this, null)}/>
                    }
                    {showPropositions
                        ? <div className={coreStyle.button_container}>
                            <Button colorType={ButtonTypes.negative}
                                    onClick={this.showConfirm.bind(this, ProposeAction.REJECT, null)}
                                    isLoading={isWorking}>
                                Отклонить
                            </Button>
                            <Button colorType={ButtonTypes.positive}
                                    onClick={this.showConfirm.bind(this, ProposeAction.CONFIRM, null)}
                                    isLoading={isWorking}>
                                Принять
                            </Button>
                        </div>
                        : <div className={coreStyle.button_container}>
                            <CancelButton onClick={onClose.bind(this, false)}/>
                            <DeleteButton disabled={!schemaExist || !notifier} onClick={this.onDeleteClick.bind(this)}/>
                            <Button colorType={ButtonTypes.warning}
                                    onClick={this.propose.bind(this)}
                                    isLoading={isWorking}>
                                Предложить
                            </Button>
                            <SaveButton disabled={!schemaExist}
                                        onClick={this.updateNotifier.bind(this, this.state)}/>
                        </div>}
                </>}
            {confirmIsOpen
                ? <Confirm question={question}
                           accept={accept.bind(this)}
                           isWorking={isWorking}
                           error={error}
                           onClose={this.closeDeleteModal.bind(this)}/>
                : null}
        </Window>;
    }
}
