import React, { RefObject } from 'react';
import { Route, Switch } from 'react-router';

import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Input } from '../../../ui/Input';
import { IPlaneTreeItem, VirtualTreeList } from '../../../ui/VirtualTreeList';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import * as style from './index.css';
import NotifierModal from './NotifierModal';
import { NotifierTreeListItem } from './NotifierTreeListItem/component';
import { NOTIFIERS_REQUESTS, REQUESTS } from './request';
import { INotifier, NEW_ITEM_BACK_URL, NEW_ITEM_URL } from './types';

export interface Proposition {
    proposition_author: string;
    proposition_description: string;
    proposition_id: string;
    name: string;
    revision?: number;
    type: string;
    meta?: Record<string,any>;
}

interface INotifiersState {
    notifiers: INotifier[];
    isWorking: boolean;
    dataIsLoading: boolean;
    planeTree: IPlaneTreeItem<INotifier>[];
    error: Error | null;
    filterValue: string;
    showPropositions: boolean;
    propositionIsLoading: boolean;
    propositions?: Proposition[];
    users: Array<Record<string, any>>;
    loadingError: Error | null;
}

export default class Notifiers extends React.Component<{}, INotifiersState> {
    state: INotifiersState = {
        notifiers: [],
        isWorking: false,
        dataIsLoading: false,
        planeTree: [],
        error: null,
        filterValue: '',
        showPropositions: false,
        propositionIsLoading: false,
        users: [],
        loadingError: null,
    };
    request = new Request2({ requestConfigs: NOTIFIERS_REQUESTS });
    listRef: RefObject<HTMLDivElement> | null = null;

    componentDidMount() {
        this.getNotifiers(true);
    }

    getNotifiers(force = false) {
        this.setState({
            error: null,
            dataIsLoading: force,
        }, () => {
            this.request.exec(REQUESTS.GET_NOTIFIERS)
                .then(response => {
                    const notifiers: INotifier[] = response?.objects || [];
                    this.setState({ dataIsLoading: false, notifiers });
                })
                .catch(error => {
                    this.setState({
                        error,
                        dataIsLoading: false,
                    });
                });

        });
    }

    getTree(notifiers: INotifier[]): IPlaneTreeItem<INotifier>[] {
        let types: string[] = notifiers.map(notifier => notifier.type);
        types = Array.from(new Set(types));

        return types.reduce((result: IPlaneTreeItem<INotifier>[], type) => {
            const children: IPlaneTreeItem<INotifier>[] = notifiers?.filter(notifier => notifier.type === type)
                .map(notifier => {
                    return {
                        data: notifier,
                        meta: {
                            children: null,
                            id: `${type}_${notifier.name}`,
                            nestingLevel: 1,
                            active: true,
                        },
                    };
                });

            result.push({
                data: {
                    name: type,
                    type,
                },
                meta: {
                    children,
                    nestingLevel: 0,
                    id: type,
                    active: false,
                },
            });

            return result;
        }, []);
    }

    filterTree() {
        if (this.state.filterValue) {
            return (notifier: IPlaneTreeItem<INotifier>) => {
                return notifier.data.name?.toLowerCase().includes(this.state.filterValue?.toLowerCase());
            };
        }

        return null;

    }

    onFilterChange(filterValue: string) {
        this.setState({ filterValue });
    }

    onAddClick() {
        location.hash = NEW_ITEM_URL;
    }

    closeNotifierModal(success: boolean, isPropositions?: boolean) {
        location.hash = NEW_ITEM_BACK_URL;
        if (success) {
            isPropositions ? this.getPropositions() : this.getNotifiers();
        }
    }

    getPropositions() {
        this.setState({ propositionIsLoading: true }, () => {
            this.request.exec(REQUESTS.GET_PROPOSITIONS)
                .then((response) => {
                    this.setState({
                        propositionIsLoading: false,
                        propositions: response?.propositions || [],
                        users: response?.propositions_users || [],
                    });
                })
                .catch((loadingError) => {
                    this.setState({ loadingError, propositionIsLoading: false });
                });
        });
    }

    onPropositionsChange(showPropositions: boolean) {
        if (!this.state.propositions && showPropositions) {
            this.getPropositions();
        }

        this.setState({ showPropositions, loadingError: null });
    }

    render() {
        const {
            notifiers, dataIsLoading, filterValue, showPropositions, loadingError,
            propositionIsLoading, propositions, users,
        } = this.state;
        const treeBuilder = this.getTree.bind(this, showPropositions ? propositions : notifiers);
        const treeFilter = this.filterTree.bind(this);

        return <div className={style.notifiers}>
            <Switch>
                <Route path={`${NEW_ITEM_URL}:notifier_name?`}
                       render={() => {
                           return <NotifierModal onClose={this.closeNotifierModal.bind(this)}
                                                 showPropositions={showPropositions}
                                                 users={users}
                                                 getPropositions={this.getPropositions.bind(this)}/>;
                       }}/>
            </Switch>
            {dataIsLoading || propositionIsLoading
                ? <Spin/>
                : <div>
                    <div className={style.controls}>
                        <Input value={filterValue}
                               className={style.filter_input}
                               onChange={this.onFilterChange.bind(this)}
                               placeholder={'Фильтр'}
                               disabled={showPropositions}/>
                        <Button basic
                                onClick={this.onAddClick.bind(this)}
                                disabled={showPropositions}>
                            Добавить оповещатель
                        </Button>
                        <Checkbox checked={showPropositions}
                                  className={style.propositions_checkbox}
                                  onChange={this.onPropositionsChange.bind(this)}/>
                        <span>Показать предложения</span>
                    </div>
                    {loadingError
                        ? <SimpleError error={loadingError}/>
                        : <VirtualTreeList treeBuilder={treeBuilder}
                                           treeFilter={treeFilter}
                                           isFilterValueExist={!!filterValue}
                                           treeListItem={NotifierTreeListItem}
                                           showPropositions={showPropositions}/>}
                </div>}
        </div>;
    }
}
