import React from 'react';

import { Dict } from '../../../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../../../constants';
import FormatDate from '../../../../../ui/FormatDate';
import { Link } from '../../../../../ui/Link';
import { TLabel } from '../../../../../ui/Table';
import * as styleTable from '../../../../../ui/Table/index.css';
import { Request2 } from '../../../../../utils/request';
import { SimpleError } from '../../../../SimpleError';
import Spin from '../../../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../../../request';
import { AreaTagsHistory } from '../AreaTagsHistory/component';
import * as style from '../index.css';

interface IAreaHistoryProps {
    areaId: string;
}

interface IAreaHistoryState {
    isLoadingAreaHistory: boolean;
    error: Error | null;
    areaHistory: any[];
}

export class AreaHistory extends React.Component<IAreaHistoryProps, IAreaHistoryState> {
    state: IAreaHistoryState = {
        isLoadingAreaHistory: false,
        error: null,
        areaHistory: [],
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    componentDidMount(): void {
        this.getAreaHistory();
    }

    componentDidUpdate(prevProps: Readonly<IAreaHistoryProps>): void {
        if (this.props.areaId !== prevProps.areaId) {
            this.getAreaHistory();
        }
    }

    getAreaHistory() {
        const areaId = this.props.areaId;

        this.setState({ isLoadingAreaHistory: true }, () => {
            this.request.exec(REQUESTS.AREAS_HISTORY)
                .then(response => {
                    const areaHistory = response.areas_history
                        ? response.areas_history
                            .filter((area: Dict<any>) => area.area_id === areaId)
                        : [];

                    this.setState({
                        isLoadingAreaHistory: false,
                        areaHistory,
                    });
                })
                .catch(error => {
                    this.setState({ isLoadingAreaHistory: false, error });
                });
        });
    }

    render() {
        if (this.state.error) {
            return <SimpleError error={this.state.error} data={{ label: 'Ошибка при загрузке текущих тегов' }}/>;
        }

        const areaHistory = this.state.areaHistory;

        return <div className={style.area_info_modal_item}>
            <h3>История полигона:</h3>
            {this.state.isLoadingAreaHistory
                ? <Spin/>
                : areaHistory && areaHistory.length
                    ? <table className={styleTable.table}>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Дата</th>
                                <th>Действие</th>
                                <th>id/title</th>
                                <th>tags/tooltip</th>
                                <th>Тип полигона</th>
                                <th/>
                            </tr>
                        </thead>
                        <tbody>
                            {
                                areaHistory.map((areaHistoryItem: Dict<any>, index: number) => {
                                    return <tr key={`${areaHistoryItem.area_id}_${areaHistoryItem.revision}`}>
                                        <td>{index + 1}</td>
                                        <td>
                                            <FormatDate value={+areaHistoryItem.history_timestamp * ONE_SECOND}/>
                                        </td>
                                        <td>
                                            <div>
                                                <TLabel text={areaHistoryItem.history_action}
                                                        status={AreaTagsHistory
                                                            .getHistoryStatus(areaHistoryItem.history_action)}/>
                                            </div>
                                        </td>
                                        <td>
                                            <div>{areaHistoryItem.area_id}</div>
                                            <div className={style.secondary}>
                                                {areaHistoryItem.area_title || EMPTY_DATA}
                                            </div>
                                        </td>
                                        <td>
                                            <div>{areaHistoryItem.area_tags}</div>
                                            <div className={style.secondary}>
                                                {areaHistoryItem.area_tooltip || EMPTY_DATA}
                                            </div>
                                        </td>
                                        <td>
                                            {areaHistoryItem.area_type}
                                        </td>
                                        <td>
                                            <Link href={`#/clients/${areaHistoryItem.history_user_id}/info`}>
                                            Пользователь
                                            </Link>
                                        </td>
                                    </tr>;
                                })
                            }
                        </tbody>
                    </table>
                    : <h4 className={style.secondary}>Истории полигона не найдено</h4>}
        </div>;
    }
}
