import React from 'react';

import { Dict } from '../../../../../../types';
import { ONE_SECOND } from '../../../../../constants';
import FormatDate from '../../../../../ui/FormatDate';
import { Link } from '../../../../../ui/Link';
import { LabelStatus, TLabel } from '../../../../../ui/Table';
import * as styleTable from '../../../../../ui/Table/index.css';
import { Request2 } from '../../../../../utils/request';
import { SimpleError } from '../../../../SimpleError';
import Spin from '../../../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../../../request';
import * as style from '../index.css';

interface IAreaTagsHistoryProps {
    areaId: string;
}

interface IAreaTagsHistoryState {
    isLoadingTagHistory: boolean;
    error: Error | null;
    tagsHistory: any[];
}

export class AreaTagsHistory extends React.Component<IAreaTagsHistoryProps, IAreaTagsHistoryState> {
    state: IAreaTagsHistoryState = {
        isLoadingTagHistory: false,
        error: null,
        tagsHistory: [],
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    componentDidMount(): void {
        this.getAreaTagsHistory();
    }

    componentDidUpdate(prevProps: Readonly<IAreaTagsHistoryProps>): void {
        if (this.props.areaId !== prevProps.areaId) {
            this.getAreaTagsHistory();
        }
    }

    getAreaTagsHistory() {
        const areaId = this.props.areaId;

        this.setState({ isLoadingTagHistory: true }, () => {
            this.request.exec(REQUESTS.AREA_TAG_HISTORY, { queryParams: { object_id: areaId } })
                .then(response => {
                    this.setState({
                        isLoadingTagHistory: false,
                        tagsHistory: response.records || [],
                    });
                })
                .catch(error => {
                    this.setState({ isLoadingTagHistory: false, error });
                });
        });
    }

    static getHistoryStatus(action: string) {
        let status = LabelStatus.DEFAULT;
        switch (action) {
        case 'add':
            status = LabelStatus.POSITIVE;
            break;
        case 'remove':
            status = LabelStatus.NEGATIVE;
            break;
        case 'update_data':
            status = LabelStatus.INFO;
            break;
        }

        return status;
    }

    render() {
        if (this.state.error) {
            return <SimpleError error={this.state.error} data={{ label: 'Ошибка при загрузке текущих тегов' }}/>;
        }

        const tagsHistory = this.state.tagsHistory;

        return <div className={style.area_info_modal_item}>
            <h3>История тегов:</h3>
            {this.state.isLoadingTagHistory
                ? <Spin/>
                : tagsHistory && tagsHistory.length
                    ? <table className={styleTable.table}>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Дата</th>
                                <th>Действие</th>
                                <th>Тег</th>
                                <th>Пользователь</th>
                            </tr>
                        </thead>
                        <tbody>
                            {
                                tagsHistory.map((tag: Dict<any>, index) => {
                                    return <tr key={`${tag.tag_id}_${tag.event_id}`}>
                                        <td>{index + 1}</td>
                                        <td>
                                            <FormatDate value={tag.timestamp * ONE_SECOND}/>
                                        </td>
                                        <td>
                                            <TLabel status={AreaTagsHistory.getHistoryStatus(tag.action)}
                                                    text={tag.action}/>
                                        </td>
                                        <td>
                                            <div title={tag.tag_display_name}>
                                                {tag.tag_display_name}
                                            </div>
                                            <div className={style.secondary} title={tag.tag_name}>
                                                {tag.tag_name}
                                            </div>
                                        </td>
                                        <td>
                                            {tag.user_data_full
                                                ? <Link target={'_blank'}
                                                        href={`#/clients/${tag.user_data_full
                                            && tag.user_data_full.id}`}>
                                                    {tag.user_data_full.username}
                                                </Link>
                                                : null}
                                        </td>
                                    </tr>;
                                })
                            }
                        </tbody>
                    </table>
                    : <h4 className={style.secondary}>Истории тегов не найдено</h4>}
        </div>;
    }
}
