import React from 'react';

import { Dict } from '../../../../../../types';
import { Button } from '../../../../../ui/Button';
import { Confirm } from '../../../../../ui/FullModal';
import * as coreStyle from '../../../../../ui/index.css';
import { Link } from '../../../../../ui/Link';
import * as styleTable from '../../../../../ui/Table/index.css';
import { Request2 } from '../../../../../utils/request';
import { SimpleError } from '../../../../SimpleError';
import Spin from '../../../../Spin';
import TagModal from '../../../../TagModal';
import { ITag, OBJECT_TYPES, Operations } from '../../../../TagModal/component';
import { REQUESTS, SETTINGS_REQUESTS } from '../../../request';
import { IPolygonItem } from '../../component';
import * as style from '../index.css';

interface ICurrentAreaTagsProps {
    areaId: string;
}

interface ICurrentAreaTagsState {
    isLoading: boolean;
    error: Error | null;
    area: IPolygonItem | null;
    editingTag: ITag | null;
    confirmIsOpen: boolean;
    question: string;
    isWorking: boolean;
    deleteTag: () => void;
    isAddingTagOpen: boolean;
}

export class CurrentAreaTags extends React.Component<ICurrentAreaTagsProps, ICurrentAreaTagsState> {
    state: ICurrentAreaTagsState = {
        isLoading: false,
        error: null,
        area: null,
        editingTag: null,
        confirmIsOpen: false,
        question: '',
        isWorking: false,
        deleteTag: () => {
        },
        isAddingTagOpen: false,
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    componentDidMount(): void {
        this.getPolyData();
    }

    componentDidUpdate(prevProps: Readonly<ICurrentAreaTagsProps>): void {
        if (this.props.areaId !== prevProps.areaId) {
            this.getPolyData();
        }
    }

    getPolyData() {
        this.setState({ isLoading: true, error: null }, () => {
            this.request.exec(REQUESTS.AREAS_INFO)
                .then(response => {
                    const areaId = this.props.areaId;

                    const area = response.areas
                        && response.areas.filter(area => {
                            return area.area_id === areaId;
                        })[0] || null;

                    this.setState({ isLoading: false, area });
                })
                .catch(error => {
                    this.setState({ isLoading: false, error });
                });
        });
    }

    editTag(editingTag: ITag) {
        this.setState({ editingTag });
    }

    closeEditTag() {
        this.setState({ editingTag: null }, () => {
            this.getPolyData();
        });
    }

    openDeleteTagConfirm(tag: ITag) {
        this.setState({
            confirmIsOpen: true,
            question: `Удалить тег ${tag.tag}?`,
            deleteTag: this.deleteTag.bind(this, tag),
        });
    }

    closeDeleteTagConfirm() {
        this.setState({ confirmIsOpen: false }, () => {
            this.getPolyData();
        });
    }

    deleteTag(tag: ITag) {
        this.setState({
            isWorking: true,
        }, () => {
            this.request.exec(REQUESTS.AREA_TAG_REMOVE, { queryParams: { tag_id: tag.tag_id } })
                .then(() => {
                    this.closeDeleteTagConfirm();
                })
                .catch((error) => {
                    this.setState({ isWorking: false, error });
                });
        });
    }

    openAddTagModal() {
        this.setState({ isAddingTagOpen: true });
    }

    closeAddTagModal() {
        this.setState({ isAddingTagOpen: false });
    }

    render() {
        if (this.state.error) {
            return <SimpleError error={this.state.error} data={{ label: 'Ошибка при загрузке текущих тегов' }}/>;
        }

        const area: IPolygonItem | null = this.state.area;
        const hardTags: ITag[] = area && area.hard_tags || [];

        return <div className={style.area_info_modal_item}>
            <h3>Текущие теги:</h3>
            {this.state.isLoading
                ? <Spin/>
                : hardTags && hardTags.length
                    ? <>
                        <table className={styleTable.table}>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Приоритет</th>
                                    <th>Тег</th>
                                    <th>Информация</th>
                                    <th/>
                                </tr>
                            </thead>
                            <tbody>
                                {
                                    hardTags.map((tagItem: ITag & { infos: Dict<any>[] }, index) => {
                                        return <tr key={tagItem.tag_id}>
                                            <td>{index + 1}</td>
                                            <td>{tagItem.priority}</td>
                                            <td>
                                                <div>
                                                    <Link onClick={this.editTag.bind(this, tagItem)}>
                                                        {tagItem.tag}
                                                    </Link>
                                                </div>
                                                <div className={style.secondary}>{tagItem.comment}</div>
                                            </td>
                                            <td className={style.secondary}>
                                                {
                                                    tagItem.infos && tagItem.infos.map((info, infoItemIndex) => {
                                                        return <div key={infoItemIndex}>
                                                            {info.key}: {info.value}
                                                        </div>;
                                                    })
                                                }
                                            </td>
                                            <td>
                                                <Link onClick={this.openDeleteTagConfirm.bind(this, tagItem)}>
                                                Удалить
                                                </Link>
                                            </td>
                                        </tr>;
                                    })
                                }
                            </tbody>
                        </table>

                    </>
                    : <h4 className={style.secondary}>Текущих тегов не найдено</h4>
            }
            <div className={coreStyle.button_container}>
                <Button basic onClick={this.openAddTagModal.bind(this)}>Добавить тег</Button>
            </div>
            {this.state.editingTag
                ? <TagModal objectId={{ type: OBJECT_TYPES.AREA, id: area && area.area_id || '' }}
                            operation={Operations.EDIT}
                            initialData={this.state.editingTag || {}}
                            onClose={this.closeEditTag.bind(this)}/>
                : null}
            {this.state.confirmIsOpen ?
                <Confirm onClose={this.closeDeleteTagConfirm.bind(this)}
                         error={null}
                         question={this.state.question}
                         accept={this.state.deleteTag}
                         isWorking={this.state.isWorking}/> : null}
            {this.state.isAddingTagOpen
                ? <TagModal objectId={{ type: OBJECT_TYPES.AREA, id: area && area.area_id || '' }}
                            onClose={this.closeAddTagModal.bind(this)}/>
                : null}
        </div>;
    }
}
