import React, { useEffect, useState } from 'react';
import VirtualList, { ItemStyle } from 'react-tiny-virtual-list';

import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { useWindowResize } from '../../../../hooks/useWindowResize';
import FormatDate from '../../../../ui/FormatDate';
import { Input } from '../../../../ui/Input';
import { Link } from '../../../../ui/Link';
import { NoInformation } from '../../../../ui/NoInformation';
import { Request2 } from '../../../../utils/request';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../../request';
import { IPolygonItem } from '../component';
import * as style from '../index.css';

interface IAreasHistoryProps {
    selectItem: (area: IPolygonItem) => void;
}

export class AreasHistory extends React.Component<IAreasHistoryProps, any> {
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });
    state = {
        error: null,
        isLoading: false,
        data: [],
    };

    componentDidMount(): void {
        this.getData();
    }

    getData() {
        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.AREA_HISTORY)
                .then(response => {
                    this.setState({
                        data: response?.areas_history || [],
                        isLoading: false,
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });

    }

    render() {
        return this.state.isLoading
            ? <Spin/> : this.state.error
                ? <SimpleError error={this.state.error}/>
                : this.state.data?.length
                    ? <Items data={this.state.data} selectItem={this.props.selectItem.bind(this)}/> : <NoInformation/>;
    }
}

const Items = React.memo((props: { data: any[]; selectItem: (area: IPolygonItem) => void }) => {
    const itemSize = 84;
    const margin = 280;
    const [filter, onchange] = useState('');
    const [filteredData, setFilter] = useState(props.data);
    useEffect(() => {
        if (filter) {
            setFilter(props.data?.filter(item => {
                return item.area_title.toUpperCase().includes(filter.toUpperCase())
                    || item.area_id.toUpperCase().includes(filter.toUpperCase());
            }));
        } else {
            setFilter(props.data);
        }
    }, [filter]);

    const window = useWindowResize();

    return <>
        <Input onChange={onchange} value={filter} placeholder={'фильтр'}/>
        <VirtualList width={'100%'}
                     height={window.height - margin}
                     itemCount={filteredData.length}
                     itemSize={itemSize}
                     renderItem={({ index, style }) => {
                         const item = filteredData[index];

                         return <Item item={item}
                                      style={style}
                                      key={index}
                                      data={filteredData}
                                      selectItem={props.selectItem.bind(null, item)}/>;
                     }}/>
    </>;
});

const Item = React.memo((props: { item: any; style: ItemStyle; data: any[]; selectItem: () => void }) => {
    return <div style={props.style}
                className={style.history_item}
                onClick={props.selectItem}>
        <div>
            <FormatDate value={
                +props.item.history_timestamp * ONE_SECOND
            }/> : <strong className={style[props.item.history_action]}>
                {props.item.history_action}</strong>
        </div>
        <div> id: <i>{props.item.area_id}</i></div>
        <div className={style.history_title}>{props.item.area_title || EMPTY_DATA}</div>
        <div>
            <Link href={`${location.origin}${location.pathname}#/clients/${props.item.history_user_id}/info`}
                  target={'_blank'}>
                {props.item.history_user_id}
            </Link>
        </div>
    </div>;
});
