import produce from 'immer';
import React from 'react';

import { Button, ButtonTypes } from '../../../../ui/Button';
import { Cross } from '../../../../ui/Cross';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import { FormConstructor } from '../../../FormConstructor';
import { SimpleError } from '../../../SimpleError';
import { ProposeCommentModal } from '../../ProposeCommentModal';
import { REQUESTS, SETTINGS_REQUESTS } from '../../request';
import { IPolygonItem } from '../component';
import { ImportGeoJson } from '../ImportGeoJson';
import * as style from '../index.css';
import { shemaPoly } from '../shema';

interface IEditAreaProps {
    area: IPolygonItem | null;
    onClose: () => void;
    update: () => void;
    getPoly: (area, callback) => void;
}

export class EditArea extends React.Component<IEditAreaProps> {
    state = {
        isWorking: false,
        area: {} as any,
        error: null,

        showProposeComment: false,
        proposeError: null,
        showImport: false,

        initialData: {},
        dataFromGeoJson: false,
    };

    showComment(state) {
        this.setState(produce(this.state, draft => {
            draft.showProposeComment = state;
            if(!state) {
                draft.proposeError = null;
            }
        }));
    }

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    clickInside(e: KeyboardEvent) {
        e.stopPropagation();
    }

    onChange(area) {
        this.setState({
            area,
        });
    }

    format(area) {
        return Object.assign({
            area_id: '',
            area_title: '',
            area_type: 'ordinary',
            revision: '',
            area_tags: '',
            area_index: '',
            area_coords: '',
        }, area, {
            area_tooltip: area && area.area_tooltip && JSON.stringify(area.area_tooltip) || '{}',
            area_details: area && area.area_details && JSON.stringify(area.area_details) || '{}',
            area_coords: area && area.area_coords && area.area_coords.toString().replace(/\,/ig, ' '),
        });
    }

    prepare(callback) {
        this.props.getPoly(this.props.area, (_poly) => {
            this.setState(produce(this.state, draft => {
                draft.isWorking = true;
            }),
            () => {
                const area = Object.assign({}, this.state.area);
                let area_tooltip: any = {};
                let area_details: any = {};
                try {
                    area_tooltip = JSON.parse(this.state.area.area_tooltip);
                    area.area_tooltip = area_tooltip;
                    if (!area.area_tooltip || !Object.keys(area.area_tooltip).length) {
                        delete area['area_tooltip'];
                    }
                } catch (err) {
                    delete area['area_tooltip'];
                }

                try {
                    area_details = JSON.parse(this.state.area.area_details);
                    area.area_details = area_details;
                    if (!area.area_details || !Object.keys(area.area_details).length) {
                        delete area['area_details'];
                    }
                } catch (err) {
                    delete area['area_details'];
                }

                const coordinates = _poly && _poly.geometry && _poly.geometry.getCoordinates();
                if (coordinates && !this.state.dataFromGeoJson) { // Update poly coords
                    area.area_coords = _poly.geometry.getCoordinates().toString().replace(/\,/ig, ' ');
                }

                callback(area);
            });
        });
    }

    save() {
        this.prepare((area) => {
            this.request.exec(REQUESTS.AREA_UPSERT, {
                body: {
                    areas: [area],
                },
            })
                .then(() => {
                    this.setState(produce(this.state, draft => {
                        draft.isWorking = false;
                        draft.error = null;
                    }));
                    this.props.update();
                    this.props.onClose();
                })
                .catch((error) => {
                    this.setState(produce(this.state, draft => {
                        draft.isWorking = false;
                        draft.error = error;
                    }));
                });
        });
    }

    propose(comment) {
        this.prepare((area) => {
            this.request.exec(REQUESTS.AREA_PROPOSE, {
                queryParams: {
                    comment: encodeURIComponent(comment),
                },
                body: area,
            })
                .then(() => {
                    this.setState(produce(this.state, draft => {
                        draft.isWorking = false;
                        draft.error = null;
                    }));
                    this.props.update();
                    this.props.onClose();
                })
                .catch((error) => {
                    this.setState(produce(this.state, draft => {
                        draft.isWorking = false;
                        draft.error = error;
                        draft.proposeError = error;
                    }));
                });
        });
    }

    confirmAction(action) {
        const { proposition_id = '' } = this.props.area || {};
        proposition_id
        && this.setState(produce(this.state, draft => {
            draft.isWorking = true;
        }), () => {
            this.request.exec(REQUESTS[`AREA_${action.toUpperCase()}`], {
                queryParams: {
                    comment: this.props.area && this.props.area.proposition_description,
                },
                body: { proposition_ids: [proposition_id] },
            })
                .then(() => {
                    this.setState(produce(this.state, draft => {
                        draft.isWorking = false;
                        draft.error = null;
                    }));
                    this.props.update();
                    this.props.onClose();
                })
                .catch((error) => {
                    this.setState(produce(this.state, draft => {
                        draft.isWorking = false;
                        draft.error = error;
                    }));
                });
        });
    }

    showGeoJsonDialog(showImport) {
        this.setState({
            showImport,
        });
    }

    componentDidMount(): void {
        this.setState({
            initialData: this.format(this.props.area),
        });
    }

    componentDidUpdate(prevProps: Readonly<IEditAreaProps>, prevState: Readonly<{}>, snapshot?: any): void {
        if (this.props.area && !isObjectEqual(this.props.area, prevProps.area)) {
            this.setState({
                initialData: this.format(this.props.area),
            });
        }
    }

    import(data) {
        const initialData = Object.assign({}, this.state.initialData, { area_coords: data });
        this.setState({
            initialData,
            dataFromGeoJson: true,
        });
    }

    render() {
        return <div className={style.edit_area_dialog_bg} /*onClick={this.props.onClose.bind(this)}*/>
            <div className={style.edit_area_dialog_content} onClick={this.clickInside.bind(this)}>
                <div className={style.edit_area_dialog_content_close}>
                    <Cross onClick={this.props.onClose.bind(this)}/>
                </div>
                {
                    this.state.showImport
                    &&
                    <ImportGeoJson onClose={this.showGeoJsonDialog.bind(this, false)} import={this.import.bind(this)}/>
                }
                {
                    this.state.showProposeComment
                    && <ProposeCommentModal isWorking={this.state.isWorking}
                                            error={this.state.proposeError}
                                            setComment={this.propose.bind(this)}
                                            onClose={this.showComment.bind(this, false)}/>
                }
                <div className={style.edit_area_fc}>
                    {this.state.error && <SimpleError error={this.state.error}/>}
                    <FormConstructor schema={shemaPoly}
                                     onChange={this.onChange.bind(this)}
                                     initialData={this.state.initialData}/>
                    {
                        this.props.area && this.props.area.proposition_id
                            ? <div className={style.edit_controls}>
                                <Button isLoading={this.state.isWorking}
                                        colorType={ButtonTypes.positive}
                                        onClick={this.confirmAction.bind(this, 'confirm')}>Принять</Button>
                                <Button isLoading={this.state.isWorking}
                                        onClick={this.confirmAction.bind(this, 'reject')}
                                        colorType={ButtonTypes.negative}>Отклонить</Button>
                            </div>
                            : <div className={style.edit_controls}>
                                <Button isLoading={this.state.isWorking}
                                        disabled={!this.state.area.area_id}
                                        onClick={this.save.bind(this)}>Сохранить</Button>

                                <Button isLoading={this.state.isWorking}
                                        onClick={this.showGeoJsonDialog.bind(this, true)}
                                        colorType={ButtonTypes.positive}>GeoJSON</Button>

                                <Button isLoading={this.state.isWorking}
                                        disabled={!this.state.area.area_id}
                                        onClick={this.showComment.bind(this, true)}
                                        colorType={ButtonTypes.warning}>Предложить</Button>
                            </div>
                    }
                </div>
            </div>
        </div>;
    }
}
