import React from 'react';

import { EMPTY_DATA } from '../../../../constants';
import { GeoJson, GeoJsonHandler } from '../../../../models/geojson';
import { Button, ButtonTypes } from '../../../../ui/Button';
import Checkbox from '../../../../ui/Checkbox';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Input } from '../../../../ui/Input';
import { Request2 } from '../../../../utils/request';
import { initMap } from '../../../MainMap/utils';
import { REQUESTS, SETTINGS_REQUESTS } from '../../request';
import * as style from '../index.css';

declare let ymaps: any;

interface IImportFewProps {
    onClose: (hasChanged?: boolean) => void;
}

export class ImportFew extends React.Component<IImportFewProps> {
    state: any = {
        isLoading: false,
        data: {} as GeoJson,
        checks: [],
        uploadingProgress: {},
        error: null,
        prefix: "areas_",
        tags: "",
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });
    refAreasPreview: any;

    openFile(files) {
        const reader = new FileReader();
        reader.onload = (event: any) => {
            const contents = event.target.result;
            try {
                const geoJSON: GeoJson = JSON.parse(contents) || {} as GeoJson;
                this.setState({
                    data: geoJSON || {},
                    checks: new Array(GeoJsonHandler.getFeatures.call(geoJSON).length).fill(true),
                }, () => {
                    this.draw(geoJSON);
                });
            } catch (e) {
                console.warn(e);
            }
        };

        reader.onerror = (event: any) => {};

        reader.readAsText(files[0]);
    }

    draw(data) {
        this.refAreasPreview
    && this.refAreasPreview.current
    && this.refAreasPreview.current.drawPoly(data);
    }

    componentDidMount(): void {
        this.refAreasPreview = React.createRef();
    }

    onChange(index, value) {
        const checks: boolean[] = this.state.checks || [];
        checks[index] = value;
        this.setState({
            checks,
        }, () => {
            const newPoly = JSON.parse(JSON.stringify(this.state.data));
            const features = newPoly.features.filter((el, index) => {
                return this.state.checks[index];
            });
            newPoly.features = features;
            this.draw(newPoly);
        });
    }

    import() {
        this.setState({ isLoading: true }, () => {

            const areas: any[] = [];
            GeoJsonHandler.getFeatures.call(this.state.data).forEach((_, index) => {
                if (this.state.checks[index]) {
                    const description = GeoJsonHandler.getDescription.call(this.state.data, index);
                    const title = GeoJsonHandler.getName.call(this.state.data, index);
                    const area_title = description !== EMPTY_DATA ? description : title + "_" + index;
                    const area_id = this.state.prefix + GeoJsonHandler.getId.call(this.state.data, index);
                    const area_tags = this.state.tags;
                    areas.push({
                        area_id,
                        area_title,
                        area_tags,
                        area_type: "ordinary",
                        "area_coords": GeoJsonHandler.getPlainCoordinates.call(this.state.data, index),
                    });
                }
            });

            this.request.exec(REQUESTS.AREA_UPSERT, { body: { areas } })
                .then(() => {
                    this.setState({ isLoading: false });
                    this.props.onClose(true);
                })
                .catch((error) => {
                    this.setState({ isLoading: false, error });
                });
        });
    }

    changeInput(type, value) {
        this.setState({
            [type]: value,
        });
    }

    render() {
        const count = this.state.checks.filter(el => el).length;

        return <Window title={"Импорт пакета полигонов"}
                       onClose={this.props.onClose.bind(this)}
                       error={this.state.error}>
            <div className={style.export_wrap}>
                <div className={style.export_controls}>
                    <Button onClick={this.openFile.bind(this)} file>geoJSON</Button>
                    <div className={style.batch_poly}>
                        {
                            GeoJsonHandler.getFeatures.call(this.state.data).map((geo, index) => {
                                const description = GeoJsonHandler.getDescription.call(this.state.data, index);
                                const name = GeoJsonHandler.getName.call(this.state.data) + "_" + index;
                                const id = GeoJsonHandler.getId.call(this.state.data, index);

                                return <div key={index} className={style.batch_poly_item}>
                                    <Checkbox checked={this.state.checks[index]}
                                              onChange={this.onChange.bind(this, index)} />
                                    <span className={style.area_id_label}> {id}: </span> {
                                        description !== EMPTY_DATA
                                            ? description
                                            : name
                                    }
                                </div>;
                            })
                        }
                    </div>
                    <Input value={this.state.prefix}
                           onChange={this.changeInput.bind(this, "prefix")}
                           placeholder={"префикс для группы полигонов"} />
                    <Input value={this.state.tags}
                           onChange={this.changeInput.bind(this, "tags")}
                           placeholder={"теги через зпт"} />
                    <div className={`${coreStyle.full_width} ${coreStyle.button_container}`}>
                        <Button colorType={ButtonTypes.negative} onClick={this.props.onClose.bind(this, false)}>
                            Закрыть
                        </Button>
                        <Button colorType={ButtonTypes.positive}
                                onClick={this.import.bind(this)}
                                isLoading={this.state.isLoading}
                                disabled={!count}>Импорт ({count})</Button>
                    </div>
                </div>
                <div className={style.poly_export_map}>
                    <AreasPreview ref={this.refAreasPreview} />
                </div>
            </div>
        </Window>;
    }
}

class AreasPreview extends React.Component<any, any> {
    map: any;
    objectManager: any;

    componentDidMount(): void {
        initMap("areas_preview", (map) => {
            this.map = map;
            this.map.options.set("autoFitToViewport", "always");
            this.objectManager = new ymaps.ObjectManager();
            this.map.geoObjects.add(this.objectManager);
        });
    }

    drawPoly(geoJson: any) {
        this.objectManager && this.objectManager.removeAll();
        geoJson && geoJson.features.forEach(function(obj) {
            obj.properties.balloonContent = obj.properties.description;
            obj.properties.hintContent = obj.properties.description;
            if (obj.properties.iconCaption) {
                obj.options = {
                    preset: "islands#greenDotIconWithCaption",
                };
            }
        });
        geoJson && this.objectManager?.add(geoJson);
    }

    componentWillUnmount(): void {
        this.map && this.map.destroy();
    }

    render() {
        return <div id={"areas_preview"} className={style.map} />;
    }
}
