import produce from 'immer';
import React from 'react';

import { GeoJson, GeoJsonHandler } from '../../../../models/geojson';
import { Button, ButtonTypes } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import TextArea from '../../../../ui/TextArea';
import { SimpleError } from '../../../SimpleError';

interface IImportGeoJsonProps {
    onClose: () => void;
    import: (data) => void;
}

interface IImportGeoJsonState {
    isWorking: boolean;
    error: Error | null;
    data: GeoJson;
}

export class ImportGeoJson extends React.Component<IImportGeoJsonProps, IImportGeoJsonState> {
    state: IImportGeoJsonState = {
        isWorking: false,
        error: null,
        data: {} as GeoJson,
    };

    openFile(files) {
        this.setState(produce(this.state, draft => {
            draft.isWorking = true;
            draft.error = null;
        }));

        const reader = new FileReader();
        reader.onload = (event: any) => {
            const contents = event.target.result;
            this.setState(produce(this.state, draft => {
                draft.isWorking = false;
                draft.error = null;
                try {
                    draft.data = JSON.parse(contents);
                } catch (e) {
                    draft.error = e;
                }
            }));
        };

        reader.onerror = (event: any) => {
            this.setState(produce(this.state, draft => {
                draft.isWorking = false;
                draft.error = event.target;
            }));
        };

        reader.readAsText(files[0]);
    }

    import() {
        this.props.import(GeoJsonHandler.getPlainCoordinates.call(this.state.data));
        this.props.onClose();
    }

    render() {
        return <Window onClose={this.props.onClose.bind(this)} title={'Импорт GeoJson'}>
            {this.state.error && <SimpleError error={this.state.error}/>}
            <div>name: <strong>{GeoJsonHandler.getName.call(this.state.data)}</strong></div>
            <div>description: <strong>{GeoJsonHandler.getDescription.call(this.state.data)}</strong></div>
            <TextArea value={GeoJsonHandler.getPlainCoordinates.call(this.state.data)}
                      disabled={true}
                      onChange={() => {
                      }}
                      placeholder={'coordinates'}/>
            <div className={`${coreStyle.button_container} ${coreStyle.full_width}`}>
                <Button onClick={this.openFile.bind(this)} isLoading={this.state.isWorking} file>файл</Button>
                <Button onClick={this.import.bind(this)}
                        isLoading={this.state.isWorking}
                        disabled={!this.state.data}
                        colorType={ButtonTypes.positive}>import</Button>
            </div>
        </Window>;
    }
}
