import * as React from 'react';
import { useState } from 'react';

import { IRequestQueueItem, useRequestQueueHandler } from '../../../../../hooks/useRequestQueueHandler';
import { Button } from '../../../../../ui/Button';
import { Confirm, Window } from '../../../../../ui/FullModal';
import { Link } from '../../../../../ui/Link';
import { Request2 } from '../../../../../utils/request';
import { ProgressBar } from '../../../../ProgressBar';
import Spin from '../../../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../../../request';
import { IPolygonItem } from '../../component';
import style from './index.css';

const request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

interface IModalWindow {
    onClose: () => void;
    isWorking: boolean;
    areas: IPolygonItem[];
    updateAreas: () => void;
}

interface ITableObject {
    [key: string]: number[];
}

export const ModalWindow = (props: IModalWindow) => {
    const { onClose, isWorking, areas, updateAreas } = props;
    const [showConfirmIsOpen, setShowConfirmIsOpen] = React.useState(false);
    const [question, setQuestion] = React.useState('');
    const [stateTableObject, setStateTableObject] = React.useState<ITableObject>({});
    const [tagsCounter, setTagsCounter] = React.useState(0);
    const [tagsArrayForDelete, setTagsArrayForDelete] = React.useState([]);
    const [requestOptions, setRequestOptions] = useState<IRequestQueueItem[]>([]);

    React.useEffect(() => {
        const tableObject: ITableObject = {};
        setTagsCounter(0);
        areas.map(area => {
            area.hard_tags?.map?.(tag => {
                setTagsCounter(prev => prev += 1);
                if (tableObject.hasOwnProperty(tag.tag)) {
                    tableObject[tag.tag].push(tag.tag_id);
                } else {
                    tableObject[tag.tag] = [tag.tag_id];
                }
            });
        });
        setStateTableObject(tableObject);
    }, [areas]);

    const openConfirm = (tagName, tagsArray) => {
        setQuestion(tagName);
        setShowConfirmIsOpen(true);
        setTagsArrayForDelete(tagsArray);
    };

    const [isLoading, response, , makeResponse, progressBarObj] = useRequestQueueHandler(request, requestOptions);

    const deleteTags = () => {
        setRequestOptions([]);
        setShowConfirmIsOpen(false);
        tagsArrayForDelete.map((tagId, index) => {
            setRequestOptions(prev => {
                return [
                    ...prev,
                    {
                        requestName: REQUESTS.AREA_TAG_REMOVE,
                        requestOptions: {
                            queryParams: { tag_id: tagId },
                        },
                    },
                ];
            });
        });

        makeResponse();
    };

    React.useEffect(() => {
        if (response) {
            updateAreas();
        }
    }, [response]);

    return (
        <Window onClose={onClose}
                title={'Пакетная обработка полигонов'}>
            <div>
                <div>
                    Получено <b>{tagsCounter}</b> тег(а/ов) по <b>{areas.length}</b> полигонам
                </div>
                <ProgressBar allLength={progressBarObj?.allLength}
                             successLength={progressBarObj?.successLength}
                             errors={progressBarObj?.errors}/>
                {
                    isWorking || isLoading
                        ? <div><Spin/></div>
                        : <table>
                            <thead>
                                <tr>
                                    <th>тег</th>
                                    <th>удалить<br/>тег</th>
                                </tr>
                            </thead>
                            <tbody>
                                {
                                    Object.entries(stateTableObject)
                                        .map(item => {
                                            const [tagName, tagsArray] = item;

                                            return <tr key={tagName}>
                                                <td className={style.tagName}>{tagName}</td>
                                                <td>
                                                    <Link onClick={() => openConfirm(tagName, tagsArray)}>
                                                        {tagsArray.length}
                                                    </Link>
                                                </td>
                                            </tr>;
                                        })
                                }
                            </tbody>
                        </table>
                }
                <Button isLoading={isWorking || isLoading}
                        onClick={onClose}
                        className={style.closeButton}>
                    Закрыть
                </Button>
                {
                    showConfirmIsOpen
                    && <Confirm accept={deleteTags}
                                onClose={() => setShowConfirmIsOpen(false)}
                                isWorking={isWorking}
                                question={`Снять c полигонов тег ${question}`}
                                error={null}/>
                }
            </div>
        </Window>
    );
};
