import produce from 'immer';
import React from 'react';

import * as mapStyle from '../../../MainMap/index.css';
import { fillColor, initMap, stroke } from '../../../MainMap/utils';
import { WIDTH_MAP_CONTROL } from '../../../Map2/helpers/widthMapControl';
import Spin from '../../../Spin';
import { IPolygonItem } from '../component';

declare let ymaps: any;

interface IPolyMapProps {
    updateDrawingNewPoly: Function;
    startDrawingNewPoly: Function;
    cancelDrawingNewPoly: Function;
    areas: IPolygonItem[];
}

export class PolyMap extends React.Component<IPolyMapProps> {
    map: any;
    polygonsCollection: any;
    myNewCollection: any;
    newPolygon: any;

    state = {
        mapIsLoading: false,
    };

    filterPoly(filter: string) {
        this.polygonsCollection.each((poly: any) => {

            const matchId = poly.options.get('area_id')
                && (poly.options.get('area_id').toUpperCase()).includes(filter.toUpperCase());

            const matchTitle = poly.options.get('area_title')
                && (poly.options.get('area_title').toUpperCase()).includes(filter.toUpperCase());

            const matchTags = poly.options.get('area_tags')
                && (poly.options.get('area_tags').toUpperCase()).includes(filter.toUpperCase());

            if (!filter) {
                poly.options.set({
                    strokeWidth: 1,
                    strokeColor: stroke(poly.options.get('area_tags')),
                    fill: true,
                });
            } else if (matchId || matchTags || matchTitle) {
                poly.options.set({
                    strokeWidth: 3,
                    strokeColor: '#ff0000',
                    fill: true,
                });
            } else {
                poly.options.set({
                    strokeWidth: 0,
                    strokeColor: '#000000FF',
                    fill: false,
                });
            }
        });
    }

    showCollections(item, options: { edit?: boolean }) {
        this.polygonsCollection.each((poly: any) => {
            if (poly.options.get('area_id') === item.area_id + '_' + item.proposition_id) {
                poly.options.set({
                    strokeWidth: 4,
                    strokeColor: '#ff7000',
                });
                poly.geometry?._bounds && this.map.setBounds(poly.geometry._bounds);
                if (options && options.edit) {
                    poly.editor.startEditing();
                }
            } else {
                poly.editor.stopDrawing();
                poly.editor.stopEditing();
                poly.options.set({
                    strokeWidth: 1,
                    strokeColor: stroke(poly.options.get('area_tags')),
                });
            }
        });
    }

    getPoly(item, callback) {
        let find = false;
        item && this.polygonsCollection.each((poly: any) => {
            if (poly.options.get('area_id') === item.area_id + '_' + item.proposition_id) {
                callback(poly);
                find = true;
            }
        });
        if (!find) {
            callback({});
        }
    }

    createNewPoly(e?: KeyboardEvent) {
        e && e.preventDefault();
        this.myNewCollection && this.myNewCollection.removeAll();

        this.newPolygon = new ymaps.Polygon([], {}, {
            // Курсор в режиме добавления новых вершин.
            editorDrawingCursor: 'crosshair',
            // Максимально допустимое количество вершин.
            // editorMaxPoints: 5,
            // minPoints: 3,
            // Цвет заливки.
            fillColor: '#00FF0000',
            // Цвет обводки.
            strokeColor: '#0000FF',
            // Ширина обводки.
            strokeWidth: 2,
            draggable: true,
            zIndex: 2,
        });
        // Добавляем многоугольник на карту.
        this.map && this.myNewCollection && this.myNewCollection.add(this.newPolygon);

        // В режиме добавления новых вершин меняем цвет обводки многоугольника.
        const stateMonitor = new ymaps.Monitor(this.newPolygon.editor.state);
        stateMonitor.add('drawing', (newValue: any) => {
            this.newPolygon.options.set('strokeColor', newValue ? '#FF0000' : '#0000FF');
        });

        this.newPolygon.editor.events.add(['vertexadd', 'vertexdragend'], (e) => {
            this.props.updateDrawingNewPoly(this.newPolygon?.geometry?.getCoordinates());
        });
        this.props.startDrawingNewPoly();
        this.newPolygon.editor.startDrawing();
    }

    clearNewPoly(e?: KeyboardEvent) {
        e && e.preventDefault();
        this.newPolygon?.editor?.stopDrawing();
        this.myNewCollection?.removeAll();
        this.props.cancelDrawingNewPoly();
    }

    loadMap() {
        this.setState(produce(this.state, draft => {
            draft.mapIsLoading = true;
        }), () => {
            initMap('polyMap', (map) => {
                this.setState(produce(this.state, draft => {
                    draft.mapIsLoading = false;
                }), () => {
                    this.map = map;
                    this.polygonsCollection = new ymaps.GeoObjectCollection();
                    this.map.geoObjects.add(this.polygonsCollection);

                    this.myNewCollection = new ymaps.GeoObjectCollection();
                    this.map.geoObjects.add(this.myNewCollection);

                    const addNewPoly = new ymaps.control.Button({
                        data: { content: '+ полигон' },
                        options: {
                            maxWidth: WIDTH_MAP_CONTROL,
                        },
                    });
                    addNewPoly.events
                        .add('select', (event: any) => {
                            this.createNewPoly();
                            addNewPoly.deselect();
                        });

                    this.map.controls.add(addNewPoly, { float: 'left' });
                    this.map.controls.add(new ymaps.control.SearchControl(), { float: 'right' });
                    this.map.controls.add(new ymaps.control.TypeSelector(), { float: 'right' });

                    this.drawPoly();
                });
            });
        });
    }

    makePoly(item, options) {
        const proposition = item.proposition_id ? `propose: ${item.proposition_description};` : '';

        return new ymaps.Polygon([item.area_coords], {
            hintContent: `${proposition} ${item.area_title || item.area_id} - ${item.area_tags}`,
        }, Object.assign({},
            {
                fillColor: fillColor({ area_tags: item.area_tags }),
                strokeWidth: 1,
                strokeColor: stroke(item.area_tags),
                area_id: item.area_id + '_' + item.proposition_id,
                revision: item.revision,
                area_tags: item.area_tags,
                area_title: item.area_title,
                area_index: item.area_index || 0,
                area_type: item.area_type || '',
                area_tooltip: item.area_tooltip && JSON.stringify(item.area_tooltip) || '{}',
                area_details: item.area_details && JSON.stringify(item.area_details) || '{}',
            }
            , options),
        );
    }

    drawPoly() {
        this.props.areas.map((item: any, index: number) => {
            this.polygonsCollection.add(this.makePoly(item, {}));
            if (item && item.propositions) {
                item.propositions.forEach((_p) => {
                    this.polygonsCollection.add(this.makePoly(_p, {
                        strokeStyle: {
                            style: 'dot',
                            offset: 40,
                            strokeWidth: 4,
                        },
                    }));
                });
            }
        });
    }

    componentDidMount(): void {
        this.loadMap();
        window.addEventListener('keyup', this.keyUp.bind(this));
    }

    componentDidUpdate(prevProps: IPolyMapProps): void {
        if (this.props.areas.length !== prevProps.areas.length) {
            this.map && this.polygonsCollection && this.polygonsCollection.removeAll();
            this.map && this.drawPoly();
        }
    }

    keyUp(e: KeyboardEvent) {
        if (e.key === 'Escape') {
            if (confirm('Закончить редактирование')) {
                this.cancelEditing();
            }
        }
    }

    cancelEditing() {
        this.polygonsCollection?.each((poly: any) => {
            poly.editor.stopDrawing();
            poly.editor.stopEditing();
        });
        this.clearNewPoly();
    }

    componentWillUnmount(): void {
        this.map && this.map.destroy();
        window.removeEventListener('keyup', this.keyUp.bind(this));
    }

    render() {
        return <div className={mapStyle.map} id={'polyMap'}>
            {
                this.state.mapIsLoading && <Spin/>
            }
        </div>;
    }
}
