import React from 'react';
import { Route, Switch } from 'react-router';
import XLSX from 'xlsx';

import { Dict } from '../../../../types';
import { ONE_SECOND } from '../../../constants';
import { Button, ButtonTypes } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import FormatDate from '../../../ui/FormatDate';
import { Confirm } from '../../../ui/FullModal';
import * as coreStyle from '../../../ui/index.css';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import { TabItem, Tabs } from '../../../ui/Tabs';
import { readTable } from '../../../utils/readTable';
import { Request2 } from '../../../utils/request';
import { buttonNameDetails } from '../../../utils/sendLogs/eventTypes/buttonDetails';
import { deepCopy, download } from '../../../utils/utils';
import { IStore } from '../../App/store';
import { IProgressBarProps } from '../../ProgressBar/types';
import Spin from '../../Spin';
import TagModal from '../../TagModal';
import { OBJECT_TYPES } from '../../TagModal/component';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import { AddTagsToAreas } from './AddTagsToAreas';
import AreaInfoModal from './AreaInfoModal/';
import { AreasHistory } from './AreasHistory';
import { EditArea } from './EditArea';
import { ImportFew } from './ImportFew';
import * as style from './index.css';
import { PackageDeleteTags } from './PackageDeleteTags';
import { PolyMap } from './PolyMap';

const DEFAULT_TAB = 'areas';
const FILTER_DELAY = 200;

type IPolygonsProps = IStore

enum TabItemLink {
    areas = 'areas',
    propositions = 'propositions',
    history = 'history'
}

interface IPolygonsState {
    currentTab: string;
    polyIsLoading: boolean;
    polyError: any;
    propositions: any[];
    areas: IPolygonItem[];
    tabs: TabItem<TabItemLink>[];
    filter: string;

    showEditDialog: boolean;
    editablePoly: IPolygonItem | null;
    showAllPolygons: boolean;
}

export interface IPolygonItem {
    area_id: string;
    revision: string;
    area_title: string;
    area_coords: number[][];
    area_index: number;
    area_tags: string;
    area_tooltip?: {
        url: string;
        title: string;
        message: string;
    };
    area_details: any;
    area_type: string;
    hard_tags: any[];

    propositions?: any[];
    confirmations?: any[];
    confirmation_count?: number;
    confirmation_need?: number;
    proposition_description?: string;
    proposition_id?: string;
}

export class Polygons extends React.Component<IPolygonsProps, IPolygonsState> {
    state: IPolygonsState = {
        currentTab: DEFAULT_TAB,
        polyIsLoading: false,
        polyError: null,
        propositions: [],
        areas: [],
        tabs: [
            { link: TabItemLink.areas, name: 'Список' },
            { link: TabItemLink.propositions, name: 'Предложения' },
            { link: TabItemLink.history, name: 'История' },
        ],

        showEditDialog: false,
        editablePoly: null,
        filter: '',
        showAllPolygons: false,
    };
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });
    polyRef: any = null;

    constructor(props: any) {
        super(props);
        this.polyRef = React.createRef();
    }

    selectTab(currentTab) {
        this.setState({
            currentTab,
        });
    }

    componentDidMount(): void {
        this.getData();
    }

    getData() {
        this.setState({
            polyIsLoading: true,
            polyError: null,
        }, () => {
            this.request.exec(REQUESTS.AREAS_INFO)
                .then(response => {

                    const areas = response.areas || [];
                    const propositions = response.propositions || []; //new propositions without poly
                    const tabs = this.state.tabs;
                    tabs.forEach(tab => {
                        if (tab.link === TabItemLink.areas) {
                            tab.name = `Список (${areas.length})`;
                        }

                        if (tab.link === TabItemLink.propositions) {
                            const propose = areas.filter(a => a.hasOwnProperty(TabItemLink.propositions)) || [];
                            const _pLength = propose.reduce((_p, _c) => {
                                return _p + (_c.propositions && _c.propositions.length);
                            }, 0);
                            tab.name = `Предложения (${propositions.length + _pLength})`;
                        }
                    });
                    this.setState({
                        polyIsLoading: false,
                        polyError: null,
                        areas,
                        propositions,
                    }, () => {
                        this.closeEditing();
                    });
                })
                .catch(polyError => {
                    this.setState({
                        polyIsLoading: false,
                        polyError,
                    });
                });
        });
    }

    selectItem(area) {
        this.polyRef.current.showCollections(area);
    }

    selectAndEditItem(area) {
        this.setState({
            showEditDialog: true,
            editablePoly: area,
        }, () => {
            this.polyRef.current.showCollections(area, { edit: true });
        });
    }

    closeEditing() {
        this.polyRef.current.cancelEditing();
    }

    getPoly(area, callback) {
        return this.polyRef.current.getPoly(area, callback);
    }

    onChangeFilter(filter: string) {
        this.setState({ filter });

        return this.polyRef.current.filterPoly(filter);
    }

    updateDrawingNewPoly(coordinates) {
        const editablePoly = deepCopy(this.state.editablePoly || {});
        editablePoly.area_coords = coordinates;
        this.setState({
            editablePoly,
        });
    }

    startDrawingNewPoly() {
        this.showEditDialog(true);
    }

    cancelDrawingNewPoly() {
        this.showEditDialog(false);
        this.setState(() => ({
            editablePoly: null,
        }));
    }

    showEditDialog(showEditDialog) {
        this.setState(() => ({
            showEditDialog,
        }));
    }

    setShowAllPolygons() {
        this.setState({ showAllPolygons: !this.state.showAllPolygons });
    }

    render() {
        const {
            tabs, currentTab, showEditDialog, editablePoly,
            areas, polyIsLoading, filter, showAllPolygons,
        } = this.state;

        //support 2 schemes for poly propositions
        //TODO: keep right variant after few updates
        const propositions = this.state.propositions.reduce((_p, _c) => {
            if (_c.hasOwnProperty(TabItemLink.propositions)) {
                _p = [..._p, ..._c.propositions.filter(p => p.proposition_id)];
            }

            if (_c.hasOwnProperty('proposition_id')) {
                _p = [..._p, _c];
            }

            return _p;
        }, []);

        return <div className={style.component}>
            <div className={style.menu}>
                <Tabs tabs={tabs} selectTab={this.selectTab.bind(this)} currentTab={currentTab}/>
                {

                    showEditDialog
                    && <EditArea area={editablePoly}
                                 update={this.getData.bind(this)}
                                 getPoly={this.getPoly.bind(this)}
                                 onClose={this.showEditDialog.bind(this, false)}/>

                }
                {
                    currentTab === TabItemLink.areas
                        ? <Areas areas={[...areas]}
                                 isLoading={polyIsLoading}
                                 selectAndEditItem={this.selectAndEditItem.bind(this)}
                                 closeEditing={this.closeEditing.bind(this)}
                                 update={this.getData.bind(this)}
                                 getPoly={this.getPoly.bind(this)}
                                 onChangeFilter={this.onChangeFilter.bind(this)}
                                 selectItem={this.selectItem.bind(this)}
                                 showAllPolygons={showAllPolygons}
                                 setShowAllPolygons={this.setShowAllPolygons.bind(this)}
                                 filter={filter}/>
                        : currentTab === TabItemLink.history
                            ? <AreasHistory selectItem={this.selectItem.bind(this)}/>
                            : currentTab === TabItemLink.propositions
                                ? <Areas areas={[...propositions]}
                                         isLoading={polyIsLoading}
                                         selectAndEditItem={this.selectAndEditItem.bind(this)}
                                         closeEditing={this.closeEditing.bind(this)}
                                         update={this.getData.bind(this)}
                                         getPoly={this.getPoly.bind(this)}
                                         onChangeFilter={this.onChangeFilter.bind(this)}
                                         selectItem={this.selectItem.bind(this)}
                                         showAllPolygons={showAllPolygons}
                                         setShowAllPolygons={this.setShowAllPolygons.bind(this)}
                                         filter={filter}/>
                                : null
                }
            </div>
            <PolyMap areas={filter || showAllPolygons ? [...propositions, ...areas] : []}
                     startDrawingNewPoly={this.startDrawingNewPoly.bind(this)}
                     updateDrawingNewPoly={this.updateDrawingNewPoly.bind(this)}
                     cancelDrawingNewPoly={this.cancelDrawingNewPoly.bind(this)}
                     ref={this.polyRef}/>
        </div>;
    }
}

export interface IAreasProps {
    areas: IPolygonItem[];
    isLoading: boolean;
    selectItem: (area: IPolygonItem) => void;
    selectAndEditItem: (area: IPolygonItem) => void;
    closeEditing: () => void;
    update: () => void;
    getPoly: () => void;
    onChangeFilter: (filter: string) => void;
    filter: string;
    showAllPolygons: boolean;
    setShowAllPolygons: () => void;
}

const HEADER_LENGTH = 1;
const ID_KEY = 'area_id';
const TAG_KEY = 'hard_tags';

class Areas extends React.Component<IAreasProps> {
    state = {
        selectedItem: '',
        showEditDialog: false,
        editableArea: null,
        attachData: null as any,
        attachDataNoDuplicates: null as any,
        showDuplicateConfirm: false,
        filteredAreas: [] as IPolygonItem[],
        selectedTag: {} as Dict<any>,
        addTagToPolygons: false,
        progressBarObj: {} as IProgressBarProps,
        warning: '',
        showConfirm: false,
        isWorking: false,
        accept: () => {},
        question: '',
        error: null,
        showImportFew: false,
        exportGeoJson: [],
    };
    filterTimer: any;
    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    selectItem(area: IPolygonItem) {
        this.setState({
            selectedItem: `${area.area_id}_${area.proposition_id}`,
        });
        this.props.selectItem(area);
    }

    onChangeFilter(filter, onToggle = false) {
        const { areas, onChangeFilter, showAllPolygons: propsShow } = this.props;
        const showAllPolygons = onToggle ? !propsShow : propsShow;

        this.setState({
            filteredAreas: filter
                ? areas.filter(area => this.filterAreas(area, filter))
                : showAllPolygons ? areas : [],
        }, () => {
            clearTimeout(this.filterTimer);
            this.filterTimer = setTimeout(() => {
                onChangeFilter(filter);
            }, FILTER_DELAY);
        });
    }

    setShowAllPolygons() {
        const { filter, setShowAllPolygons } = this.props;

        setShowAllPolygons();
        !filter && this.onChangeFilter(filter, true);
    }

    componentDidMount() {
        const { areas, filter, showAllPolygons } = this.props;

        if (areas?.length && (filter || showAllPolygons)) {
            this.setState({
                filteredAreas: areas.filter(area => this.filterAreas(area, filter)),
            });
        }
    }

    componentDidUpdate(prevProps: Readonly<IAreasProps>, prevState: Readonly<{}>, snapshot?: any) {
        const { areas, filter, showAllPolygons } = this.props;

        if(areas !== prevProps.areas && (filter || showAllPolygons)) {
            this.setState({
                filteredAreas: areas.filter(area => this.filterAreas(area, filter)),
            });
        }
    }

    componentWillUnmount(): void {
        clearTimeout(this.filterTimer);
    }

    onEdit(area) {
        this.setState({
            selectedItem: area.area_id,
        });

        this.props.selectAndEditItem(area);
    }

    onDelete(area) {
        this.setState({
            showConfirm: true,
            isWorking: false,
            error: null,
            question: <><div>Удалить <b>{area.area_title || area.area_id}</b>?</div>
                {area.area_title && <div>(<i>${area.area_id}</i>)</div> || ''}</>,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    this.request.exec(REQUESTS.AREA_REMOVE, {
                        body: {
                            areas: [{ area_id: area.area_id }],
                        },
                    }).then(() => {
                        this.setState({
                            error: null,
                            showConfirm: false,
                            isWorking: false,
                        }, () => {
                            this.props.update();
                        });
                    })
                        .catch((error) => {
                            this.setState({
                                error,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    toggleConfirm(showConfirm) {
        this.setState({
            showConfirm,
        });
    }

    onCloseEdit() {
        this.setState({
            showEditDialog: false,
            editableArea: null,
        }, () => {
            this.props.closeEditing();
        });
    }

    isContain(data, filter) {
        return data && data.toString()
            .toUpperCase().includes(filter.toString().toUpperCase());
    }

    importFew(state, hasChanged) {
        this.setState(() => ({
            showImportFew: state,
        }), () => {
            !state && hasChanged && this.props.update();
        });
    }

    exportXLSX() {
        const wb = XLSX.utils.book_new();
        const ws_data: any = [];
        ws_data.push([
            'area_id',
            'area_title',
            'area_tags',
            'hard_tags',
        ], ...this.state.filteredAreas.map(area => {
            return [
                area.area_id,
                area.area_title,
                area.area_tags,
                area.hard_tags?.map(tag => tag.tag).join(','),
            ];
        }));
        const ws = XLSX.utils.aoa_to_sheet(ws_data);
        XLSX.utils.book_append_sheet(wb, ws, 'Полигоны');
        XLSX.writeFile(wb, `poly-${new Date().getTime()}.xlsx`);
    }

    exportToYMap(items) {
        const res: any = {
            'type': 'FeatureCollection',
            'features': [],
        };
        const item: any = {
            'type': 'Feature',
            id: 0,
            'geometry': {
                'type': 'Polygon',
                'coordinates': [],
            },
            'properties': {
                'name': 'unknown',
                'description': '',
                'fill': '#ff931e',
                'fill-opacity': 0.3,
                'stroke': '#e6761b',
                'stroke-opacity': 0.9,
                'stroke-width': '2',
            },
        };
        items?.forEach(polyItem => {
            const newFeature: any = deepCopy(item);
            newFeature.geometry.coordinates = [polyItem.area_coords];
            newFeature.id = polyItem.area_id;
            newFeature.properties.name = polyItem.area_title || polyItem.area_id;
            newFeature.properties.description = polyItem.area_title || polyItem.area_id;
            newFeature.properties.tags = polyItem.area_tags;
            res.features.push(newFeature);
        });

        return res;
    }

    exportGeoJson() {
        const data = JSON.stringify(this.exportToYMap(this.state.exportGeoJson));
        download(data, `poly.json`, 'json');
    }

    geoJsonExportControl(state, area) {
        if (!state) {
            this.setState((prevState: any) => {
                const exportGeoJson: any[] = prevState.exportGeoJson;

                return {
                    exportGeoJson: [...exportGeoJson, area],
                };
            });
        } else {
            this.setState((prevState: any) => {
                const exportGeoJson: any[] = prevState.exportGeoJson;

                return {
                    exportGeoJson: [...exportGeoJson.filter(item => item.area_id !== area.area_id)],
                };
            });
        }
    }

    onTagsAttachClick(result) {
        this.setState({
            warning: '',
        });
        this.updateProgressBar({
            allLength: 0,
            successLength: 0,
            errors: [],
        });
        const idKey = result?.[0]?.findIndex((cell) => ID_KEY === cell.toLowerCase()) ?? -1;
        const tagKey = result?.[0]?.findIndex((cell) => TAG_KEY === cell.toLowerCase()) ?? -1;

        if (idKey !== -1 && tagKey != -1 && result?.length > HEADER_LENGTH) {
            const attachData = result.reduce((acc: Dict<any>[], row: string[]) => {
                if (ID_KEY !== row[idKey] && row[idKey]) {
                    acc.push({
                        hardTags: row[tagKey],
                        areaId: row[idKey],
                    });
                }

                return acc;
            }, []);
            this.setState({
                attachData,
            });
        } else {
            this.setState({
                warning: 'Кажется в файле не хватает данных',
            });
        }
    }

    filterAreas(area, filter) {
        return this.isContain(area.area_id, filter)
            || this.isContain(area.area_tags, filter)
            || this.isContain(area.area_title, filter)
            || area.hard_tags && area.hard_tags
                .filter(ht => this.isContain(ht.tag, filter)).length;
    }

    closeGroupAddTagModal() {
        this.setState({
            attachData: null,
        });
    }

    checkTagsDuplicates(tag) {
        if (this.state.attachData.filter(area => area.hardTags.includes(tag)).length > 0) {
            this.setState({
                showDuplicateConfirm: true,
                attachDataNoDuplicates: this.state.attachData.filter(area => !area.hardTags.includes(tag)),
            });
        } else {
            this.setState({
                addTagToPolygons: true,
            });
        }
    }

    onGroupAttachClick(tag) {
        this.setState({
            selectedTag: tag,
        });
        this.checkTagsDuplicates(tag.tag);
    }

    duplicateConfirmHandler(noDuplicates = false) {
        if (noDuplicates) {
            this.setState({
                attachData: this.state.attachData.filter(area => !area.hardTags.includes(this.state.selectedTag.tag)),
            });
        }

        this.setState({
            addTagToPolygons: true,
            showDuplicateConfirm: false,
        });
    }

    updateProgressBar(data) {
        this.setState({
            progressBarObj: data,
        });
    }

    onAddTagToPolygonsFinish() {
        this.setState({
            attachData: null,
            addTagToPolygons: false,
        });
    }

    toggleAddTagToPolygons() {
        this.setState({ addTagToPolygons: false });
    }

    render() {
        const {
            attachData,
            attachDataNoDuplicates,
            warning,
            error,
            showConfirm,
            showDuplicateConfirm,
            showImportFew,
            exportGeoJson,
            selectedTag,
            addTagToPolygons,
            isWorking,
            filteredAreas,
            question,
            accept,
            progressBarObj,
            selectedItem,
        } = this.state;

        const { filter, update, isLoading, areas, showAllPolygons } = this.props;

        return <div className={style.areas}>
            <div className={style.area_controls}>
                <Input value={filter}
                       placeholder={'Фильтр полигонов (id, name, tag)'}
                       onChange={this.onChangeFilter.bind(this)}
                       className={style.filterText}/>
                <div className={`${coreStyle.button_container} ${coreStyle.full_width}`}>
                    <Button onClick={this.importFew.bind(this, true)}>Импорт</Button>
                    <Button disabled={!exportGeoJson.length}
                            colorType={ButtonTypes.warning}
                            onClick={this.exportGeoJson.bind(this, true)}>
                        GeoJson {exportGeoJson.length
                            ? `(${exportGeoJson.length})`
                            : ''}
                    </Button>
                    <Button disabled={!areas?.length}
                            colorType={ButtonTypes.positive}
                            onClick={this.exportXLSX.bind(this)}
                            ytLog={{ button_name: buttonNameDetails.POLYGONS_DOWNLOAD }}>XLSX</Button>
                    <AddTagsToAreas onClick={readTable.bind(this, this.onTagsAttachClick.bind(this))}
                                    areas={attachData}
                                    tag={selectedTag}
                                    addTagToPolygons={addTagToPolygons}
                                    updateProgressBar={this.updateProgressBar.bind(this)}
                                    onClose={this.onAddTagToPolygonsFinish.bind(this)}
                                    toggleAddTagToPolygons={this.toggleAddTagToPolygons.bind(this)}
                                    updatePolygons={update}/>
                    {warning.length
                        ? <Confirm question={warning}
                                   error={error}
                                   isWorking={isWorking}
                                   onClose={() => {
                                       this.setState({
                                           warning: '',
                                       });
                                   }}
                                   accept={readTable.bind(this, this.onTagsAttachClick.bind(this))}
                                   acceptTitle={'Открыть другой файл'}/>
                        : null
                    }
                </div>
                <div className={`${coreStyle.button_container} ${coreStyle.full_width}`}>
                    <PackageDeleteTags areas={filteredAreas}
                                       isLoading={isLoading}
                                       updateAreas={update}/>
                    <Checkbox className={style.checkbox_control}
                              onChange={this.setShowAllPolygons.bind(this)}
                              checked={showAllPolygons}/>
                    <div className={style.title}>Все полигоны</div>
                </div>
            </div>
            {
                showImportFew
                && <ImportFew onClose={this.importFew.bind(this, false)}/>
            }

            {
                showConfirm
                && <Confirm error={error}
                            isWorking={isWorking}
                            onClose={this.toggleConfirm.bind(this)}
                            accept={accept}
                            question={question}/>
            }
            {
                showDuplicateConfirm
                && <Confirm error={error}
                            isWorking={isWorking}
                            onClose={() => {
                                this.setState({
                                    showDuplicateConfirm: false,
                                });
                            }}
                            accept={this.duplicateConfirmHandler.bind(this, false)}
                            question={
                                `Выбранный тег уже навешан на
                                ${attachData.length - attachDataNoDuplicates.length}/${attachData.length} полигонов`
                            }
                            acceptTitle={'Навесить на все'}
                            ignoreCancelBtn
                            additionalControls={
                                <Button onClick={this.duplicateConfirmHandler.bind(this, true)}
                                        colorType={ButtonTypes.positive}
                                        disabled={!attachDataNoDuplicates.length}>
                                    Навесить на остальные
                                </Button>
                            }/>
            }
            {
                attachData
                    ? <TagModal objectId={{ type: OBJECT_TYPES.AREA }}
                                title={`Создание тега (${attachData.length}) полигона`}
                                onClose={this.closeGroupAddTagModal.bind(this)}
                                attachmentHandler={this.onGroupAttachClick.bind(this)}
                                progressBarObj={progressBarObj}/>
                    : null
            }
            <div className={style.areas_content}>
                {
                    isLoading
                        ? <Spin/>
                        : filteredAreas?.length
                            ? filteredAreas
                                .map((area, index) => {
                                    const isSelectedToGeoJsonExport = exportGeoJson
                                        .some((item: IPolygonItem) => item.area_id === area.area_id);

                                    return <AreaItem area={area}
                                                     selectedItem={selectedItem}
                                                     onEdit={this.onEdit.bind(this, area)}
                                                     geoJsonExportControl={this.geoJsonExportControl
                                                         .bind(this, isSelectedToGeoJsonExport, area)}
                                                     isSelectedToGeoJsonExport={isSelectedToGeoJsonExport}
                                                     onDelete={this.onDelete.bind(this, area)}
                                                     key={index}
                                                     isProposition={!!area.proposition_id}
                                                     onClick={this.selectItem.bind(this, area)}/>;
                                })
                            : !filter ? <span>Введите фильтр, чтобы посмотреть список полигонов</span> : null
                }
            </div>
            <Switch>
                <Route path="/settings/poly/:polyId" component={AreaInfoModal}/>
            </Switch>
        </div>;
    }
}

interface IAreaItemProps {
    area: IPolygonItem;
    onClick: () => void;
    onEdit: () => void;
    geoJsonExportControl: () => void;
    isSelectedToGeoJsonExport: boolean;
    selectedItem: string;
    onDelete?: () => void;
    isProposition?: boolean;
}

const AreaItem = React.memo((props: IAreaItemProps) => {
    const { area, selectedItem, isSelectedToGeoJsonExport, geoJsonExportControl } = props;
    const isDeny = area.area_tags && area.area_tags.includes('deny');

    return <div className={`${style.area_item} ${isDeny && style.area_item_deny || ''}`
        + ` ${selectedItem === area.area_id + '_' + area.proposition_id ? style.area_item_selected : ''}`
        + ` ${area.proposition_id ? style.area_proposition : ''}`}
                onClick={props.onClick}>
        <div className={style.area_item_id}>
            <div className={style.text}>
                {area.area_title || area.area_id}
                <sup className={style.revision}>{area.revision}</sup>
            </div>
            {
                area.proposition_id && <div className={style.area_proposition_description}>
                    <strong>propose:</strong> <FormatDate withSecond
                                                          value={area.confirmations
                                                                && area.confirmations[0]
                                                                && area.confirmations[0].history_instant * ONE_SECOND}/>
                    <div><strong>comment:</strong> {area.proposition_description}</div>
                </div>
            }
        </div>
        <div className={style.area_item_edit}>
            <div><Link onClick={props.onEdit}>edit</Link></div>
            {!props.isProposition &&
            <div><Link onClick={props.onDelete} className={style.delete}>del</Link></div>}
            {
                <div><Link onClick={geoJsonExportControl} className={style.export}>
                    {isSelectedToGeoJsonExport
                        ? <span title={'remove from export GeoJson'}>-</span>
                        : <span title={'add to export GeoJson'}>+</span>}
                </Link></div>
            }

        </div>
        <div className={style.area_item_tags}>{area.area_tags}</div>
        <div className={style.area_item_hard_tags}>{area.hard_tags && area.hard_tags.map(ht => ht.tag).join(', ')}</div>
        <div className={style.area_item_edit_hard_tags}>
            <Link href={`#/settings/poly/${encodeURIComponent(area.area_id)}`} className={style.info}>info</Link>
        </div>
    </div>;
});
