import * as React from 'react';
import { connect } from 'react-redux';

import { Dict } from '../../../../types';
import { Button } from '../../../ui/Button';
import { Input } from '../../../ui/Input';
import { Request2 } from '../../../utils/request';
import { deepCopy } from '../../../utils/utils';
import { IStore } from '../../App/store';
import { FormConstructor } from '../../FormConstructor';
import { controlType } from '../../FormConstructor/types';
import Spin from '../../Spin';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import * as styles from './index.css';
import { FALLBACK_PROMO_SCHEMA } from './schema';

interface IAddPromoProps {
    setError: (error) => void;
    afterFetch?: () => {};
    newGenerator?: string[];
}

interface Internal {
    promo_generators?: any;
}

interface IAddPromoInternalState {
    meta_1: Dict<any>;
    meta_2: Dict<any>;
    schema_1: Dict<any>;
    schema_2: Dict<any>;
    limit: Dict<number>;
    isButtonLoading: boolean;
    isLoading: boolean;

    [x: number]: any;
}

function mapStateToProps(store: IStore) {
    const { promo_generators = [] } = store?.AdminUser?.permissions;

    return {
        promo_generators,
    };
}

class AddPromoInternal extends React.Component<IAddPromoProps & Internal, IAddPromoInternalState> {
    state: IAddPromoInternalState = {
        meta_1: {},
        meta_2: {},
        schema_1: {},
        schema_2: {},
        limit: {},
        isButtonLoading: false,
        isLoading: false,
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    componentDidMount() {
        this.setState({
            isLoading: true,
        }, async () => {
            let schema: Dict<any>;
            try {
                schema = await this.request.exec(REQUESTS.GENERATE_PROMO, {
                    queryParams: {
                        scheme: true,
                    },
                });
            }
            catch(error) {
                schema = {
                    ...FALLBACK_PROMO_SCHEMA,
                    generator: {
                        ...FALLBACK_PROMO_SCHEMA.generator,
                        variants: [...this.props.promo_generators, ...(this.props.newGenerator ?? [])],
                    },
                };
            }

            delete schema.usage_limit;

            const schema_2 = deepCopy(schema);

            delete schema_2.prefix;
            delete schema_2.count_symbols;
            delete schema_2.generator;
            delete schema_2.count;

            this.setState({
                schema_1: {
                    generator: {
                        ...schema.generator,
                        default: this.props?.newGenerator?.[0],
                    },
                    _bothUpperAndLowercase: {
                        type: controlType.bool,
                        display_name: 'Cоздать промокод и в верхнем, и в нижнем регистре',
                        default: false,
                        order: 4,
                    },
                    prefix: schema.prefix,
                    count_symbols: schema.count_symbols,
                    count: schema.count,
                },
                schema_2,
                isLoading: false,
            });
        });
    }
    onChange(key, value) {
        this.setState({
            [key]: value,
        });
    }

    onChangeLimit(key, value) {
        this.setState((prev) => ({
            limit: {
                ...prev.limit,
                [key]: value,
            },
        }));
    }

    makeRequest(props: { prefix: string; usage_limit: number }): Promise<any> {
        const { meta_1, meta_2 } = this.state;
        const { prefix, usage_limit } = props;

        return this.request.exec(REQUESTS.GENERATE_PROMO, {
            queryParams: {
                count: meta_1.count,
                generator: meta_1.generator,
            },
            body: {
                ...meta_1,
                ...meta_2,
                prefix,
                usage_limit,
            },
        });
    }

    sendData() {
        const { meta_1, limit } = this.state;
        const { afterFetch } = this.props;

        this.setState({
            isButtonLoading: true,
        }, async () => {
            let requests: Promise<any>[];

            if (meta_1?._bothUpperAndLowercase) {
                requests = [
                    this.makeRequest({
                        prefix: meta_1.prefix.toUpperCase(),
                        usage_limit: limit?.usage_limit_upper,
                    }),
                    this.makeRequest({
                        prefix: meta_1.prefix.toLowerCase(),
                        usage_limit: limit?.usage_limit_lower,
                    }),
                ];
            } else {
                requests = [
                    this.makeRequest({
                        prefix: meta_1.prefix,
                        usage_limit: limit?.usage_limit,
                    }),
                ];
            }

            try {
                await Promise.all(requests);
                this.setState({
                    isButtonLoading: false,
                });

                afterFetch && afterFetch();
            } catch (error) {
                this.props.setError(error);
                this.setState({
                    isButtonLoading: false,
                });
            }
        });
    }

    render() {
        const { isButtonLoading, meta_1, meta_2, limit, isLoading } = this.state;

        return (
            <>
                {isLoading ? <Spin/> : (
                    <>
                        <FormConstructor schema={this.state.schema_1}
                                         onChange={this.onChange.bind(this, 'meta_1')}
                                         initialData={meta_1}/>
                        {meta_1?._bothUpperAndLowercase
                            ? <div>
                                <Input className={styles.usage_input}
                                       value={limit?.usage_limit_lower}
                                       required
                                       placeholder={'Количество использований (нижний регистр) (Обязательное поле)'}
                                       onChange={this.onChangeLimit.bind(this, 'usage_limit_lower')}/>
                                <Input className={styles.usage_input}
                                       value={limit?.usage_limit_upper}
                                       required
                                       placeholder={'Количество использований (верхний регистр) (Обязательное поле)'}
                                       onChange={this.onChangeLimit.bind(this, 'usage_limit_upper')}/>
                            </div>
                            : <Input value={limit?.usage_limit}
                                     required
                                     placeholder={'Количество использований'}
                                     onChange={this.onChangeLimit.bind(this, 'usage_limit')}/>
                        }
                        <FormConstructor schema={this.state.schema_2}
                                         onChange={this.onChange.bind(this, 'meta_2')}
                                         initialData={meta_2}/>
                        <div className={styles.addPromoButton}>
                            <Button onClick={this.sendData.bind(this)}
                                    isLoading={isButtonLoading}>Сохранить</Button>
                        </div>
                    </>
                )}
            </>
        );
    }
}

export const AddPromo = connect(mapStateToProps)(AddPromoInternal);
