import * as React from 'react';

import { ONE_SECOND } from '../../../constants';
import { Button, ButtonTypes } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import DatePicker from '../../../ui/DatePicker';
import { Input } from '../../../ui/Input';
import { Request2, specialQueryParamsKeys } from '../../../utils/request';
import { IPromo, PromoTable, SearchPromoModal } from '../../Clients/BonusView/BonusViewPromo';
import { SimpleError } from '../../SimpleError';
import { REQUESTS, SETTINGS_REQUESTS } from '../request';
import { AddPromoModal } from './AddPromoModal';
import { GiveOutPromoModal } from './GiveOutPromoModal';
import * as styles from './index.css';

interface IPromoProps {

}

interface IPromoState {
    [x: number]: any;

    filters: {
        [x: number]: any;
        active_only: boolean;
        since: number | null;
        until: number | null;
        prefix: string;
        count: number | null;
        given_out: string;
        twoCases: boolean;
        generator: string;
    };
    isLoading: boolean;
    codes: IPromo[];
    error: Error | null;
    showSearch: boolean;
    showAdd: boolean;
    showGiveOut: boolean;
}

export class Promo extends React.Component<IPromoProps, IPromoState> {
    state: IPromoState = {
        filters: {
            active_only: true,
            since: null,
            until: null,
            prefix: '',
            count: 1000,
            given_out: '',
            twoCases: false,
            generator: '',
        },
        isLoading: false,
        codes: [],
        error: null,
        showSearch: false,
        showAdd: false,
        showGiveOut: false,
    };

    request = new Request2({ requestConfigs: SETTINGS_REQUESTS });

    makeRequest(prefix) {
        const { active_only, count, given_out, since, until, generator } = this.state.filters;

        return this.request.exec(REQUESTS.GET_PROMO, {
            queryParams: {
                active_only: active_only.toString(),
                since: since && Math.round(since / ONE_SECOND),
                until: until && Math.round(until / ONE_SECOND),
                prefix,
                count,
                generator,
            },
            specialQueryParams: {
                given_out: {
                    key: specialQueryParamsKeys.FORCE,
                    value: given_out,
                },
            },
        });
    }

    getPromo() {
        const { prefix, twoCases } = this.state.filters;

        this.setState({
            isLoading: true,
        }, () => {
            const request = twoCases
                ? Promise.all([this.makeRequest(prefix.toLowerCase()), this.makeRequest(prefix.toUpperCase())])
                : this.makeRequest(prefix);
            request
                .then((response) => {
                    let codes: IPromo[];
                    if (twoCases) {
                        codes = response[0].codes?.concat(response[1].codes);
                    } else {
                        codes = response.codes;
                    }

                    codes = codes.sort((a, b) => {
                        return (a.hasOwnProperty('prefix') && b.hasOwnProperty('prefix'))
                            ? a.prefix.localeCompare(b.prefix) : 0;
                    });

                    this.setState({
                        codes,
                        isLoading: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        isLoading: false,
                        error,
                    });
                });
        });

    }

    toggle(key) {
        const value = !this.state[key];

        this.setState({
            [key]: value,
        });
    }

    onFiltersChange(key, value) {
        const { filters } = this.state;

        if (key === 'since' || key === 'until') {
            value = +value;
        }

        this.setState({
            filters: {
                ...filters,
                [key]: value,
            },
        });
    }

    componentDidMount() {
        this.getPromo();
    }

    render() {
        const { codes, isLoading, error, showSearch, showAdd, showGiveOut } = this.state;
        const { prefix, active_only, count, given_out, since, until, twoCases, generator } = this.state.filters;

        return (
            <>
                {showSearch && (
                    <SearchPromoModal onClose={this.toggle.bind(this, 'showSearch')}/>
                )}
                {showAdd && (
                    <AddPromoModal onClose={this.toggle.bind(this, 'showAdd')}
                                   fetchData={this.getPromo.bind(this)}/>
                )}
                {showGiveOut && (
                    <GiveOutPromoModal onClose={this.toggle.bind(this, 'showGiveOut')}
                                       data={codes}
                                       fetchData={this.getPromo.bind(this)}/>
                )}
                <div className={styles.header}>
                    <div className={styles.buttonContainer}>
                        <Button onClick={this.toggle.bind(this, 'showAdd')}>Добавить серию</Button>
                        <Button onClick={this.toggle.bind(this, 'showGiveOut')}
                                disabled={!codes.length}
                                colorType={ButtonTypes.positive}
                                basic>Раздать промо и сохранить в файл ({codes.length})</Button>
                        <Button onClick={this.toggle.bind(this, 'showSearch')}
                                colorType={ButtonTypes.warning}
                                basic>Поиск по промокоду</Button>
                    </div>
                    <Input className={styles.filterInput}
                           value={prefix}
                           onChange={this.onFiltersChange.bind(this, 'prefix')}
                           placeholder={'Префикс'}/>
                    <DatePicker className={styles.filterInput}
                                value={since ? since : ''}
                                onChange={this.onFiltersChange.bind(this, 'since')}
                                placeholder={'C'}/>
                    <DatePicker className={styles.filterInput}
                                value={until ? until : ''}
                                onChange={this.onFiltersChange.bind(this, 'until')}
                                placeholder={'По'}/>
                    <Input className={styles.filterInput}
                           value={given_out}
                           onChange={this.onFiltersChange.bind(this, 'given_out')}
                           placeholder={'Кому отдали'}/>
                    <Input className={styles.filterInput}
                           value={count ? count : ''}
                           onChange={this.onFiltersChange.bind(this, 'count')}
                           placeholder={'Запрашиваемое кол-во'}/>
                    <Input className={styles.filterInput}
                           value={generator}
                           onChange={this.onFiltersChange.bind(this, 'generator')}
                           placeholder={'Генератор'}/>
                    <div className={styles.filterInput}>
                        <div className={styles.checkbox_wrapper}>
                            <span>Только активные</span>
                            <Checkbox checked={active_only}
                                      className={styles.checkbox}
                                      onChange={this.onFiltersChange.bind(this, 'active_only')}/>
                        </div>
                        <div className={styles.checkbox_wrapper}>
                            <span>В двух регистрах</span>
                            <Checkbox checked={twoCases}
                                      onChange={this.onFiltersChange.bind(this, 'twoCases')}/>
                        </div>
                    </div>
                    <Button className={styles.filterButton}
                            onClick={this.getPromo.bind(this)}
                            basic>Поиск</Button>
                </div>
                <PromoTable codes={codes}
                            isLoading={isLoading}
                            error={error ? <SimpleError error={error}/> : null}
                            settings
                            sortByGenerator/>
            </>
        );
    }
}
