import * as React from 'react';
import { Route, Switch } from 'react-router-dom';

import { Dict } from '../../../../types';
import { Request2 } from '../../../utils/request';
import ChoiceQuestion from './ChoiceQuestion';
import { PROMOCODE_WIZARD_URI } from './constants';
import { DEFAULT_PROCENT_ACTION, DEFAULT_PROCENT_TAG } from './defaults/procentDefaults';
import { DEFAULT_RUB_ACTION, DEFAULT_RUB_TAG } from './defaults/rubDefaults';
import GeneratorForm from './GeneratorForm';
import PromocodeForm from './PromocodeForm';
import { PW_REQUESTS, REQUESTS } from './request';
import { PROCENT_ACTION_SCHEMA, PROCENT_TAG_SCHEMA } from './schemas/procentSchemas';
import { RUB_ACTION_SCHEMA, RUB_TAG_SCHEMA } from './schemas/rubSchemas';
import { PROMOCODE_WIZARD_ROUTE } from './types';

const RUB_GENERATOR = 'promo_generator_billing_tag';
const PROCENT_GENERATOR = 'promo_generator_rides';
const PROCENT_ACTION_TYPE = 'offer_corrector_discounts';
const DEFAULT_PROMO_TYPES_USAGE_ACTION = 'promo_20';

interface IPromocodeWizardProps {
    constants: any;
}

interface IPromocodeWizardState {
    wizardStep: PROMOCODE_WIZARD_ROUTE;
    tagForm: {
        name: string;
        comment: string;
        meta: {
            amount: number;
            deep_history_deep: string;
            amount_lifetime: string;
            amount_deadline: number;
        };
    } | null;
    actionForm: any;
    error: Error | null;
    iface: Dict<any>;
    rubTagSchema: Dict<any>;
    actions: any;
    procentTagSchema: Dict<any>;
    progress: {title: string; isLoading: boolean} | null;
}

export class PromocodeWizard extends React.Component<IPromocodeWizardProps, IPromocodeWizardState> {
    state: IPromocodeWizardState = {
        wizardStep: PROMOCODE_WIZARD_ROUTE.initial,
        tagForm: null,
        actionForm: {},
        iface: {},
        error: null,
        progress: null,
        rubTagSchema: RUB_TAG_SCHEMA,
        actions: [],
        procentTagSchema: PROCENT_TAG_SCHEMA,
    };

    request = new Request2({ requestConfigs: PW_REQUESTS });

    changeStep(stepConfig) {
        if (location?.href) {
            location.href = `#${PROMOCODE_WIZARD_URI}${stepConfig}`;
        }

        this.setState({
            wizardStep: stepConfig,
        });
    }

    onChangeTag(tagForm) {
        this.setState({
            tagForm,
        });
    }

    onChangeAction(actionForm) {
        this.setState({
            actionForm,
        });
    }

    componentDidMount() {
        this.props.constants
            .then(responseConstants => {
                this.setState({
                    iface: responseConstants?.iface,
                });
            })
            .catch((error) => {
                this.setState({
                    error,
                });
            });
    }

    componentDidUpdate(prevProps: Readonly<IPromocodeWizardProps>, prevState: Readonly<IPromocodeWizardState>) {
        if (prevState.wizardStep !== this.state.wizardStep) {
            if (this.state.wizardStep === PROMOCODE_WIZARD_ROUTE.generator_procent) {
                this.request.exec(REQUESTS.GET_ACTIONS)
                    .then((response) => {
                        const actions = response?.report
                            ?.reduce((_p, _c) => {
                                if (_c.action_type === PROCENT_ACTION_TYPE) {
                                    _p.push(_c.action_id);
                                }

                                return _p;
                            }, []);

                        this.setState({
                            procentTagSchema: {
                                ...PROCENT_TAG_SCHEMA,
                                meta: {
                                    ...PROCENT_TAG_SCHEMA.meta,
                                    structure: {
                                        ...PROCENT_TAG_SCHEMA.meta.structure,
                                        action_ids: {
                                            ...PROCENT_TAG_SCHEMA.meta.structure.action_ids,
                                            variants: actions,
                                        },
                                    },
                                },
                            },
                        });
                    })
                    .catch(() => {
                        this.setState({
                            error: new Error(`Не получены экшены ${PROCENT_ACTION_TYPE}. Перезагрузите страницу или введите данные вручную`),
                        });
                    });
            }

            if (this.state.wizardStep === PROMOCODE_WIZARD_ROUTE.generator_rub) {
                this.props.constants
                    .then(responseConstants => {
                        this.setState({
                            rubTagSchema: {
                                ...RUB_TAG_SCHEMA,
                                meta: {
                                    ...RUB_TAG_SCHEMA.meta,
                                    structure: {
                                        ...RUB_TAG_SCHEMA.meta.structure,
                                        accounts: {
                                            ...RUB_TAG_SCHEMA.meta.structure.accounts,
                                            variants: responseConstants?.iface_tag_descriptions
                                                ?.fixed_sum_tag?.accounts?.variants,
                                        },
                                    },
                                },
                            },
                        });
                    })
                    .catch(() => {
                        this.setState({
                            error: new Error('Не получены доступные для списания аккаунты. Перезагрузите страницу или введите данные вручную'),
                        });
                    });
            }
        }
    }

    onSubmit(
        type: PROMOCODE_WIZARD_ROUTE.generator_procent | PROMOCODE_WIZARD_ROUTE.generator_rub,
        data: Dict<string>,
    ) {
        const { tagForm, actionForm } = this.state;
        let generator;
        let defaultAction;
        let defaultTag;
        let robotId = {};
        let sinceUntil = {};

        this.setState({
            error: null,
        });

        if (type === PROMOCODE_WIZARD_ROUTE.generator_rub) {
            generator = RUB_GENERATOR;
            defaultAction = DEFAULT_RUB_ACTION;
            defaultTag = DEFAULT_RUB_TAG;
        } else {
            generator = PROCENT_GENERATOR;
            defaultAction = DEFAULT_PROCENT_ACTION;
            defaultTag = DEFAULT_PROCENT_TAG;
        }

        const schema = this.state?.iface?.[generator];

        if (type === PROMOCODE_WIZARD_ROUTE.generator_rub) {
            robotId = {
                robot_id: schema?.action_meta?.structure?.promo_info?.structure?.robot_id?.default,
            };
        }

        if (type === PROMOCODE_WIZARD_ROUTE.generator_procent) {
            sinceUntil = {
                since: actionForm.since,
                until: actionForm.until,
            };
        }

        const actionRequest = (data) => {
            this.setState({
                progress: {
                    title: 'Заводим генератор',
                    isLoading: true,
                },
            }, () => {
                this.request.exec(REQUESTS.ADD_ACTION, {
                    body: {
                        ...defaultAction,
                        action_id: actionForm.action_id,
                        action_description: data?.comment,
                        action_meta: {
                            ...defaultAction.action_meta,
                            experiment: {
                                min_shard: schema?.action_meta?.structure?.experiment?.structure?.min_shard?.default,
                                max_shard: schema?.action_meta?.structure?.experiment?.structure?.max_shard?.default,
                            },
                            promo_info: {
                                ...defaultAction.action_meta.promo_info,
                                ...robotId,
                                tag_name: data?.name,
                                first_only: actionForm.first_only,
                                ...sinceUntil,
                            },
                        },
                    },
                })
                    .then(() => {
                        this.setState({
                            progress: {
                                title: 'Подключаем генератор к promo_20',
                                isLoading: true,
                            },
                        }, () => {
                            this.request.exec(REQUESTS.GET_ACTIONS)
                                .then((response) => {
                                    const initAction = response?.report
                                        ?.filter(role => role.action_id === DEFAULT_PROMO_TYPES_USAGE_ACTION)[0];

                                    if (initAction) {
                                        this.request.exec(REQUESTS.ADD_ACTION, {
                                            body: {
                                                ...initAction,
                                                action_meta: {
                                                    ...initAction.action_meta,
                                                    available_promo_types: [
                                                        ...initAction.action_meta.available_promo_types,
                                                        actionForm.action_id,
                                                    ],
                                                },
                                            },
                                        }).then(() => {
                                            this.setState({
                                                progress: {
                                                    title: 'Генератор заведен! ✅',
                                                    isLoading: false,
                                                },
                                            });
                                        });
                                    } else {
                                        this.setState({
                                            error: new Error('Экшен promo_20 не найден'),
                                        });
                                    }
                                })
                                .catch((error) => {
                                    this.setState({
                                        error,
                                    });
                                });
                        });
                    })
                    .catch((error) => {
                        this.setState({
                            error,
                        });
                    });
            });
        };

        if (data) {
            actionRequest(data);
        } else {
            this.setState({
                progress: {
                    title: 'Заводим тег',
                    isLoading: true,
                },
            }, () => {
                this.request.exec(REQUESTS.CREATE_TAG, {
                    body: {
                        name: tagForm?.name,
                        comment: tagForm?.comment,
                        type: defaultTag.type,
                        display_name: defaultTag.display_name,
                        default_priority: defaultTag.default_priority,
                        meta: {
                            ...defaultTag.meta,
                            ...tagForm?.meta,
                        },
                    },
                })
                    .then(() => {
                        actionRequest({ name: tagForm?.name, comment: tagForm?.comment });
                    })
                    .catch((error) => {
                        this.setState({
                            error,
                        });
                    });
            });
        }
    }

    render() {
        const { error, progress, actionForm, rubTagSchema, procentTagSchema } = this.state;

        return (
            <>
                <Switch>
                    <Route path={`${PROMOCODE_WIZARD_URI}${PROMOCODE_WIZARD_ROUTE.generator_rub}`}
                           render={() => {
                               return <GeneratorForm error={error}
                                                     progress={progress}
                                                     title={'Заводим рублевую скидку'}
                                                     onChangeTag={this.onChangeTag.bind(this)}
                                                     onChangeAction={this.onChangeAction.bind(this)}
                                                     tagSchema={rubTagSchema}
                                                     actionSchema={RUB_ACTION_SCHEMA}
                                                     onSubmit={this.onSubmit
                                                         .bind(this, PROMOCODE_WIZARD_ROUTE.generator_rub)}
                                                     changeStep={this.changeStep.bind(this)}/>;
                           }}/>
                    <Route path={`${PROMOCODE_WIZARD_URI}${PROMOCODE_WIZARD_ROUTE.generator_procent}`}
                           render={() => {
                               return <GeneratorForm error={error}
                                                     progress={progress}
                                                     title={'Заводим процентную скидку'}
                                                     onChangeTag={this.onChangeTag.bind(this)}
                                                     onChangeAction={this.onChangeAction.bind(this)}
                                                     tagSchema={procentTagSchema}
                                                     actionSchema={PROCENT_ACTION_SCHEMA}
                                                     onSubmit={this.onSubmit
                                                         .bind(this, PROMOCODE_WIZARD_ROUTE.generator_procent)}
                                                     changeStep={this.changeStep.bind(this)}/>;
                           }}/>
                    <Route path={`${PROMOCODE_WIZARD_URI}${PROMOCODE_WIZARD_ROUTE.promo}`}
                           render={() => {
                               const newGenerator = actionForm?.action_id ? [actionForm?.action_id] : [];

                               return <PromocodeForm newGenerator={newGenerator}/>;
                           }}/>
                    <Route path={`${PROMOCODE_WIZARD_URI}`}
                           render={() => {
                               const options = [
                                   { name: 'Рублевая скидка', stepConfig: PROMOCODE_WIZARD_ROUTE.generator_rub },
                                   { name: 'Процентная скидка', stepConfig: PROMOCODE_WIZARD_ROUTE.generator_procent },
                               ];

                               return <ChoiceQuestion options={options}
                                                      question={'Для какой скидки заводим генератор?'}
                                                      changeStep={this.changeStep.bind(this)}/>;
                           }}/>
                </Switch>
            </>
        );
    }
}
