/*eslint-disable no-magic-numbers*/

import {
    drag,
    forceLink,
    forceManyBody,
    forceSimulation,
    forceX,
    forceY,
    scaleOrdinal,
    schemeCategory10,
    select,
    zoom,
} from 'd3';
import React from 'react';

import { GraphHelper } from '../graphHelper';

interface ILink {
    source: string;
    target: string;
    type: string;
}

export default function RoleGraph({ roles, role_id, buildList }) {
    const [nodes, setNodes] = React.useState<{ id: string }[]>([]);
    const [links, setLinks] = React.useState<ILink[]>([]);
    const graph = new GraphHelper(roles?.report || []);

    const prepare = () => {
        const nodesMap = {};
        const _links: ILink[] = [];

        roles?.report?.forEach(role => {
            graph.addNode(role);
            nodesMap[role.role_id] = true;

            if (role?.slave_roles) {
                role.slave_roles?.forEach((slave_role) => {
                    _links.push({
                        source: role.role_id,
                        target: slave_role.slave_role_id,
                        type: 'roles',
                    });
                    nodesMap[role.role_id] = true;
                    nodesMap[slave_role.slave_role_id] = true;
                });
            }
            /* if (role?.actions) {
                 role.actions.forEach((action) => {
                     _links.push({
                         source: role.role_id,
                         target: action.action_id,
                         type: 'actions'
                     });
                     nodesMap[role.role_id] = true;
                     nodesMap[action.action_id] = true;
                 });
             }*/
        });

        if (!role_id) {
            setNodes(Object.entries(nodesMap).map(el => ({ id: el[0] })));
            setLinks(_links);
        } else {
            const [, self, parents] = graph.find(role_id);
            const _links: any[] = [];
            let _nodes: any[] = [];
            const _nodesObj = new Set();

            const deep = (node) => {
                _nodesObj.add(node.node.node_id);
                if (node.children.length) {
                    node.children.forEach(el => {
                        _links.push({
                            source: node.node.node_id,
                            target: el.node.node_id,
                            type: 'roles',
                        });

                        return deep(el);
                    });
                }
            };

            if (parents.length) {
                parents?.forEach(([parent_name, node]) => {
                    deep(node);
                });
            } else {
                self && deep(self);
            }

            _nodes = [..._nodesObj].map(id => ({ id }));
            setNodes(_nodes);
            setLinks(_links);
        }
    };

    React.useEffect(() => {
        prepare();
    }, [roles, role_id]);

    React.useEffect(() => {
        drawCanvas();
    }, []);

    React.useEffect(() => {
        nodes.length && init();
        buildList(nodes);
    }, [nodes]);

    const types = ['actions', 'roles'];
    const color = scaleOrdinal(types, schemeCategory10);
    const strokeWidth = (type) => type === 'actions' ? 1.5 : 2.5;

    const drawCanvas = () => {
        const [width, height] = [1000, 600];
        const svg = select('#graph')
            .append('svg')
            .attr('viewBox', [-width / 2, -height / 2, width, height].join(' '))
            .style('font', '14px sans-serif')
            .call(zoom().on('zoom', function (e) {
                svg.attr('transform', e.transform);
            })).append('g').attr('id', 'content');

        select('#graph svg').append('defs').selectAll('marker')
            .data(types)
            .join('marker')
            .attr('id', d => `arrow-${d}`)
            .attr('viewBox', '0 -5 10 10')
            .attr('refX', 15)
            .attr('refY', -0.5)
            .attr('markerWidth', 6)
            .attr('markerHeight', 6)
            .attr('orient', 'auto')
            .append('path')
            .attr('fill', color)
            .attr('d', 'M0,-5L10,0L0,5');
    };

    const init = () => {
        const simulation = forceSimulation(nodes as any)
            .force('link', forceLink(links).id((d:any) => d.id))
            .force('charge', forceManyBody().strength(-800))
            .force('x', forceX())
            .force('y', forceY());

        const svg = select('#content');

        svg.selectAll('*').remove();

        const link = svg.append('g')
            .attr('id', 'link')
            .attr('fill', 'none')
            .selectAll('path')
            .data(links)
            .join('path')
            .attr('stroke', d => color(d.type))
            .attr('stroke-width', d => strokeWidth(d.type))
            .attr('marker-end', d => `url(/#arrow-${d.type})`);

        const _drag = (simulation: any) => {

            function dragstarted(event, d) {
                if (!event.active) {
                    simulation.alphaTarget(0.3).restart();
                }

                d.fx = d.x;
                d.fy = d.y;
            }

            function dragged(event, d) {
                d.fx = event.x;
                d.fy = event.y;
            }

            function dragended(event, d) {
                if (!event.active) {
                    simulation.alphaTarget(0);
                }

                d.fx = null;
                d.fy = null;
            }

            return drag()
                .on('start', dragstarted)
                .on('drag', dragged)
                .on('end', dragended);
        };

        const node = svg.append('g')
            .attr('id', 'nodes')
            .attr('fill', 'currentColor')
            .attr('stroke-linecap', 'round')
            .attr('stroke-linejoin', 'round')
            .selectAll('g')
            .data(nodes)
            .join('g')
            .call(_drag(simulation) as any);

        node.append('circle')
            .attr('stroke', 'white')
            .attr('stroke-width', 1)
            .attr('r', 6);

        node.append('text')
            .attr('x', 8)
            .attr('y', '0.31em')
            .attr('fill', d => d.id == role_id && '#0f0')
            .attr('font-size', d => d.id == role_id && '20px')
            .text(d => d.id)
            .clone(true).lower()
            .attr('fill', 'none')
            .attr('stroke', 'white')
            .attr('stroke-width', 3);

        simulation.on('tick', () => {
            link.attr('d', linkArc);
            node.attr('transform', (d: any) => `translate(${d.x},${d.y})`);
        });

    };

    return <div>
        <div id={'graph'}/>
    </div>;
}

function linkArc(d) {
    const r = Math.hypot(d.target.x - d.source.x, d.target.y - d.source.y);

    return `M${d.source.x},${d.source.y} A${r},${r} 0 0,1 ${d.target.x},${d.target.y} `;
}
